package group.flyfish.rest.utils;

import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * 请求上下文
 *
 * @author wangyu
 * 基于spring安全调用
 */
public final class RequestContext {

    public static final String AUTHORIZATION_KEY = "Authorization";

    /**
     * 获取当前request
     *
     * @return 结果
     */
    public static Optional<HttpServletRequest> getRequest() {
        RequestAttributes attributes = RequestContextHolder.getRequestAttributes();
        if (attributes instanceof ServletRequestAttributes) {
            ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) attributes;
            return Optional.ofNullable(servletRequestAttributes.getRequest());
        }
        return Optional.empty();
    }

    /**
     * 获取响应
     *
     * @return 结果
     */
    public static Optional<HttpServletResponse> getResponse() {
        RequestAttributes attributes = RequestContextHolder.getRequestAttributes();
        if (attributes instanceof ServletRequestAttributes) {
            ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) attributes;
            return Optional.ofNullable(servletRequestAttributes.getResponse());
        }
        return Optional.empty();
    }

    /**
     * 获取所有的cookie
     *
     * @return 结果
     */
    public static List<Cookie> getCookies() {
        return getRequest().flatMap(request -> Optional.ofNullable(request.getCookies()))
                .map(cookies -> Arrays.stream(cookies).collect(Collectors.toList()))
                .orElse(Collections.emptyList());
    }

    /**
     * 获取并过滤cookie
     *
     * @param predicate 匹配
     * @return 结果
     */
    public static List<Cookie> getCookies(Predicate<? super Cookie> predicate) {
        return getRequest().flatMap(request -> Optional.ofNullable(request.getCookies()))
                .map(cookies -> Arrays.stream(cookies).filter(predicate).collect(Collectors.toList()))
                .orElse(Collections.emptyList());
    }

    /**
     * 获取鉴权（token）相关的cookie
     *
     * @return 结果
     */
    public static Optional<String> getCredential() {
        return getRequest().map(request -> request.getHeader(AUTHORIZATION_KEY))
                .filter(DataUtils::isNotBlank)
                .map(Optional::of)
                .orElseGet(() -> getCookies(cookie -> AUTHORIZATION_KEY.equals(cookie.getName())).stream()
                        .findAny().map(Cookie::getValue));
    }
}
