package id.unum

import com.ionspin.kotlin.bignum.integer.BigInteger
import com.ionspin.kotlin.bignum.integer.Sign
import id.unum.exceptions.ParseException


internal object Base58 {
    private const val BASE58CHARS = "123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz"
    private val FIFTY_EIGHT = BigInteger.parseString("58")

    /**
     * Takes a series of bytes and return a base58 encoded string
     *
     * @param bytes Array of bytes to include
     * @return Base58 encoded string
     */
    fun encode(bytes: ByteArray): String {
        val extraZero = ByteArray(bytes.size + 1)
        var result = ""
        extraZero[0] = 0
        arrayCopy(bytes, 0, extraZero, 1, bytes.size)
        var bn = BigInteger.fromByteArray(extraZero, Sign.POSITIVE)
        while (bn.compareTo(BigInteger.ZERO) == 1) {
            val div = bn.divideAndRemainder(FIFTY_EIGHT)
            bn = div.first
            val c = BASE58CHARS[div.second.intValue()]
            result += c
        }

        // pad leading zeros
        result = result.reversed()
        if (bytes.isNotEmpty()) {
            var i = 0
            while (bytes[i] == 0.toByte()) {
                result = BASE58CHARS[0].toString() + result
                i++
            }
        }
        return result
    }

    /**
     * Decode an encoded base58 string
     *
     * @param encoded Base58 encoded string
     * @return The decoded byte array
     */
    fun decode(encoded: String): ByteArray {
        var bn = BigInteger.ZERO
        var mult: BigInteger
        val reversed = encoded.reversed()
        var raw: ByteArray
        for (pos in reversed.indices) {
            val letter = reversed[pos]
            val value = BASE58CHARS.indexOf(letter)
            if (value == -1) {
                throw ParseException(reversed, pos)
            }
            mult = FIFTY_EIGHT.pow(pos)
            bn = bn.plus(mult.times(BigInteger.fromLong(value.toLong())))
        }
        raw = bn.toByteArray()
        if (raw[0] == 0.toByte()) {
            val raw2 = ByteArray(raw.size - 1)
            raw.copyInto(raw2, 0, 1, raw.size - 1)
            raw = raw2
        }
        var leadingZeroes = 0
        var pos = 0
        while (encoded[pos] == '1' && pos < encoded.length) {
            leadingZeroes++
            pos++
        }
        val fin = ByteArray(raw.size + leadingZeroes)
        raw.copyInto(fin, leadingZeroes, 0, raw.size)
        return fin
    }

    /**
     * Decodes a base58 encoded string with a checksum
     *
     * @param encoded The base58 encoded string with checksum
     * @return The decoded byte array
     * @throws ParseException
     */
//    @Throws(ParseException::class)
//    fun decodeCheck(encoded: String): ByteArray {
//        val raw = decode(encoded)
//        if (raw.size < 4) {
//            throw ParseException("Decoded bytes too small for checksum", 0)
//        }
//        val decoded = ByteArray(raw.size - 4)
//        System.arraycopy(raw, 0, decoded, 0, raw.size - 4)
//        val hash = doubleSha256(decoded)
//        for (i in 0..3) {
//            if (hash!![i] != raw[decoded.size + i]) {
//                throw ParseException("Incorrect checksum", decoded.size)
//            }
//        }
//        return decoded
//    }

    /**
     * Encodes a byte array into a base58 string with a checksum
     *
     * @param bytes The byte array to encode
     * @return The base58 encoded string with a checksum
     */
//    fun encodeCheck(bytes: ByteArray): String {
//        val hash = doubleSha256(bytes)
//        val toHash = ByteArray(bytes.size + 4)
//        System.arraycopy(bytes, 0, toHash, 0, bytes.size)
//        System.arraycopy(hash!!, 0, toHash, bytes.size, 4)
//        return encode(toHash)
//    }

    internal fun arrayCopy(src: ByteArray, srcIndex: Int, dest: ByteArray, destIndex: Int, size: Int) {
        for (i in 0 until size) {
            dest[destIndex + i] = src[srcIndex + i]
        }
    }
}
