/*
 * Decompiled with CFR 0.152.
 */
package com.cory.db.processor;

import com.cory.constant.Constants;
import com.cory.constant.ErrorCode;
import com.cory.context.CoryEnv;
import com.cory.db.annotations.Model;
import com.cory.db.config.CoryDbProperties;
import com.cory.db.jdbc.Column;
import com.cory.db.jdbc.CoryDb;
import com.cory.db.jdbc.CoryModelUtil;
import com.cory.db.jdbc.CorySqlBuilder;
import com.cory.db.jdbc.Table;
import com.cory.exception.CoryException;
import com.google.common.base.CaseFormat;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.data.util.AnnotatedTypeScanner;

public class CoryDbChecker
implements InitializingBean {
    private static final Logger log = LoggerFactory.getLogger(CoryDbChecker.class);
    private static final String COLUMN_SQL = "select * from information_schema.columns where table_schema = ?";
    private static final String BASE_COLUMNS_DDL = "`id` bigint(20) NOT NULL AUTO_INCREMENT COMMENT 'ID',`creator` bigint(20) NOT NULL COMMENT '\u521b\u5efa\u4eba',`modifier` bigint(20) NOT NULL COMMENT '\u4fee\u6539\u4eba',`create_time` datetime NOT NULL DEFAULT '1970-01-01 00:00:00' COMMENT '\u521b\u5efa\u65f6\u95f4',`modify_time` datetime NOT NULL DEFAULT '1970-01-01 00:00:00' COMMENT '\u4fee\u6539\u65f6\u95f4',`is_deleted` SMALLINT(1) NOT NULL DEFAULT 0 COMMENT '\u662f\u5426\u5220\u9664',";
    private static final String CREATE_TABLE_HEADER = "CREATE TABLE `%s` (";
    private static final String CREATE_TABLE_FOOTER = "PRIMARY KEY (`id`)) ENGINE=InnoDB AUTO_INCREMENT=10001 DEFAULT CHARSET=utf8mb4 COMMENT='%s';";
    private static final String DROP_COLUMN_SQL = "alter table %s drop column %s";
    private static final String ADD_COLUMN_SQL = "alter table %s add column %s";
    private static final String MODIFY_COLUMN_SQL = "alter table %s modify column %s";
    private CoryDbProperties coryDbProperties;
    private CoryDb coryDb;
    private String database;

    public CoryDbChecker(CoryDb coryDb, CoryDbProperties coryDbProperties, String database) {
        this.coryDb = coryDb;
        this.coryDbProperties = coryDbProperties;
        this.database = database;
    }

    public void afterPropertiesSet() throws Exception {
        Map<String, Table> newTableMap = this.queryNowTables();
        if (MapUtils.isEmpty(newTableMap)) {
            log.info("no Model found, skip db check.");
            return;
        }
        Map<String, Table> dbTableMap = this.queryNotNullDbTables();
        if (CoryEnv.IS_DEV) {
            log.info("check and sync db tables and columns from code to db.");
        }
        boolean same = true;
        StringBuilder msgBuilder = new StringBuilder();
        for (Map.Entry<String, Table> entry : newTableMap.entrySet()) {
            String tableName = entry.getKey();
            Table table = entry.getValue();
            Table dbTable = dbTableMap.get(tableName);
            if (null == dbTable) {
                same = false;
                this.createTableForDev(table, msgBuilder);
                continue;
            }
            if (table.equals(dbTable)) continue;
            same = false;
            msgBuilder.append("table " + tableName + " does not match in db:\n");
            List<Column> addColumnList = table.differentColumns(dbTable, Table.COLUMN_TYPE.ADD);
            List<Column> deleteColumnList = table.differentColumns(dbTable, Table.COLUMN_TYPE.DELETE);
            List<Column> modifyColumnList = table.differentColumns(dbTable, Table.COLUMN_TYPE.MODIFY);
            this.syncColumn(addColumnList, deleteColumnList, modifyColumnList, msgBuilder);
        }
        if (!same && CoryEnv.IS_PROD) {
            throw new CoryException(ErrorCode.DB_ERROR, new Object[]{msgBuilder.toString()});
        }
    }

    private void createTableForDev(Table table, StringBuilder msgBuilder) {
        StringBuilder builder = new StringBuilder();
        builder.append(String.format(CREATE_TABLE_HEADER, table.getName()));
        builder.append(BASE_COLUMNS_DDL);
        for (Column column : table.getColumnList()) {
            builder.append(column.buildDDL());
            builder.append(",");
        }
        builder.append(String.format(CREATE_TABLE_FOOTER, table.getComment()));
        String sql = builder.toString();
        if (CoryEnv.IS_DEV) {
            this.coryDb.executeSql(sql);
            log.info("create table " + table.getName() + ": " + sql);
        }
        if (CoryEnv.IS_PROD) {
            msgBuilder.append("table " + table.getName() + " does not exist, please create using follow sql;\n");
            msgBuilder.append(sql);
            msgBuilder.append("\n");
        }
    }

    private void syncColumn(List<Column> addColumnList, List<Column> deleteColumnList, List<Column> modifyColumnList, StringBuilder msgBuilder) {
        String sql;
        if (CollectionUtils.isNotEmpty(addColumnList)) {
            for (Column column : addColumnList) {
                if (CoryEnv.IS_DEV) {
                    sql = String.format(ADD_COLUMN_SQL, column.getTableName(), column.buildDDL());
                    this.coryDb.executeSql(sql);
                    log.info("add column " + column.getTableName() + "." + column.getName() + ": " + sql);
                }
                if (!CoryEnv.IS_PROD) continue;
                msgBuilder.append("\tplease add column: " + column.buildDDL() + ",\n");
            }
        }
        if (CollectionUtils.isNotEmpty(deleteColumnList)) {
            for (Column column : deleteColumnList) {
                if (CoryEnv.IS_DEV) {
                    sql = String.format(DROP_COLUMN_SQL, column.getTableName(), column.getName());
                    this.coryDb.executeSql(sql);
                    log.info("drop column " + column.getTableName() + "." + column.getName() + ": " + sql);
                }
                if (!CoryEnv.IS_PROD) continue;
                msgBuilder.append("\tplease drop column: " + column.buildDDL() + ",\n");
            }
        }
        if (CollectionUtils.isNotEmpty(modifyColumnList)) {
            for (Column column : modifyColumnList) {
                if (CoryEnv.IS_DEV) {
                    sql = String.format(MODIFY_COLUMN_SQL, column.getTableName(), column.buildDDL());
                    this.coryDb.executeSql(sql);
                    log.info("modify column " + column.getTableName() + "." + column.getName() + ": " + sql);
                }
                if (!CoryEnv.IS_PROD) continue;
                msgBuilder.append("\tplease modify column: " + column.buildDDL() + ",\n");
            }
        }
    }

    private Map<String, Table> queryNowTables() {
        AnnotatedTypeScanner scanner = new AnnotatedTypeScanner(true, new Class[]{Model.class});
        Set set = scanner.findTypes(this.coryDbProperties.getModelPackages());
        if (CollectionUtils.isEmpty((Collection)set)) {
            log.info("no model found in packages: {}", (Object[])this.coryDbProperties.getModelPackages());
            return null;
        }
        HashMap<String, Table> tableMap = new HashMap<String, Table>();
        HashMap<String, Integer> tableDuplicateMap = new HashMap<String, Integer>();
        for (Class cls : set) {
            Model model = cls.getAnnotation(Model.class);
            if (null == model || model.noTable()) continue;
            String tableName = CoryModelUtil.buildTableName(cls);
            Integer ct = (Integer)tableDuplicateMap.get(tableName);
            if (null == ct) {
                ct = 0;
            }
            tableDuplicateMap.put(tableName, ct + 1);
            Field[] fields = cls.getDeclaredFields();
            if (null == fields && fields.length <= 0) continue;
            for (Field javaField : fields) {
                Table table;
                com.cory.db.annotations.Field field = javaField.getAnnotation(com.cory.db.annotations.Field.class);
                if (null == field) continue;
                String columnName = CoryModelUtil.buildColumnName(javaField.getName());
                String columnDefault = field.defaultValue();
                boolean nullable = field.nullable();
                String columnType = field.type().buildDbType(field.len());
                String columnComment = field.comment();
                if (StringUtils.isBlank((CharSequence)columnDefault)) {
                    columnDefault = null;
                }
                if (null == (table = (Table)tableMap.get(tableName))) {
                    table = Table.builder().name(tableName).columnList(new ArrayList<Column>()).columnMap(new HashMap<String, Column>()).comment(model.name()).build();
                    tableMap.put(tableName, table);
                }
                Column column = Column.builder().name(columnName).tableName(tableName).defaultValue(columnDefault).nullable(nullable).columnType(columnType).columnComment(columnComment).build();
                table.getColumnList().add(column);
                table.getColumnMap().put(columnName, column);
            }
        }
        this.checkDuplicated(tableDuplicateMap);
        return tableMap;
    }

    private void checkDuplicated(Map<String, Integer> tableDuplicateMap) {
        boolean duplicated = false;
        StringBuilder msgBuilder = new StringBuilder();
        for (Map.Entry<String, Integer> entry : tableDuplicateMap.entrySet()) {
            if (entry.getValue() <= 1) continue;
            duplicated = true;
            msgBuilder.append("Dao table name: " + entry.getKey() + " is duplicated");
        }
        if (duplicated) {
            throw new CoryException(ErrorCode.DB_ERROR, new Object[]{msgBuilder.toString()});
        }
    }

    private Map<String, Table> queryNotNullDbTables() {
        CorySqlBuilder.CorySqlInfo sqlInfo = CorySqlBuilder.CorySqlInfo.builder().sql(COLUMN_SQL).params(Arrays.asList(this.database)).build();
        List<Map<String, Object>> list = this.coryDb.select(sqlInfo);
        if (CollectionUtils.isEmpty(list)) {
            return new HashMap<String, Table>();
        }
        HashMap<String, Table> tableMap = new HashMap<String, Table>();
        for (Map<String, Object> map : list) {
            String tableName = (String)map.get("TABLE_NAME");
            String columnName = (String)map.get("COLUMN_NAME");
            String columnDefault = (String)map.get("COLUMN_DEFAULT");
            boolean nullable = "YES".equals(map.get("IS_NULLABLE"));
            String columnType = (String)map.get("COLUMN_TYPE");
            String columnComment = (String)map.get("COLUMN_COMMENT");
            if (this.isBaseModelColumn(columnName)) continue;
            Table table = (Table)tableMap.get(tableName);
            if (null == table) {
                table = Table.builder().name(tableName).columnList(new ArrayList<Column>()).columnMap(new HashMap<String, Column>()).build();
                tableMap.put(tableName, table);
            }
            Column column = Column.builder().name(columnName).tableName(tableName).defaultValue(columnDefault).nullable(nullable).columnType(columnType).columnComment(columnComment).build();
            table.getColumnList().add(column);
            table.getColumnMap().put(columnName, column);
        }
        return tableMap;
    }

    private boolean isBaseModelColumn(String columnName) {
        for (String baseModelColumn : Constants.BASE_MODEL_COLUMNS) {
            if (!(baseModelColumn = CaseFormat.LOWER_CAMEL.to(CaseFormat.LOWER_UNDERSCORE, baseModelColumn).toUpperCase()).equals(columnName = columnName.toUpperCase())) continue;
            return true;
        }
        return false;
    }
}

