/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.d2.balancer.strategies;

import com.linkedin.common.callback.Callback;
import com.linkedin.common.util.None;
import com.linkedin.d2.balancer.clients.TrackerClient;
import com.linkedin.d2.balancer.strategies.degrader.DegraderLoadBalancerStrategyConfig;
import com.linkedin.d2.balancer.util.healthcheck.HealthCheck;
import com.linkedin.d2.balancer.util.healthcheck.HealthCheckClientBuilder;
import com.linkedin.d2.balancer.util.healthcheck.HealthCheckOperations;
import com.linkedin.util.RateLimitedLogger;
import com.linkedin.util.clock.Clock;
import java.net.URISyntaxException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LoadBalancerQuarantine {
    private static final Logger _log = LoggerFactory.getLogger(LoadBalancerQuarantine.class);
    private static final long ERROR_REPORT_PERIOD = 60000L;
    private static final long QUARANTINE_MIN_REENTRY_TIME_MS = 30000L;
    private final TrackerClient _trackerClient;
    private final HealthCheck _healthCheckClient;
    private final String _serviceName;
    private final ScheduledExecutorService _executorService;
    private final Clock _clock;
    private final long _timeBetweenHC;
    private final long _updateIntervalMs;
    private volatile QuarantineStates _quarantineState;
    private volatile boolean _isShutdown;
    private long _lastChecked;
    private long _timeTilNextCheck;
    private final RateLimitedLogger _rateLimitedLogger;

    public LoadBalancerQuarantine(TrackerClient client, DegraderLoadBalancerStrategyConfig config, String serviceName) {
        this(client, config.getExecutorService(), config.getClock(), config.getUpdateIntervalMs(), config.getQuarantineLatency(), config.getHealthCheckMethod(), config.getHealthCheckPath(), serviceName, config.getServicePath(), config.getHealthCheckOperations());
    }

    public LoadBalancerQuarantine(TrackerClient trackerClient, ScheduledExecutorService executorService, Clock clock, long updateIntervalMs, long quarantineLatency, String healthCheckMethod, String healthCheckPath, String serviceName, String servicePath, HealthCheckOperations healthCheckOperations) {
        this._trackerClient = trackerClient;
        this._executorService = executorService;
        this._clock = clock;
        this._timeBetweenHC = 1000L;
        this._serviceName = serviceName;
        this._quarantineState = QuarantineStates.FAILURE;
        this._timeTilNextCheck = updateIntervalMs;
        this._updateIntervalMs = updateIntervalMs;
        this._lastChecked = Integer.MIN_VALUE;
        this._isShutdown = false;
        this._rateLimitedLogger = new RateLimitedLogger(_log, 60000L, clock);
        if (this._timeBetweenHC < quarantineLatency) {
            _log.error("Illegal quarantine configurations for service {}: Interval {} too short", (Object)this._serviceName, (Object)this._timeBetweenHC);
            throw new IllegalArgumentException("Quarantine interval too short");
        }
        HealthCheck healthCheckClient = null;
        try {
            healthCheckClient = new HealthCheckClientBuilder().setHealthCheckOperations(healthCheckOperations).setHealthCheckPath(healthCheckPath).setServicePath(servicePath).setClock(clock).setLatency(quarantineLatency).setMethod(healthCheckMethod).setClient(this._trackerClient).build();
        }
        catch (URISyntaxException e) {
            _log.error("Error to generate healthCheckClient", (Throwable)e);
        }
        this._healthCheckClient = healthCheckClient;
    }

    private void healthCheckNTimes(final int n) {
        if (n <= 0 || this._isShutdown) {
            return;
        }
        final long startTime = this._clock.currentTimeMillis();
        Callback<None> healthCheckCallback = new Callback<None>(){

            public void onError(Throwable e) {
                LoadBalancerQuarantine.this._rateLimitedLogger.warn("Healthchecking failed for {} (service={}): {}", new Object[]{LoadBalancerQuarantine.this._trackerClient.getUri(), LoadBalancerQuarantine.this._serviceName, e});
                LoadBalancerQuarantine.this._quarantineState = QuarantineStates.FAILURE;
            }

            public void onSuccess(None result) {
                if (n > 1) {
                    if (!LoadBalancerQuarantine.this._isShutdown) {
                        long nextCheckDelay = LoadBalancerQuarantine.this._timeBetweenHC - (LoadBalancerQuarantine.this._clock.currentTimeMillis() - startTime);
                        if (nextCheckDelay > 0L) {
                            LoadBalancerQuarantine.this._executorService.schedule(() -> LoadBalancerQuarantine.this.healthCheckNTimes(n - 1), nextCheckDelay, TimeUnit.MILLISECONDS);
                        } else {
                            _log.error("Delay exceeded the defined checking interval");
                        }
                    }
                } else {
                    LoadBalancerQuarantine.this._quarantineState = QuarantineStates.SUCCESS;
                }
            }
        };
        this._healthCheckClient.checkHealth(healthCheckCallback);
    }

    public boolean checkUpdateQuarantineState() {
        this._lastChecked = this._clock.currentTimeMillis();
        int repeatNum = 5;
        switch (this._quarantineState) {
            case DISABLED: {
                throw new IllegalStateException("State update for disabled quarantine");
            }
            case FAILURE: {
                if (this._isShutdown) {
                    _log.error("Could not check quarantine state since the executor is shutdown");
                    break;
                }
                this._executorService.schedule(() -> this.healthCheckNTimes(repeatNum), this._timeTilNextCheck, TimeUnit.MILLISECONDS);
                this._timeTilNextCheck *= 2L;
                this._quarantineState = QuarantineStates.WAIT;
                break;
            }
            case WAIT: {
                if (this._timeTilNextCheck <= 60000L) break;
                this._rateLimitedLogger.error("Client {}  for service {} is being kept in quarantine for {} seconds, Please check to make sure it is healthy", new Object[]{this._trackerClient.getUri(), this._serviceName, 1.0 * (double)this._timeTilNextCheck / 1000.0});
                break;
            }
            case SUCCESS: {
                this._quarantineState = QuarantineStates.DISABLED;
                _log.info("checkUpdateQuarantineState: quarantine state for client {} service {} is DISABLED", (Object)this._trackerClient.getUri(), (Object)this._serviceName);
                return true;
            }
        }
        return false;
    }

    public void shutdown() {
        if (this._isShutdown) {
            _log.error("Quarantine already shutdown");
            return;
        }
        this._isShutdown = true;
    }

    public void reset(long currentTime) {
        boolean resetInterval;
        this._quarantineState = QuarantineStates.FAILURE;
        boolean bl = resetInterval = currentTime - this.getLastChecked() > 30000L;
        if (resetInterval) {
            this._timeTilNextCheck = this._updateIntervalMs;
        } else {
            _log.warn("HealthCheck: Interval {}ms is not reset for client {}, because it is quarantined again within 30s. This can happen if current health checking method is not sufficient for capturing when a node should stay in quarantine, for example it returns fast but the real queries return slow.", (Object)this._timeTilNextCheck, (Object)this._trackerClient.getUri());
        }
    }

    public long getLastChecked() {
        return this._lastChecked;
    }

    public long getTimeTilNextCheck() {
        return this._timeTilNextCheck;
    }

    public boolean isInQuarantine() {
        return this._quarantineState == QuarantineStates.FAILURE || this._quarantineState == QuarantineStates.WAIT;
    }

    public HealthCheck getHealthCheckClient() {
        return this._healthCheckClient;
    }

    public String toString() {
        return "TrackerClientQuarantine [_client=" + this._trackerClient.getUri() + ", _quarantineState=" + (Object)((Object)this._quarantineState) + ", _timeTilNextCheck=" + this._timeTilNextCheck / 1000L + "s]";
    }

    private static enum QuarantineStates {
        FAILURE,
        WAIT,
        SUCCESS,
        DISABLED;

    }
}

