/*
 * Decompiled with CFR 0.152.
 */
package io.apiman.manager.api.jpa.blobstore;

import com.google.common.base.Preconditions;
import com.google.common.collect.Iterators;
import com.google.common.io.FileBackedOutputStream;
import io.apiman.common.logging.ApimanLoggerFactory;
import io.apiman.common.logging.IApimanLogger;
import io.apiman.manager.api.beans.blobs.BlobEntity;
import io.apiman.manager.api.beans.download.BlobDto;
import io.apiman.manager.api.beans.download.BlobMapper;
import io.apiman.manager.api.beans.download.BlobRef;
import io.apiman.manager.api.beans.download.ExportedBlobDto;
import io.apiman.manager.api.core.IBlobStore;
import io.apiman.manager.api.core.exceptions.StorageException;
import io.apiman.manager.api.jpa.blobstore.BlobStoreRepository;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.time.OffsetDateTime;
import java.util.Iterator;
import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import javax.annotation.PostConstruct;
import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import javax.transaction.Transactional;
import net.jpountz.xxhash.StreamingXXHash64;
import net.jpountz.xxhash.XXHashFactory;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.engine.jdbc.BlobProxy;
import org.jetbrains.annotations.NotNull;

@ApplicationScoped
@Transactional
public class SqlBlobStoreService
implements IBlobStore {
    private final IApimanLogger LOGGER = ApimanLoggerFactory.getLogger(SqlBlobStoreService.class);
    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor();
    private BlobStoreRepository blobStoreRepository;
    private final BlobMapper mapper = BlobMapper.INSTANCE;

    @Inject
    public SqlBlobStoreService(BlobStoreRepository blobStoreRepository) {
        this.blobStoreRepository = blobStoreRepository;
    }

    public SqlBlobStoreService() {
    }

    @PostConstruct
    public void runReaper() {
        this.executor.scheduleAtFixedRate(() -> {
            this.LOGGER.debug("Scheduled task reaping old unattached SQL blobs");
            OffsetDateTime anHourAgo = OffsetDateTime.now().minusHours(1L);
            this.blobStoreRepository.deleteUnattachedByAge(anHourAgo);
        }, 1L, 1L, TimeUnit.HOURS);
    }

    public void attachToBlob(String id) {
        if (id == null) {
            return;
        }
        Preconditions.checkArgument((boolean)StringUtils.isNotBlank((CharSequence)id), (Object)"id name must not be blank");
        this.blobStoreRepository.increaseRefCount(id);
    }

    public BlobRef storeBlob(@NotNull String name, @NotNull String mimeType, @NotNull FileBackedOutputStream fbos) {
        return this.storeBlob(name, mimeType, fbos, 1);
    }

    public BlobRef storeBlob(@NotNull String name, @NotNull String mimeType, @NotNull FileBackedOutputStream fbos, int initRefCount) {
        Preconditions.checkArgument((boolean)StringUtils.isNotBlank((CharSequence)name), (Object)"Blob name must not be blank");
        Preconditions.checkArgument((initRefCount >= 0 ? 1 : 0) != 0, (Object)"Init ref count must be gte 0");
        String resourceId = this.calculateOid(name);
        try {
            long hash = this.hashBlob(fbos);
            BlobEntity blobEntity = new BlobEntity().setId(resourceId).setName(name).setMimeType(mimeType).setBlob(BlobProxy.generateProxy((InputStream)fbos.asByteSource().openStream(), (long)fbos.asByteSource().size())).setRefCount(initRefCount).setHash(Long.valueOf(hash));
            return this.toBlobRef(this.deduplicateOrStore(name, mimeType, hash, blobEntity));
        }
        catch (StorageException e) {
            throw new RuntimeException(e);
        }
        catch (IOException ioe) {
            throw new UncheckedIOException(ioe);
        }
    }

    public BlobRef storeBlob(@NotNull String name, @NotNull String mimeType, byte[] bytes) {
        return this.storeBlob(name, mimeType, bytes, 1);
    }

    public BlobRef storeBlob(@NotNull String name, @NotNull String mimeType, byte[] bytes, int initRefCount) {
        Preconditions.checkArgument((boolean)StringUtils.isNotBlank((CharSequence)name), (Object)"Blob name must not be blank");
        String resourceId = this.calculateOid(name);
        long hash = this.hashBlob(bytes);
        BlobEntity blobEntity = new BlobEntity().setId(resourceId).setName(name).setMimeType(mimeType).setBlob(BlobProxy.generateProxy((byte[])bytes)).setRefCount(initRefCount).setHash(Long.valueOf(hash));
        try {
            return this.toBlobRef(this.deduplicateOrStore(name, mimeType, hash, blobEntity));
        }
        catch (StorageException e) {
            throw new RuntimeException(e);
        }
    }

    public BlobDto getBlob(@NotNull String oid) {
        try {
            return this.mapper.toDto(this.blobStoreRepository.getById(oid));
        }
        catch (StorageException e) {
            throw new RuntimeException(e);
        }
    }

    public SqlBlobStoreService remove(@NotNull String oid) {
        try {
            this.blobStoreRepository.deleteById(oid);
            return this;
        }
        catch (StorageException e) {
            throw new RuntimeException(e);
        }
    }

    public IBlobStore reimportBlob(ExportedBlobDto blobDto) {
        try {
            FileBackedOutputStream fbos = blobDto.getBlob();
            long hash = this.hashBlob(blobDto.getBlob());
            BlobEntity blobEntity = new BlobEntity().setId(blobDto.getId()).setName(blobDto.getName()).setMimeType(blobDto.getMimeType()).setBlob(BlobProxy.generateProxy((InputStream)fbos.asByteSource().openStream(), (long)fbos.asByteSource().size())).setRefCount(blobDto.getRefCount()).setHash(Long.valueOf(hash));
            this.blobStoreRepository.create(blobEntity);
        }
        catch (IOException ioe) {
            throw new UncheckedIOException(ioe);
        }
        catch (StorageException e) {
            throw new RuntimeException(e);
        }
        return this;
    }

    public Iterator<ExportedBlobDto> getAll() {
        try {
            return Iterators.transform(this.blobStoreRepository.getAll(), arg_0 -> ((BlobMapper)this.mapper).toExportedDto(arg_0));
        }
        catch (StorageException e) {
            throw new RuntimeException(e);
        }
    }

    private String calculateOid(String name) {
        String safeName = name.replaceAll("[\\P{Print}\\W]+", "_");
        return UUID.randomUUID().toString().substring(14) + "/" + safeName;
    }

    private long hashBlob(byte[] bytes) {
        return XXHashFactory.fastestInstance().hash64().hash(bytes, 0, bytes.length, 0L);
    }

    private long hashBlob(FileBackedOutputStream fbos) {
        try {
            int len;
            StreamingXXHash64 streamingHash = XXHashFactory.fastestInstance().newStreamingHash64(0L);
            InputStream fbosStream = fbos.asByteSource().openBufferedStream();
            byte[] hashBuffer = new byte[8192];
            while ((len = fbosStream.read(hashBuffer)) != -1) {
                streamingHash.update(hashBuffer, 0, len);
            }
            return streamingHash.getValue();
        }
        catch (IOException ioe) {
            throw new UncheckedIOException(ioe);
        }
    }

    private BlobEntity deduplicateOrStore(String name, String mimeType, long hash, BlobEntity candidate) throws StorageException {
        BlobEntity duplicate = this.blobStoreRepository.getByNaturalId(name, mimeType, hash);
        if (duplicate == null) {
            this.blobStoreRepository.create(candidate);
            return candidate;
        }
        this.blobStoreRepository.increaseRefCount(duplicate.getId());
        return duplicate;
    }

    private BlobRef toBlobRef(BlobEntity entity) {
        return new BlobRef().setId(entity.getId()).setName(entity.getName()).setMimeType(entity.getMimeType()).setHash(entity.getHash().longValue()).setCreatedOn(entity.getCreatedOn()).setModifiedOn(entity.getModifiedOn());
    }
}

