package io.avaje.inject.generator;

import static io.avaje.inject.generator.APContext.*;
import static io.avaje.inject.generator.ProcessingContext.*;

import javax.annotation.processing.*;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.AnnotationMirror;
import javax.lang.model.element.Element;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.TypeElement;
import javax.lang.model.util.ElementFilter;
import javax.lang.model.util.Elements;
import javax.tools.StandardLocation;

import io.avaje.prism.GenerateAPContext;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URI;
import java.util.*;
import java.util.stream.Collectors;

@GenerateAPContext
@SupportedAnnotationTypes({
  Constants.INJECTMODULE,
  Constants.FACTORY,
  Constants.SINGLETON,
  Constants.COMPONENT,
  Constants.PROTOTYPE,
  Constants.SCOPE,
  Constants.TESTSCOPE,
  Constants.CONTROLLER,
  ImportPrism.PRISM_TYPE,
  AspectImportPrism.PRISM_TYPE
})
public final class Processor extends AbstractProcessor {

  private Elements elementUtils;
  private ScopeInfo defaultScope;
  private AllScopes allScopes;
  private boolean readModuleInfo;
  private final Set<String> pluginFileProvided = new HashSet<>();
  private final Set<String> moduleFileProvided = new HashSet<>();

  @Override
  public SourceVersion getSupportedSourceVersion() {
    return SourceVersion.latest();
  }

  @Override
  public synchronized void init(ProcessingEnvironment processingEnv) {
    super.init(processingEnv);
    loadProvidedFiles(processingEnv.getFiler());
    ProcessingContext.init(processingEnv, moduleFileProvided);
    this.elementUtils = processingEnv.getElementUtils();
    this.allScopes = new AllScopes();
    this.defaultScope = allScopes.defaultScope();
    ExternalProvider.registerPluginProvidedTypes(defaultScope);
    pluginFileProvided.forEach(defaultScope::pluginProvided);
  }

  /**
   * Loads provider files generated by avaje-inject-maven-plugin
   */
  void loadProvidedFiles(Filer filer) {
    pluginFileProvided.addAll(lines(filer, "target/avaje-plugin-provides.txt", "/target/classes"));
    moduleFileProvided.addAll(lines(filer, "target/avaje-module-provides.txt", "/target/classes"));
    pluginFileProvided.addAll(lines(filer, "build/avaje-plugin-provides.txt", "/build/classes/java/main"));
    moduleFileProvided.addAll(lines(filer, "build/avaje-module-provides.txt", "/build/classes/java/main"));
  }

  private static List<String> lines(Filer filer, String relativeName, String replace) {
    try {
      final String resource = resource(filer, relativeName, replace);
      try (var inputStream = new URI(resource).toURL().openStream();
           var reader = new BufferedReader(new InputStreamReader(inputStream))) {
        return reader.lines().collect(Collectors.toList());
      }
    } catch (final Exception e) {
      return Collections.emptyList();
    }
  }

  private static String resource(Filer filer, String relativeName, String replace) throws IOException {
    return filer
      .getResource(StandardLocation.CLASS_OUTPUT, "", relativeName)
      .toUri()
      .toString()
      .replace(replace, "");
  }

  @Override
  public boolean process(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) {

    APContext.setProjectModuleElement(annotations, roundEnv);
    readModule(roundEnv);
    addImportedAspects(importedAspects(roundEnv));
    readScopes(roundEnv.getElementsAnnotatedWith(typeElement(Constants.SCOPE)));
    readFactories(roundEnv.getElementsAnnotatedWith(typeElement(Constants.FACTORY)));
    if (defaultScope.includeSingleton()) {
      readBeans(roundEnv.getElementsAnnotatedWith(typeElement(Constants.SINGLETON)));
    }
    readBeans(roundEnv.getElementsAnnotatedWith(typeElement(Constants.COMPONENT)));
    readBeans(roundEnv.getElementsAnnotatedWith(typeElement(Constants.PROTOTYPE)));
    readImported(importedElements(roundEnv));
    readBeans(roundEnv.getElementsAnnotatedWith(typeElement(Constants.PROTOTYPE)));
    final var typeElement = elementUtils.getTypeElement(Constants.CONTROLLER);
    if (typeElement != null) {
      readBeans(roundEnv.getElementsAnnotatedWith(typeElement));
    }
    readBeans(roundEnv.getElementsAnnotatedWith(typeElement(Constants.PROXY)));
    allScopes.readBeans(roundEnv);
    defaultScope.write(roundEnv.processingOver());
    allScopes.write(roundEnv.processingOver());

    if (roundEnv.processingOver()) {
      ProcessingContext.clear();
    }
    return false;
  }

  private Set<TypeElement> importedElements(RoundEnvironment roundEnv) {
    return roundEnv.getElementsAnnotatedWith(typeElement(ImportPrism.PRISM_TYPE)).stream()
      .map(ImportPrism::getInstanceOn)
      .flatMap(p -> p.value().stream())
      .map(ProcessingContext::asElement)
      .filter(this::notAlreadyProvided)
      .collect(Collectors.toSet());
  }

  private boolean notAlreadyProvided(TypeElement e) {
    final String type = e.getQualifiedName().toString();
    return !moduleFileProvided.contains(type) && !pluginFileProvided.contains(type);
  }

  private static Map<String, AspectImportPrism> importedAspects(RoundEnvironment roundEnv) {
    return Optional.ofNullable(typeElement(AspectImportPrism.PRISM_TYPE))
      .map(roundEnv::getElementsAnnotatedWith)
      .stream()
      .flatMap(Set::stream)
      .map(AspectImportPrism::getInstanceOn)
      .collect(Collectors.toMap(p -> p.value().toString(), p -> p));
  }

  private void readScopes(Set<? extends Element> scopes) {
    for (final Element element : scopes) {
      if ((element.getKind() == ElementKind.ANNOTATION_TYPE) && (element instanceof TypeElement)) {
        final var type = (TypeElement) element;
        allScopes.addScopeAnnotation(type);
      }
    }
    addTestScope();
  }

  /**
   * Add built-in test scope for <code>@TestScope</code> if available.
   */
  private void addTestScope() {
    final var testScopeType = elementUtils.getTypeElement(Constants.TESTSCOPE);
    if (testScopeType != null) {
      allScopes.addScopeAnnotation(testScopeType);
    }
  }

  private void readFactories(Set<? extends Element> beans) {
    readChangedBeans(ElementFilter.typesIn(beans), true, false);
  }

  private void readBeans(Set<? extends Element> beans) {
    readChangedBeans(ElementFilter.typesIn(beans), false, false);
  }

  private void readImported(Set<? extends Element> beans) {
    readChangedBeans(ElementFilter.typesIn(beans), false, true);
  }

  /** Read the beans that have changed. */
  private void readChangedBeans(Set<TypeElement> beans, boolean factory, boolean importedComponent) {
    for (final var typeElement : beans) {
      if (typeElement.getKind() == ElementKind.INTERFACE) {
        continue;
      }
      final var scope = findScope(typeElement);
      if (!factory) {
        // will be found via custom scope so effectively ignore additional @Singleton
        if (scope == null) {
          defaultScope.read(typeElement, false, importedComponent);
        }
      } else if (scope != null) {
        scope.read(typeElement, true, false);
      } else {
        defaultScope.read(typeElement, true, false);
      }
    }
  }

  /**
   * Find the scope if the Factory has a scope annotation.
   */
  private ScopeInfo findScope(Element element) {
    for (final AnnotationMirror annotationMirror : element.getAnnotationMirrors()) {
      final var scopeInfo = allScopes.get(annotationMirror.getAnnotationType().toString());
      if (scopeInfo != null) {
        return scopeInfo;
      }
    }
    return null;
  }

  /**
   * Read the existing meta-data from InjectModule (if found) and the factory bean (if exists).
   */
  private void readModule(RoundEnvironment roundEnv) {
    if (readModuleInfo) {
      // only read the module meta data once
      return;
    }
    readModuleInfo = true;
    final var factory = loadMetaInfServices();
    if (factory != null) {
      final var moduleType = elementUtils.getTypeElement(factory);
      if (moduleType != null) {
        defaultScope.readModuleMetaData(moduleType);
      }
    }
    allScopes.readModules(loadMetaInfCustom());
    readInjectModule(roundEnv);
  }

  /**
   * Read InjectModule for things like package-info etc (not for custom scopes)
   */
  private void readInjectModule(RoundEnvironment roundEnv) {
    // read other that are annotated with InjectModule
    for (final Element element : roundEnv.getElementsAnnotatedWith(typeElement(Constants.INJECTMODULE))) {
      final var scope = ScopePrism.getInstanceOn(element);
      if (scope == null) {
        // it it not a custom scope annotation
        final var annotation = InjectModulePrism.getInstanceOn(element);
        if (annotation != null) {
          defaultScope.details(annotation.name(), element);
        }
      }
    }
  }
}
