/*
 * Decompiled with CFR 0.152.
 */
package ortus.boxlang.runtime.loader.util;

import java.io.File;
import java.io.IOException;
import java.lang.annotation.Annotation;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.net.URLDecoder;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.jar.JarFile;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ortus.boxlang.runtime.BoxRuntime;
import ortus.boxlang.runtime.types.exceptions.BoxRuntimeException;

public class ClassDiscovery {
    private static final Logger logger = LoggerFactory.getLogger(ClassDiscovery.class);
    private static final String JAR_FILE_EXTENSION = "jar";
    private static final String CLASS_FILE_EXTENSION = ".class";

    public static Stream<String> getClassFilesAsStream(String packageName, Boolean recursive) throws IOException {
        ClassLoader classLoader = BoxRuntime.getInstance().getClass().getClassLoader();
        String path = packageName.replace('.', '/');
        Enumeration<URL> resources = classLoader.getResources(path);
        return Collections.list(resources).stream().flatMap(url -> {
            if (url.getProtocol().equals(JAR_FILE_EXTENSION)) {
                List<Class<?>> classesFound = ClassDiscovery.findClassesInJar(url, path, classLoader, new Class[0]);
                return classesFound.stream().map(clazz -> clazz.getName());
            }
            File directory = new File(url.getFile());
            return Stream.of(ClassDiscovery.findClassNames(directory, packageName, recursive));
        });
    }

    public static String[] getClassFiles(String packageName, Boolean recursive) throws IOException {
        return (String[])ClassDiscovery.getClassFilesAsStream(packageName, recursive).toArray(String[]::new);
    }

    public static Class<?>[] loadClassFiles(String packageName, Boolean recursive) throws IOException {
        return (Class[])((Stream)ClassDiscovery.getClassFilesAsStream(packageName, recursive).parallel()).map(className -> {
            try {
                return Class.forName(className);
            }
            catch (ClassNotFoundException e) {
                e.printStackTrace();
                return null;
            }
        }).toArray(Class[]::new);
    }

    @SafeVarargs
    public static Stream<Class<?>> findAnnotatedClasses(String startDir, ClassLoader targetLoader, String packageName, Class<? extends Annotation> ... annotations) {
        ArrayList classes = new ArrayList();
        if (packageName == null) {
            packageName = startDir.replace('/', '.');
            packageName = packageName.replace('\\', '.');
        }
        try {
            Enumeration<URL> resources;
            if (targetLoader instanceof URLClassLoader) {
                URLClassLoader URLTargetLoader = (URLClassLoader)targetLoader;
                resources = URLTargetLoader.findResources(startDir);
            } else {
                resources = targetLoader.getResources(startDir);
            }
            while (resources.hasMoreElements()) {
                URL resource = resources.nextElement();
                if (resource.getProtocol().equals(JAR_FILE_EXTENSION)) {
                    logger.debug("FindAnnotatedClasses: Jar file found: {}", (Object)resource);
                    classes.addAll(ClassDiscovery.findClassesInJar(resource, startDir, targetLoader, annotations));
                    continue;
                }
                logger.debug("FindAnnotatedClasses: Normal directory found: {}", (Object)resource);
                classes.addAll(ClassDiscovery.findClassesInDirectory(new File(resource.getFile()), packageName, targetLoader, annotations));
            }
        }
        catch (Exception e) {
            throw new BoxRuntimeException("Exception finding annotated classes in path " + startDir, e);
        }
        return classes.stream();
    }

    @SafeVarargs
    public static Stream<Class<?>> findAnnotatedClasses(String startDir, Class<? extends Annotation> ... annotations) {
        return ClassDiscovery.findAnnotatedClasses(startDir, ClassDiscovery.class.getClassLoader(), null, annotations);
    }

    @SafeVarargs
    public static Stream<Class<?>> findAnnotatedClasses(String startDir, String packageName, Class<? extends Annotation> ... annotations) {
        return ClassDiscovery.findAnnotatedClasses(startDir, ClassDiscovery.class.getClassLoader(), packageName, annotations);
    }

    public static File getFileFromResource(String resourceName) {
        return ClassDiscovery.getPathFromResource(resourceName).toFile();
    }

    public static Path getPathFromResource(String resourceName) {
        URL resourceUrl = ClassDiscovery.class.getClassLoader().getResource(resourceName);
        if (resourceUrl == null) {
            throw new BoxRuntimeException("Resource not found: " + resourceName);
        }
        try {
            return Paths.get(resourceUrl.toURI());
        }
        catch (URISyntaxException e) {
            throw new BoxRuntimeException(String.format("Cannot build an URI from the discovered resource %s", resourceUrl), e);
        }
    }

    public static String[] findClassNames(File directory, String packageName, Boolean recursive) {
        File[] aClassNames = directory.listFiles();
        if (aClassNames == null) {
            return new String[0];
        }
        return (String[])((Stream)Arrays.stream(aClassNames).parallel()).flatMap(file -> {
            if (file.isDirectory()) {
                if (recursive.booleanValue()) {
                    return Arrays.stream(ClassDiscovery.findClassNames(file, packageName + "." + file.getName(), recursive));
                }
                return Stream.empty();
            }
            return file.getName().endsWith(CLASS_FILE_EXTENSION) ? Stream.of(packageName + "." + file.getName().replace(CLASS_FILE_EXTENSION, "")) : Stream.empty();
        }).toArray(String[]::new);
    }

    public static List<Class<?>> findClassesInJar(URL jarURL, String startDir, ClassLoader classLoader, Class<? extends Annotation>[] annotations) {
        List<Class<?>> classes;
        String jarPath = jarURL.getPath().substring(5, jarURL.getPath().indexOf("!"));
        try (JarFile jar = new JarFile(URLDecoder.decode(jarPath, "UTF-8"));){
            classes = ((Stream)jar.stream().parallel()).filter(entry -> entry.getName().startsWith(startDir) && entry.getName().endsWith(CLASS_FILE_EXTENSION)).map(entry -> entry.getName().replace('/', '.').substring(0, entry.getName().length() - 6)).map(className -> {
                try {
                    return Class.forName(className, false, classLoader);
                }
                catch (ClassNotFoundException e) {
                    logger.error("Class not found: {}", className, (Object)e);
                    return null;
                }
            }).filter(clazz -> clazz != null && ClassDiscovery.isAnnotated(clazz, annotations)).collect(Collectors.toList());
        }
        catch (IOException e) {
            logger.error("Error while processing JAR file", e);
            throw new BoxRuntimeException("Error while processing JAR file", e);
        }
        return classes;
    }

    public static List<Class<?>> findClassesInDirectory(File directory, String packageName, ClassLoader classLoader, Class<? extends Annotation>[] annotations) {
        List<Class<?>> classes;
        Path directoryPath = directory.toPath();
        try (Stream<Path> pathStream = Files.walk(directoryPath, new FileVisitOption[0]);){
            classes = ((Stream)pathStream.parallel()).filter(path -> path.toString().endsWith(CLASS_FILE_EXTENSION)).map(path -> directoryPath.relativize((Path)path).toString().replace(File.separator, ".")).map(path -> packageName + "." + StringUtils.replaceIgnoreCase(path, CLASS_FILE_EXTENSION, "")).map(className -> {
                try {
                    return Class.forName(className, false, classLoader);
                }
                catch (ClassNotFoundException e) {
                    logger.error("Class not found: {}", className, (Object)e);
                    return null;
                }
            }).filter(clazz -> clazz != null && ClassDiscovery.isAnnotated(clazz, annotations)).collect(Collectors.toList());
        }
        catch (IOException e) {
            throw new BoxRuntimeException("Exception finding classes in directory " + String.valueOf(directory), e);
        }
        return classes;
    }

    public static boolean isAnnotated(Class<?> clazz, Class<? extends Annotation>[] annotations) {
        return annotations.length > 0 ? Arrays.stream(annotations).anyMatch(clazz::isAnnotationPresent) : true;
    }
}

