/**
 * [BoxLang]
 *
 * Copyright [2024] [Ortus Solutions, Corp]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ortus.boxlang.web.components;

import java.io.File;
import java.util.Set;

import ortus.boxlang.runtime.components.Attribute;
import ortus.boxlang.runtime.components.BoxComponent;
import ortus.boxlang.runtime.components.Component;
import ortus.boxlang.runtime.context.IBoxContext;
import ortus.boxlang.runtime.dynamic.casters.StringCaster;
import ortus.boxlang.runtime.scopes.Key;
import ortus.boxlang.runtime.types.IStruct;
import ortus.boxlang.runtime.types.exceptions.AbortException;
import ortus.boxlang.runtime.util.FileSystemUtil;
import ortus.boxlang.runtime.validation.Validator;
import ortus.boxlang.web.context.WebRequestBoxContext;
import ortus.boxlang.web.exchange.IBoxHTTPExchange;

@BoxComponent( allowsBody = true )
public class Content extends Component {

	public Content() {
		super();
		declaredAttributes = new Attribute[] {
		    new Attribute( Key.type, "string", Set.of( Validator.NON_EMPTY ) ),
		    new Attribute( Key.deleteFile, "boolean", false ),
		    new Attribute( Key.file, "string" ),
		    new Attribute( Key.variable, "any" ),
		    new Attribute( Key.reset, "boolean", true )
		};
	}

	/**
	 * Does either or both of the following:
	 * - Sets the MIME content encoding header for the current page
	 * - Sends the contents of a file from the server as the page output
	 *
	 * @param context        The context in which the Component is being invoked
	 * @param attributes     The attributes to the Component
	 * @param body           The body of the Component
	 * @param executionState The execution state of the Component
	 *
	 *
	 * @atribute.type The MIME content type of the page, optionally followed by a semicolon and the character encoding. By default, sends pages as
	 *                text/html content type in the UTF-8 character encoding.
	 *
	 * @arguments.deleteFile Applies only if you specify a file with the file attribute.
	 *                       - Yes: deletes the file on the server after sending its contents to the client.
	 *                       - No: leaves the file on the server.
	 *
	 * @atribute.file Name of file whose contents will be the page output. The file attribute must refer to a path on the system on which the web server
	 *                runs. When you use this attribute, any other output on the current page is ignored; only the contents of the file is sent to the
	 *                client.
	 *
	 * @atribute.variable Name of a variable whose contents can be displayed by the browser, such as the contents of a chart generated by the chart
	 *                    component
	 *                    or a PDF or Excel file retrieved by a file action="readBinary" component. When you use this attribute, any other output on the
	 *                    current page is ignored; only the contents of the file are sent to the client.
	 *
	 * @arguments.reset The reset and file attributes are mutually exclusive. If you specify a file, this attribute has no effect.
	 *                  - Yes: discards output that precedes call to content
	 *                  - No: preserves output that precedes call to content. In this case all output is sent with the specified type.
	 *
	 *
	 */
	public BodyResult _invoke( IBoxContext context, IStruct attributes, ComponentBody body, IStruct executionState ) {
		String					type			= attributes.getAsString( Key.type );
		Boolean					deleteFile		= attributes.getAsBoolean( Key.deleteFile );
		String					file			= attributes.getAsString( Key.file );
		Object					variable		= attributes.get( Key.variable );
		Boolean					reset			= attributes.getAsBoolean( Key.reset );

		WebRequestBoxContext	requestContext	= context.getParentOfType( WebRequestBoxContext.class );
		IBoxHTTPExchange		exchange		= requestContext.getHTTPExchange();

		if ( type != null ) {
			exchange.setResponseHeader( "content-type", type );
		}
		if ( file != null ) {
			file = FileSystemUtil.expandPath( context, file ).absolutePath().toString();
			context.clearBuffer();
			exchange.sendResponseFile( new File( file ) );
			if ( deleteFile ) {
				FileSystemUtil.deleteFile( file );
			}
			// I'm not sure CF actually aborts here if. If not, we need a flag in the context
			// to stop writing to the output buffer
			throw new AbortException();
		} else if ( variable != null ) {
			if ( reset ) {
				context.clearBuffer();
			} else {
				context.flushBuffer( false );
			}
			byte[] bytesToWrite;
			if ( variable instanceof byte[] barr ) {
				bytesToWrite = barr;
			} else {
				bytesToWrite = StringCaster.cast( variable ).getBytes();
			}
			exchange.sendResponseBinary( bytesToWrite );
			// I'm not sure CF actually aborts here if. If not, we need a flag in the context
			// to stop writing to the output buffer
			throw new AbortException();
		} else {
			if ( reset ) {
				context.clearBuffer();
			}
		}
		if ( body != null ) {
			BodyResult bodyResult = processBody( context, body );
			if ( bodyResult.isEarlyExit() ) {
				return bodyResult;
			}
		}
		return DEFAULT_RETURN;
	}
}
