/*
 * Decompiled with CFR 0.152.
 */
package com.auth0;

import com.auth0.SignatureVerifier;
import com.auth0.TokenValidationException;
import com.auth0.jwt.interfaces.DecodedJWT;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import org.apache.commons.lang3.Validate;

class IdTokenVerifier {
    private static final Integer DEFAULT_CLOCK_SKEW = 60;
    private static final String NONCE_CLAIM = "nonce";
    private static final String AZP_CLAIM = "azp";
    private static final String AUTH_TIME_CLAIM = "auth_time";

    IdTokenVerifier() {
    }

    void verify(String token, Options verifyOptions) throws TokenValidationException {
        Validate.notNull((Object)verifyOptions);
        if (this.isEmpty(token)) {
            throw new TokenValidationException("ID token is required but missing");
        }
        DecodedJWT decoded = verifyOptions.verifier.verifySignature(token);
        if (this.isEmpty(decoded.getIssuer())) {
            throw new TokenValidationException("Issuer (iss) claim must be a string present in the ID token");
        }
        if (!decoded.getIssuer().equals(verifyOptions.issuer)) {
            throw new TokenValidationException(String.format("Issuer (iss) claim mismatch in the ID token, expected \"%s\", found \"%s\"", verifyOptions.issuer, decoded.getIssuer()));
        }
        if (this.isEmpty(decoded.getSubject())) {
            throw new TokenValidationException("Subject (sub) claim must be a string present in the ID token");
        }
        List audience = decoded.getAudience();
        if (audience == null) {
            throw new TokenValidationException("Audience (aud) claim must be a string or array of strings present in the ID token");
        }
        if (!audience.contains(verifyOptions.audience)) {
            throw new TokenValidationException(String.format("Audience (aud) claim mismatch in the ID token; expected \"%s\" but found \"%s\"", verifyOptions.audience, decoded.getAudience()));
        }
        if (verifyOptions.organization != null) {
            String org = verifyOptions.organization.trim();
            if (org.startsWith("org_")) {
                String orgIdClaim = decoded.getClaim("org_id").asString();
                if (this.isEmpty(orgIdClaim)) {
                    throw new TokenValidationException("Organization Id (org_id) claim must be a string present in the ID token");
                }
                if (!org.equals(orgIdClaim)) {
                    throw new TokenValidationException(String.format("Organization (org_id) claim mismatch in the ID token; expected \"%s\" but found \"%s\"", verifyOptions.organization, orgIdClaim));
                }
            } else {
                String orgNameClaim = decoded.getClaim("org_name").asString();
                if (this.isEmpty(orgNameClaim)) {
                    throw new TokenValidationException("Organization name (org_name) claim must be a string present in the ID token");
                }
                if (!org.toLowerCase().equals(orgNameClaim)) {
                    throw new TokenValidationException(String.format("Organization (org_name) claim mismatch in the ID token; expected \"%s\" but found \"%s\"", verifyOptions.organization, orgNameClaim));
                }
            }
        }
        Calendar cal = Calendar.getInstance();
        Date now = verifyOptions.clock != null ? verifyOptions.clock : cal.getTime();
        int clockSkew = verifyOptions.clockSkew != null ? verifyOptions.clockSkew : DEFAULT_CLOCK_SKEW;
        if (decoded.getExpiresAt() == null) {
            throw new TokenValidationException("Expiration Time (exp) claim must be a number present in the ID token");
        }
        cal.setTime(decoded.getExpiresAt());
        cal.add(13, clockSkew);
        Date expDate = cal.getTime();
        if (now.after(expDate)) {
            throw new TokenValidationException(String.format("Expiration Time (exp) claim error in the ID token; current time (%d) is after expiration time (%d)", now.getTime() / 1000L, expDate.getTime() / 1000L));
        }
        if (decoded.getIssuedAt() == null) {
            throw new TokenValidationException("Issued At (iat) claim must be a number present in the ID token");
        }
        cal.setTime(decoded.getIssuedAt());
        cal.add(13, -1 * clockSkew);
        if (verifyOptions.nonce != null) {
            String nonceClaim = decoded.getClaim(NONCE_CLAIM).asString();
            if (this.isEmpty(nonceClaim)) {
                throw new TokenValidationException("Nonce (nonce) claim must be a string present in the ID token");
            }
            if (!verifyOptions.nonce.equals(nonceClaim)) {
                throw new TokenValidationException(String.format("Nonce (nonce) claim mismatch in the ID token; expected \"%s\", found \"%s\"", verifyOptions.nonce, nonceClaim));
            }
        }
        if (audience.size() > 1) {
            String azpClaim = decoded.getClaim(AZP_CLAIM).asString();
            if (this.isEmpty(azpClaim)) {
                throw new TokenValidationException("Authorized Party (azp) claim must be a string present in the ID token when Audience (aud) claim has multiple values");
            }
            if (!verifyOptions.audience.equals(azpClaim)) {
                throw new TokenValidationException(String.format("Authorized Party (azp) claim mismatch in the ID token; expected \"%s\", found \"%s\"", verifyOptions.audience, azpClaim));
            }
        }
        if (verifyOptions.maxAge != null) {
            Date authTime = decoded.getClaim(AUTH_TIME_CLAIM).asDate();
            if (authTime == null) {
                throw new TokenValidationException("Authentication Time (auth_time) claim must be a number present in the ID token when Max Age (max_age) is specified");
            }
            cal.setTime(authTime);
            cal.add(13, verifyOptions.maxAge);
            cal.add(13, clockSkew);
            Date authTimeDate = cal.getTime();
            if (now.after(authTimeDate)) {
                throw new TokenValidationException(String.format("Authentication Time (auth_time) claim in the ID token indicates that too much time has passed since the last end-user authentication. Current time (%d) is after last auth at (%d)", now.getTime() / 1000L, authTimeDate.getTime() / 1000L));
            }
        }
    }

    private boolean isEmpty(String value) {
        return value == null || value.isEmpty();
    }

    static class Options {
        final String issuer;
        final String audience;
        final SignatureVerifier verifier;
        String nonce;
        private Integer maxAge;
        Integer clockSkew;
        Date clock;
        String organization;

        public Options(String issuer, String audience, SignatureVerifier verifier) {
            Validate.notNull((Object)issuer);
            Validate.notNull((Object)audience);
            Validate.notNull((Object)verifier);
            this.issuer = issuer;
            this.audience = audience;
            this.verifier = verifier;
        }

        void setNonce(String nonce) {
            this.nonce = nonce;
        }

        void setMaxAge(Integer maxAge) {
            this.maxAge = maxAge;
        }

        void setClockSkew(Integer clockSkew) {
            this.clockSkew = clockSkew;
        }

        void setClock(Date now) {
            this.clock = now;
        }

        Integer getMaxAge() {
            return this.maxAge;
        }

        void setOrganization(String organization) {
            this.organization = organization;
        }
    }
}

