/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.backup.s3;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import io.camunda.zeebe.backup.api.Backup;
import io.camunda.zeebe.backup.api.BackupDescriptor;
import io.camunda.zeebe.backup.api.BackupIdentifier;
import io.camunda.zeebe.backup.api.BackupIdentifierWildcard;
import io.camunda.zeebe.backup.api.BackupStatus;
import io.camunda.zeebe.backup.api.BackupStatusCode;
import io.camunda.zeebe.backup.api.BackupStore;
import io.camunda.zeebe.backup.api.NamedFileSet;
import io.camunda.zeebe.backup.common.BackupIdentifierImpl;
import io.camunda.zeebe.backup.common.BackupImpl;
import io.camunda.zeebe.backup.common.NamedFileSetImpl;
import io.camunda.zeebe.backup.s3.AsyncAggregatingSubscriber;
import io.camunda.zeebe.backup.s3.S3BackupConfig;
import io.camunda.zeebe.backup.s3.S3BackupStoreException;
import io.camunda.zeebe.backup.s3.manifest.InProgressBackupManifest;
import io.camunda.zeebe.backup.s3.manifest.Manifest;
import io.camunda.zeebe.backup.s3.manifest.NoBackupManifest;
import io.camunda.zeebe.backup.s3.manifest.ValidBackupManifest;
import java.io.IOException;
import java.net.URI;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.auth.credentials.AwsBasicCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.StaticCredentialsProvider;
import software.amazon.awssdk.awscore.defaultsmode.DefaultsMode;
import software.amazon.awssdk.core.async.AsyncRequestBody;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.async.SdkPublisher;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.http.nio.netty.NettyNioAsyncHttpClient;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.s3.S3AsyncClient;
import software.amazon.awssdk.services.s3.S3AsyncClientBuilder;
import software.amazon.awssdk.services.s3.model.NoSuchKeyException;
import software.amazon.awssdk.services.s3.model.ObjectIdentifier;
import software.amazon.awssdk.services.s3.model.PutObjectResponse;
import software.amazon.awssdk.services.s3.model.S3Object;

public final class S3BackupStore
implements BackupStore {
    static final ObjectMapper MAPPER = new ObjectMapper().registerModule((Module)new Jdk8Module()).registerModule((Module)new JavaTimeModule());
    static final String SNAPSHOT_PREFIX = "snapshot/";
    static final String SEGMENTS_PREFIX = "segments/";
    static final String MANIFEST_OBJECT_KEY = "manifest.json";
    private static final Logger LOG = LoggerFactory.getLogger(S3BackupStore.class);
    private static final Pattern BACKUP_IDENTIFIER_PATTERN = Pattern.compile("^(?<partitionId>\\d+)/(?<checkpointId>\\d+)/(?<nodeId>\\d+).*");
    private final S3BackupConfig config;
    private final S3AsyncClient client;

    public S3BackupStore(S3BackupConfig config) {
        this(config, S3BackupStore.buildClient(config));
    }

    public S3BackupStore(S3BackupConfig config, S3AsyncClient client) {
        this.config = config;
        this.client = client;
    }

    private static Optional<BackupIdentifier> tryParseKeyAsId(String key) {
        Matcher matcher = BACKUP_IDENTIFIER_PATTERN.matcher(key);
        if (matcher.matches()) {
            try {
                int nodeId = Integer.parseInt(matcher.group("nodeId"));
                int partitionId = Integer.parseInt(matcher.group("partitionId"));
                long checkpointId = Long.parseLong(matcher.group("checkpointId"));
                return Optional.of(new BackupIdentifierImpl(nodeId, partitionId, checkpointId));
            }
            catch (NumberFormatException e) {
                LOG.warn("Tried interpreting key {} as a BackupIdentifier but failed", (Object)key, (Object)e);
            }
        }
        return Optional.empty();
    }

    private static String wildcardPrefix(BackupIdentifierWildcard wildcard) {
        return Stream.of(wildcard.partitionId(), wildcard.checkpointId(), wildcard.nodeId()).takeWhile(Optional::isPresent).map(Optional::get).map(rec$ -> rec$.toString()).collect(Collectors.joining("/"));
    }

    public static String objectPrefix(BackupIdentifier id) {
        return "%s/%s/%s/".formatted(id.partitionId(), id.checkpointId(), id.nodeId());
    }

    public static void validateConfig(S3BackupConfig config) {
        if (config.bucketName() == null || config.bucketName().isEmpty()) {
            throw new IllegalArgumentException("Configuration for S3 backup store is incomplete. bucketName must not be empty.");
        }
        if (config.region().isEmpty()) {
            LOG.warn("No region configured for S3 backup store. Region will be determined from environment (see https://docs.aws.amazon.com/sdk-for-java/latest/developer-guide/region-selection.html#automatically-determine-the-aws-region-from-the-environment)");
        }
        if (config.endpoint().isEmpty()) {
            LOG.warn("No endpoint configured for S3 backup store. Endpoint will be determined from the region");
        }
        if (config.credentials().isEmpty()) {
            LOG.warn("Access credentials (accessKey, secretKey) not configured for S3 backup store. Credentials will be determined from environment (see https://docs.aws.amazon.com/sdk-for-java/latest/developer-guide/credentials.html#credentials-chain)");
        }
        S3BackupStore.buildClient(config).close();
    }

    public CompletableFuture<Void> save(Backup backup) {
        LOG.info("Saving {}", (Object)backup.id());
        return ((CompletableFuture)this.updateManifestObject(backup.id(), Manifest::expectNoBackup, manifest -> manifest.asInProgress(backup)).thenComposeAsync(status -> {
            CompletableFuture<Void> snapshot = this.saveSnapshotFiles(backup);
            CompletableFuture<Void> segments = this.saveSegmentFiles(backup);
            return ((CompletableFuture)CompletableFuture.allOf(snapshot, segments).thenComposeAsync(ignored -> this.updateManifestObject(backup.id(), Manifest::expectInProgress, InProgressBackupManifest::asCompleted))).exceptionallyComposeAsync(throwable -> this.updateManifestObject(backup.id(), manifest -> manifest.asFailed((Throwable)throwable)).thenCompose(ignore -> CompletableFuture.failedStage(throwable)));
        })).thenApply(ignored -> null);
    }

    public CompletableFuture<BackupStatus> getStatus(BackupIdentifier id) {
        LOG.info("Querying status of {}", (Object)id);
        return this.readManifestObject(id).thenApply(Manifest::toStatus);
    }

    public CompletableFuture<Collection<BackupStatus>> list(BackupIdentifierWildcard wildcard) {
        LOG.info("Querying status of {}", (Object)wildcard);
        return this.readManifestObjects(wildcard).thenApplyAsync(manifests -> manifests.stream().map(Manifest::toStatus).toList());
    }

    public CompletableFuture<Void> delete(BackupIdentifier id) {
        LOG.info("Deleting {}", (Object)id);
        return ((CompletableFuture)((CompletableFuture)this.readManifestObject(id).thenApply(manifest -> {
            if (manifest.statusCode() == BackupStatusCode.IN_PROGRESS) {
                throw new S3BackupStoreException.BackupInInvalidStateException("Can't delete in-progress backup %s, must be marked as failed first".formatted(manifest.id()));
            }
            return manifest.id();
        })).thenComposeAsync(this::listBackupObjects)).thenComposeAsync(this::deleteBackupObjects);
    }

    public CompletableFuture<Backup> restore(BackupIdentifier id, Path targetFolder) {
        LOG.info("Restoring {} to {}", (Object)id, (Object)targetFolder);
        String backupPrefix = S3BackupStore.objectPrefix(id);
        return ((CompletableFuture)this.readManifestObject(id).thenApply(Manifest::expectCompleted)).thenComposeAsync(manifest -> this.downloadNamedFileSet(backupPrefix + SEGMENTS_PREFIX, manifest.segmentFileNames(), targetFolder).thenCombineAsync(this.downloadNamedFileSet(backupPrefix + SNAPSHOT_PREFIX, manifest.snapshotFileNames(), targetFolder), (segments, snapshot) -> new BackupImpl(id, (BackupDescriptor)manifest.descriptor(), snapshot, segments)));
    }

    public CompletableFuture<BackupStatusCode> markFailed(BackupIdentifier id, String failureReason) {
        LOG.info("Marking {} as failed", (Object)id);
        return this.updateManifestObject(id, manifest -> manifest.asFailed(failureReason)).thenApply(Manifest::statusCode);
    }

    public CompletableFuture<Void> closeAsync() {
        this.client.close();
        return CompletableFuture.completedFuture(null);
    }

    private CompletableFuture<NamedFileSet> downloadNamedFileSet(String sourcePrefix, Set<String> fileNames, Path targetFolder) {
        LOG.debug("Downloading {} files from prefix {} to {}", new Object[]{fileNames.size(), sourcePrefix, targetFolder});
        ConcurrentHashMap downloadedFiles = new ConcurrentHashMap();
        CompletableFuture[] futures = (CompletableFuture[])fileNames.stream().map(fileName -> {
            Path path = targetFolder.resolve((String)fileName);
            return this.client.getObject(req -> req.bucket(this.config.bucketName()).key(sourcePrefix + fileName), path).thenApply(response -> downloadedFiles.put(fileName, path));
        }).toArray(CompletableFuture[]::new);
        return CompletableFuture.allOf(futures).thenApply(ignored -> new NamedFileSetImpl((Map)downloadedFiles));
    }

    private CompletableFuture<List<ObjectIdentifier>> listBackupObjects(BackupIdentifier id) {
        LOG.debug("Listing objects of {}", (Object)id);
        return this.client.listObjectsV2(req -> req.bucket(this.config.bucketName()).prefix(S3BackupStore.objectPrefix(id))).thenApplyAsync(objects -> objects.contents().stream().map(S3Object::key).map(key -> (ObjectIdentifier)ObjectIdentifier.builder().key(key).build()).toList());
    }

    private CompletableFuture<Void> deleteBackupObjects(Collection<ObjectIdentifier> objectIdentifiers) {
        LOG.debug("Deleting {} objects", (Object)objectIdentifiers.size());
        if (objectIdentifiers.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        return this.client.deleteObjects(req -> req.bucket(this.config.bucketName()).delete(delete -> delete.objects(objectIdentifiers).quiet(Boolean.valueOf(true)))).thenApplyAsync(response -> {
            if (!response.errors().isEmpty()) {
                throw new S3BackupStoreException.BackupDeletionIncomplete("Not all objects belonging to the backup were deleted successfully: " + response.errors());
            }
            return null;
        });
    }

    private SdkPublisher<BackupIdentifier> findBackupIds(BackupIdentifierWildcard wildcard) {
        String prefix = S3BackupStore.wildcardPrefix(wildcard);
        LOG.debug("Using prefix {} to search for manifest files matching {}", (Object)prefix, (Object)wildcard);
        return this.client.listObjectsV2Paginator(cfg -> cfg.bucket(this.config.bucketName()).prefix(prefix)).contents().filter(obj -> obj.key().endsWith(MANIFEST_OBJECT_KEY)).map(S3Object::key).map(S3BackupStore::tryParseKeyAsId).filter(Optional::isPresent).map(Optional::get).filter(arg_0 -> ((BackupIdentifierWildcard)wildcard).matches(arg_0));
    }

    private CompletableFuture<Collection<Manifest>> readManifestObjects(BackupIdentifierWildcard wildcard) {
        AsyncAggregatingSubscriber aggregator = new AsyncAggregatingSubscriber(16L);
        SdkPublisher publisher = this.findBackupIds(wildcard).map(this::readManifestObject);
        publisher.subscribe(aggregator);
        return aggregator.result();
    }

    CompletableFuture<Manifest> readManifestObject(BackupIdentifier id) {
        LOG.debug("Reading manifest object of {}", (Object)id);
        return ((CompletableFuture)this.client.getObject(req -> req.bucket(this.config.bucketName()).key(S3BackupStore.objectPrefix(id) + MANIFEST_OBJECT_KEY), AsyncResponseTransformer.toBytes()).thenApply(response -> {
            try {
                return (Manifest)MAPPER.readValue(response.asInputStream(), ValidBackupManifest.class);
            }
            catch (JsonParseException e) {
                throw new S3BackupStoreException.ManifestParseException("Failed to parse manifest object", e);
            }
            catch (IOException e) {
                throw new S3BackupStoreException.BackupReadException("Failed to read manifest object", e);
            }
        })).exceptionally(throwable -> {
            if (throwable.getCause() instanceof NoSuchKeyException) {
                LOG.debug("Found no manifest for backup {}", (Object)id);
                return new NoBackupManifest(BackupIdentifierImpl.from((BackupIdentifier)id));
            }
            Throwable patt16383$temp = throwable.getCause();
            if (patt16383$temp instanceof S3BackupStoreException) {
                S3BackupStoreException e = (S3BackupStoreException)patt16383$temp;
                throw e;
            }
            throw new S3BackupStoreException.BackupReadException("Failed to read manifest of %s".formatted(id), (Throwable)throwable);
        });
    }

    <T> CompletableFuture<ValidBackupManifest> updateManifestObject(BackupIdentifier id, Function<Manifest, T> typeExpectation, Function<T, ValidBackupManifest> update) {
        return this.updateManifestObject(id, manifest -> (ValidBackupManifest)update.apply(typeExpectation.apply((Manifest)manifest)));
    }

    CompletableFuture<ValidBackupManifest> updateManifestObject(BackupIdentifier id, Function<Manifest, ValidBackupManifest> update) {
        return ((CompletableFuture)this.readManifestObject(id).thenApply(update)).thenComposeAsync(this::writeManifestObject);
    }

    CompletableFuture<ValidBackupManifest> writeManifestObject(ValidBackupManifest manifest) {
        AsyncRequestBody body;
        LOG.debug("Updating manifest of {} to {}", (Object)manifest.id(), (Object)manifest);
        try {
            body = AsyncRequestBody.fromBytes((byte[])MAPPER.writeValueAsBytes((Object)manifest));
        }
        catch (JsonProcessingException e) {
            return CompletableFuture.failedFuture(e);
        }
        return this.client.putObject(request -> request.bucket(this.config.bucketName()).key(S3BackupStore.objectPrefix(manifest.id()) + MANIFEST_OBJECT_KEY).build(), body).thenApply(resp -> manifest);
    }

    private CompletableFuture<Void> saveSnapshotFiles(Backup backup) {
        LOG.debug("Saving snapshot files for {}", (Object)backup.id());
        String prefix = S3BackupStore.objectPrefix(backup.id()) + SNAPSHOT_PREFIX;
        List<CompletableFuture> futures = backup.snapshot().namedFiles().entrySet().stream().map(snapshotFile -> this.saveNamedFile(prefix, (String)snapshotFile.getKey(), (Path)snapshotFile.getValue())).toList();
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
    }

    private CompletableFuture<Void> saveSegmentFiles(Backup backup) {
        LOG.debug("Saving segment files for {}", (Object)backup.id());
        String prefix = S3BackupStore.objectPrefix(backup.id()) + SEGMENTS_PREFIX;
        List<CompletableFuture> futures = backup.segments().namedFiles().entrySet().stream().map(segmentFile -> this.saveNamedFile(prefix, (String)segmentFile.getKey(), (Path)segmentFile.getValue())).toList();
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
    }

    private CompletableFuture<PutObjectResponse> saveNamedFile(String prefix, String fileName, Path filePath) {
        LOG.trace("Saving file {}({}) in prefix {}", new Object[]{fileName, filePath, prefix});
        return this.client.putObject(put -> put.bucket(this.config.bucketName()).key(prefix + fileName), AsyncRequestBody.fromFile((Path)filePath));
    }

    public static S3AsyncClient buildClient(S3BackupConfig config) {
        S3AsyncClientBuilder builder = S3AsyncClient.builder();
        builder.defaultsMode(DefaultsMode.AUTO);
        builder.httpClient(NettyNioAsyncHttpClient.builder().connectionAcquisitionTimeout(Duration.ofSeconds(45L)).build());
        builder.overrideConfiguration(cfg -> cfg.retryPolicy(RetryMode.ADAPTIVE));
        config.endpoint().ifPresent(endpoint -> builder.endpointOverride(URI.create(endpoint)));
        config.region().ifPresent(region -> builder.region(Region.of((String)region)));
        config.credentials().ifPresent(credentials -> builder.credentialsProvider((AwsCredentialsProvider)StaticCredentialsProvider.create((AwsCredentials)AwsBasicCredentials.create((String)credentials.accessKey(), (String)credentials.secretKey()))));
        config.apiCallTimeout().ifPresent(timeout -> builder.overrideConfiguration(cfg -> cfg.apiCallTimeout(timeout)));
        return (S3AsyncClient)builder.build();
    }
}

