/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.journal.file;

import io.camunda.zeebe.journal.JournalReader;
import io.camunda.zeebe.journal.JournalRecord;
import io.camunda.zeebe.journal.file.SegmentedJournal;
import java.io.File;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import org.agrona.DirectBuffer;
import org.agrona.concurrent.UnsafeBuffer;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

class JournalReaderTest {
    private static final int ENTRIES = 4;
    private final DirectBuffer data = new UnsafeBuffer("test".getBytes(StandardCharsets.UTF_8));
    private JournalReader reader;
    private SegmentedJournal journal;

    JournalReaderTest() {
    }

    @BeforeEach
    public void setup(@TempDir Path tempDir) {
        File directory = tempDir.resolve("data").toFile();
        this.journal = SegmentedJournal.builder().withDirectory(directory).withJournalIndexDensity(5).build();
        this.reader = this.journal.openReader();
    }

    @Test
    void shouldSeek() {
        for (int i = 1; i <= 4; ++i) {
            this.journal.append((long)i, this.data).index();
        }
        long nextIndex = this.reader.seek(2L);
        for (int i = 0; i < 3; ++i) {
            JournalRecord record = (JournalRecord)this.reader.next();
            Assertions.assertThat((long)record.asqn()).isEqualTo(nextIndex + (long)i);
            Assertions.assertThat((long)record.index()).isEqualTo(nextIndex + (long)i);
            Assertions.assertThat((Comparable)record.data()).isEqualTo((Object)this.data);
        }
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
    }

    @Test
    void shouldSeekToFirst() {
        for (int i = 1; i <= 4; ++i) {
            this.journal.append((long)i, this.data).index();
        }
        this.reader.next();
        this.reader.next();
        long nextIndex = this.reader.seekToFirst();
        Assertions.assertThat((long)nextIndex).isEqualTo(this.journal.getFirstIndex());
        JournalRecord record = (JournalRecord)this.reader.next();
        Assertions.assertThat((long)record.index()).isEqualTo(nextIndex);
        Assertions.assertThat((long)record.asqn()).isEqualTo(1L);
    }

    @Test
    void shouldSeekToLast() {
        for (int i = 1; i <= 4; ++i) {
            this.journal.append((long)i, this.data).index();
        }
        long nextIndex = this.reader.seekToLast();
        Assertions.assertThat((long)nextIndex).isEqualTo(this.journal.getLastIndex());
        JournalRecord record = (JournalRecord)this.reader.next();
        Assertions.assertThat((long)record.index()).isEqualTo(nextIndex);
        Assertions.assertThat((long)record.asqn()).isEqualTo(4L);
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
    }

    @Test
    void shouldNotReadIfSeekIsHigherThanLast() {
        for (int i = 1; i <= 4; ++i) {
            this.journal.append((long)i, this.data).index();
        }
        long nextIndex = this.reader.seek(99L);
        Assertions.assertThat((long)nextIndex).isEqualTo(this.journal.getLastIndex() + 1L);
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
    }

    @Test
    void shouldReadAppendedDataAfterSeek() {
        for (int i = 0; i < 4; ++i) {
            this.journal.append(this.data).index();
        }
        long nextIndex = this.reader.seek(99L);
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
        this.journal.append(this.data);
        Assertions.assertThat((long)nextIndex).isEqualTo(this.journal.getLastIndex());
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
    }

    @Test
    void shouldSeekToAsqn() {
        long startAsqn = 10L;
        for (int i = 0; i < 4; ++i) {
            Assertions.assertThat((Object)this.journal.append(10L + (long)i, this.data)).isNotNull();
        }
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        long nextIndex = this.reader.seekToAsqn(12L);
        Assertions.assertThat((long)nextIndex).isEqualTo(3L);
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        JournalRecord next = (JournalRecord)this.reader.next();
        Assertions.assertThat((long)next.index()).isEqualTo(nextIndex);
        Assertions.assertThat((long)next.asqn()).isEqualTo(12L);
    }

    @Test
    void shouldSeekToHighestAsqnLowerThanProvidedAsqn() {
        JournalRecord expectedRecord = this.journal.append(1L, this.data);
        this.journal.append(5L, this.data);
        long nextIndex = this.reader.seekToAsqn(4L);
        Assertions.assertThat((long)nextIndex).isEqualTo(expectedRecord.index());
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        Assertions.assertThat((Object)((JournalRecord)this.reader.next())).isEqualTo((Object)expectedRecord);
    }

    @Test
    void shouldSeekToHighestAsqnWithinBoundIndex() {
        long firstIndex = this.journal.append(1L, this.data).index();
        long secondIndex = this.journal.append(4L, this.data).index();
        long thirdIndex = this.journal.append(this.data).index();
        long fourthIndex = this.journal.append(5L, this.data).index();
        this.journal.append(this.data).index();
        Assertions.assertThat((long)this.reader.seekToAsqn(5L, firstIndex)).isEqualTo(firstIndex);
        Assertions.assertThat((long)((JournalRecord)this.reader.next()).asqn()).isEqualTo(1L);
        Assertions.assertThat((long)this.reader.seekToAsqn(5L, secondIndex)).isEqualTo(secondIndex);
        Assertions.assertThat((long)((JournalRecord)this.reader.next()).asqn()).isEqualTo(4L);
        Assertions.assertThat((long)this.reader.seekToAsqn(5L, thirdIndex)).isEqualTo(secondIndex);
        Assertions.assertThat((long)((JournalRecord)this.reader.next()).asqn()).isEqualTo(4L);
        Assertions.assertThat((long)this.reader.seekToAsqn(5L, fourthIndex)).isEqualTo(fourthIndex);
        Assertions.assertThat((long)((JournalRecord)this.reader.next()).asqn()).isEqualTo(5L);
        Assertions.assertThat((long)this.reader.seekToAsqn(Long.MAX_VALUE)).isEqualTo(fourthIndex);
        Assertions.assertThat((long)((JournalRecord)this.reader.next()).asqn()).isEqualTo(5L);
    }

    @Test
    void shouldSeekToLastAsqn() {
        JournalRecord expectedRecord = this.journal.append(5L, this.data);
        this.journal.append(this.data);
        Assertions.assertThat((long)this.reader.seekToAsqn(Long.MAX_VALUE)).isEqualTo(expectedRecord.index());
        Assertions.assertThat((Object)((JournalRecord)this.reader.next())).isEqualTo((Object)expectedRecord);
    }

    @Test
    void shouldSeekToHighestLowerAsqnSkippingRecordsWithNoAsqn() {
        JournalRecord expectedRecord = this.journal.append(1L, this.data);
        this.journal.append(this.data);
        this.journal.append(5L, this.data);
        long nextIndex = this.reader.seekToAsqn(3L);
        Assertions.assertThat((long)nextIndex).isEqualTo(expectedRecord.index());
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        Assertions.assertThat((Object)((JournalRecord)this.reader.next())).isEqualTo((Object)expectedRecord);
    }

    @Test
    void shouldSeekToFirstWhenAllAsqnIsHigher() {
        JournalRecord expectedRecord = this.journal.append(this.data);
        this.journal.append(5L, this.data);
        long nextIndex = this.reader.seekToAsqn(1L);
        Assertions.assertThat((long)nextIndex).isEqualTo(expectedRecord.index());
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        Assertions.assertThat((Object)((JournalRecord)this.reader.next())).isEqualTo((Object)expectedRecord);
        Assertions.assertThat((long)((JournalRecord)this.reader.next()).asqn()).isEqualTo(5L);
    }

    @Test
    void shouldSeekToFirstIfLowerThanFirst() {
        for (int i = 1; i <= 4; ++i) {
            this.journal.append((long)i, this.data).index();
        }
        long nextIndex = this.reader.seek(-1L);
        Assertions.assertThat((long)nextIndex).isEqualTo(this.journal.getFirstIndex());
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        JournalRecord record = (JournalRecord)this.reader.next();
        Assertions.assertThat((long)record.asqn()).isEqualTo(1L);
        Assertions.assertThat((long)record.index()).isEqualTo(nextIndex);
        Assertions.assertThat((Comparable)record.data()).isEqualTo((Object)this.data);
    }

    @Test
    void shouldSeekAfterTruncate() {
        long lastIndex = -1L;
        for (int i = 1; i <= 4; ++i) {
            lastIndex = this.journal.append((long)i, this.data).index();
        }
        this.journal.deleteAfter(lastIndex - 2L);
        long nextIndex = this.reader.seek(lastIndex - 2L);
        Assertions.assertThat((long)nextIndex).isEqualTo(lastIndex - 2L);
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        Assertions.assertThat((long)((JournalRecord)this.reader.next()).index()).isEqualTo(nextIndex);
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
    }

    @Test
    void shouldSeekAfterCompact() {
        this.journal.append(1L, this.data).index();
        this.journal.append(2L, this.data).index();
        this.journal.append(3L, this.data).index();
        this.journal.deleteUntil(3L);
        long nextIndex = this.reader.seek(3L);
        Assertions.assertThat((long)nextIndex).isEqualTo(3L);
        JournalRecord next = (JournalRecord)this.reader.next();
        Assertions.assertThat((long)next.index()).isEqualTo(nextIndex);
        Assertions.assertThat((long)next.asqn()).isEqualTo(3L);
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
    }

    @Test
    void shouldSeekToIndex() {
        long asqn = 1L;
        JournalRecord lastRecordWritten = null;
        for (int i = 1; i <= 4; ++i) {
            JournalRecord record = this.journal.append(asqn++, this.data);
            Assertions.assertThat((long)record.index()).isEqualTo((long)i);
            lastRecordWritten = record;
        }
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        long nextIndex = this.reader.seek(lastRecordWritten.index() - 1L);
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        Assertions.assertThat((long)((JournalRecord)this.reader.next()).index()).isEqualTo(nextIndex);
    }

    @Test
    void shouldSeekToFirstWithEmptyJournal() {
        long nextIndex = this.reader.seekToFirst();
        Assertions.assertThat((long)nextIndex).isEqualTo(this.journal.getFirstIndex());
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
    }

    @Test
    void shouldSeekToLastWithEmptyJournal() {
        long nextIndex = this.reader.seekToLast();
        Assertions.assertThat((long)nextIndex).isEqualTo(this.journal.getLastIndex());
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
    }

    @Test
    void shouldSeekToFirstWhenNoRecordsWithValidAsqnExists() {
        for (int i = 0; i < 4; ++i) {
            this.journal.append(this.data);
        }
        long nextIndex = this.reader.seekToAsqn(32L);
        Assertions.assertThat((long)nextIndex).isEqualTo(this.journal.getFirstIndex());
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        Assertions.assertThat((long)((JournalRecord)this.reader.next()).asqn()).isEqualTo(-1L);
    }
}

