/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.journal.file;

import io.camunda.zeebe.journal.Journal;
import io.camunda.zeebe.journal.JournalException;
import io.camunda.zeebe.journal.JournalReader;
import io.camunda.zeebe.journal.JournalRecord;
import io.camunda.zeebe.journal.file.LogCorrupter;
import io.camunda.zeebe.journal.file.SegmentedJournal;
import io.camunda.zeebe.journal.file.SegmentedJournalBuilder;
import io.camunda.zeebe.journal.file.TestJournalRecord;
import io.camunda.zeebe.journal.file.record.CorruptedLogException;
import io.camunda.zeebe.journal.file.record.PersistedJournalRecord;
import io.camunda.zeebe.journal.file.record.RecordData;
import io.camunda.zeebe.journal.file.record.RecordMetadata;
import java.io.File;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Objects;
import java.util.function.Consumer;
import org.agrona.DirectBuffer;
import org.agrona.concurrent.UnsafeBuffer;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

class JournalTest {
    @TempDir
    Path directory;
    private byte[] entry;
    private final DirectBuffer data = new UnsafeBuffer();
    private final DirectBuffer dataOther = new UnsafeBuffer();
    private Journal journal;

    JournalTest() {
    }

    @BeforeEach
    void setup() {
        this.entry = "TestData".getBytes();
        this.data.wrap(this.entry);
        byte[] entryOther = "TestData".getBytes();
        this.dataOther.wrap(entryOther);
        this.journal = this.openJournal();
    }

    @Test
    void shouldBeEmpty() {
        Assertions.assertThat((boolean)this.journal.isEmpty()).isTrue();
    }

    @Test
    void shouldNotBeEmpty() {
        this.journal.append(1L, this.data);
        Assertions.assertThat((boolean)this.journal.isEmpty()).isFalse();
    }

    @Test
    void shouldAppendData() {
        JournalRecord recordAppended = this.journal.append(1L, this.data);
        Assertions.assertThat((long)recordAppended.index()).isEqualTo(1L);
        Assertions.assertThat((long)recordAppended.asqn()).isEqualTo(1L);
    }

    @Test
    void shouldReadRecord() {
        JournalRecord recordAppended = this.journal.append(1L, this.data);
        JournalReader reader = this.journal.openReader();
        JournalRecord recordRead = (JournalRecord)reader.next();
        Assertions.assertThat((Object)recordRead).isEqualTo((Object)recordAppended);
    }

    @Test
    void shouldAppendMultipleData() {
        JournalRecord firstRecord = this.journal.append(10L, this.data);
        JournalRecord secondRecord = this.journal.append(20L, this.dataOther);
        Assertions.assertThat((long)firstRecord.index()).isEqualTo(1L);
        Assertions.assertThat((long)firstRecord.asqn()).isEqualTo(10L);
        Assertions.assertThat((long)secondRecord.index()).isEqualTo(2L);
        Assertions.assertThat((long)secondRecord.asqn()).isEqualTo(20L);
    }

    @Test
    void shouldReadMultipleRecord() {
        JournalRecord firstRecord = this.journal.append(1L, this.data);
        JournalRecord secondRecord = this.journal.append(20L, this.dataOther);
        JournalReader reader = this.journal.openReader();
        JournalRecord firstRecordRead = (JournalRecord)reader.next();
        JournalRecord secondRecordRead = (JournalRecord)reader.next();
        Assertions.assertThat((Object)firstRecordRead).isEqualTo((Object)firstRecord);
        Assertions.assertThat((Object)secondRecordRead).isEqualTo((Object)secondRecord);
    }

    @Test
    void shouldAppendAndReadMultipleRecordsInOrder() {
        for (int i = 0; i < 10; ++i) {
            JournalRecord recordAppended = this.journal.append((long)(i + 10), this.data);
            Assertions.assertThat((long)recordAppended.index()).isEqualTo((long)(i + 1));
        }
        JournalReader reader = this.journal.openReader();
        for (int i = 0; i < 10; ++i) {
            Assertions.assertThat((boolean)reader.hasNext()).isTrue();
            JournalRecord recordRead = (JournalRecord)reader.next();
            Assertions.assertThat((long)recordRead.index()).isEqualTo((long)(i + 1));
            byte[] data = new byte[recordRead.data().capacity()];
            recordRead.data().getBytes(0, data);
            Assertions.assertThat((long)recordRead.asqn()).isEqualTo((long)(i + 10));
            Assertions.assertThat((byte[])data).containsExactly(this.entry);
        }
    }

    @Test
    void shouldAppendAndReadMultipleRecords() {
        JournalReader reader = this.journal.openReader();
        for (int i = 0; i < 10; ++i) {
            this.entry = ("TestData" + i).getBytes();
            this.data.wrap(this.entry);
            JournalRecord recordAppended = this.journal.append((long)(i + 10), this.data);
            Assertions.assertThat((long)recordAppended.index()).isEqualTo((long)(i + 1));
            Assertions.assertThat((boolean)reader.hasNext()).isTrue();
            JournalRecord recordRead = (JournalRecord)reader.next();
            Assertions.assertThat((long)recordRead.index()).isEqualTo((long)(i + 1));
            byte[] data = new byte[recordRead.data().capacity()];
            recordRead.data().getBytes(0, data);
            Assertions.assertThat((long)recordRead.asqn()).isEqualTo((long)(i + 10));
            Assertions.assertThat((byte[])data).containsExactly(this.entry);
        }
    }

    @Test
    void shouldReset() {
        long asqn = 1L;
        Assertions.assertThat((long)this.journal.getLastIndex()).isEqualTo(0L);
        this.journal.append(asqn++, this.data);
        this.journal.append(asqn++, this.data);
        this.journal.reset(2L);
        Assertions.assertThat((boolean)this.journal.isEmpty()).isTrue();
        Assertions.assertThat((long)this.journal.getLastIndex()).isEqualTo(1L);
        JournalRecord record = this.journal.append(asqn, this.data);
        Assertions.assertThat((long)record.index()).isEqualTo(2L);
    }

    @Test
    void shouldNotReadAfterJournalResetWithoutReaderReset() {
        JournalReader reader = this.journal.openReader();
        long asqn = 1L;
        Assertions.assertThat((long)this.journal.getLastIndex()).isEqualTo(0L);
        this.journal.append(asqn++, this.data);
        this.journal.append(asqn++, this.data);
        JournalRecord record1 = (JournalRecord)reader.next();
        Assertions.assertThat((long)record1.index()).isEqualTo(1L);
        this.journal.reset(2L);
        this.journal.append(asqn++, this.data);
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
    }

    @Test
    void shouldWriteToTruncatedIndex() {
        JournalReader reader = this.journal.openReader();
        Assertions.assertThat((long)this.journal.getLastIndex()).isEqualTo(0L);
        this.journal.append(1L, this.data);
        this.journal.append(2L, this.data);
        this.journal.append(3L, this.data);
        JournalRecord record1 = (JournalRecord)reader.next();
        Assertions.assertThat((long)record1.index()).isEqualTo(1L);
        this.journal.deleteAfter(1L);
        Assertions.assertThat((long)this.journal.getLastIndex()).isEqualTo(1L);
        JournalRecord record = this.journal.append(4L, this.data);
        Assertions.assertThat((long)record.index()).isEqualTo(2L);
        Assertions.assertThat((long)record.asqn()).isEqualTo(4L);
        Assertions.assertThat((boolean)reader.hasNext()).isTrue();
        JournalRecord newRecord = (JournalRecord)reader.next();
        Assertions.assertThat((Object)newRecord).isEqualTo((Object)record);
    }

    @Test
    void shouldTruncate() {
        JournalReader reader = this.journal.openReader();
        Assertions.assertThat((long)this.journal.getLastIndex()).isEqualTo(0L);
        this.journal.append(1L, this.data);
        this.journal.append(2L, this.data);
        this.journal.append(3L, this.data);
        JournalRecord record1 = (JournalRecord)reader.next();
        Assertions.assertThat((long)record1.index()).isEqualTo(1L);
        this.journal.deleteAfter(1L);
        Assertions.assertThat((long)this.journal.getLastIndex()).isEqualTo(1L);
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
    }

    @Test
    void shouldNotReadTruncatedEntries() {
        JournalRecord record;
        int readerIndex;
        int writerIndex;
        int totalWrites = 10;
        int truncateIndex = 5;
        int asqn = 1;
        HashMap<Integer, JournalRecord> written = new HashMap<Integer, JournalRecord>();
        JournalReader reader = this.journal.openReader();
        for (writerIndex = 1; writerIndex <= 10; ++writerIndex) {
            JournalRecord record2 = this.journal.append((long)asqn++, this.data);
            Assertions.assertThat((long)record2.index()).isEqualTo((long)writerIndex);
            written.put(writerIndex, record2);
        }
        for (readerIndex = 1; readerIndex <= 5; ++readerIndex) {
            Assertions.assertThat((boolean)reader.hasNext()).isTrue();
            record = (JournalRecord)reader.next();
            Assertions.assertThat((Object)record).isEqualTo(written.get(readerIndex));
        }
        this.journal.deleteAfter(5L);
        for (writerIndex = 6; writerIndex <= 10; ++writerIndex) {
            record = this.journal.append((long)asqn++, this.data);
            Assertions.assertThat((long)record.index()).isEqualTo((long)writerIndex);
            written.put(writerIndex, record);
        }
        while (readerIndex <= 10) {
            Assertions.assertThat((boolean)reader.hasNext()).isTrue();
            record = (JournalRecord)reader.next();
            Assertions.assertThat((Object)record).isEqualTo(written.get(readerIndex));
            ++readerIndex;
        }
    }

    @Test
    void shouldNotReadTruncatedEntriesWhenReaderPastTruncateIndex() {
        JournalReader reader = this.journal.openReader();
        Assertions.assertThat((long)this.journal.getLastIndex()).isEqualTo(0L);
        this.journal.append(1L, this.data);
        this.journal.append(2L, this.data);
        this.journal.append(3L, this.data);
        reader.next();
        reader.next();
        Assertions.assertThat((boolean)reader.hasNext()).isTrue();
        this.journal.deleteAfter(1L);
        Assertions.assertThat((long)this.journal.getLastIndex()).isEqualTo(1L);
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
    }

    @Test
    void shouldNotReadTruncatedEntriesWhenReaderAtTruncateIndex() {
        JournalReader reader = this.journal.openReader();
        Assertions.assertThat((long)this.journal.getLastIndex()).isEqualTo(0L);
        this.journal.append(1L, this.data);
        this.journal.append(2L, this.data);
        this.journal.append(3L, this.data);
        reader.next();
        reader.next();
        Assertions.assertThat((boolean)reader.hasNext()).isTrue();
        this.journal.deleteAfter(2L);
        Assertions.assertThat((long)this.journal.getLastIndex()).isEqualTo(2L);
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
    }

    @Test
    void shouldNotReadTruncatedEntriesWhenReaderBeforeTruncateIndex() {
        JournalReader reader = this.journal.openReader();
        Assertions.assertThat((long)this.journal.getLastIndex()).isEqualTo(0L);
        this.journal.append(1L, this.data);
        this.journal.append(2L, this.data);
        this.journal.append(3L, this.data);
        reader.next();
        Assertions.assertThat((boolean)reader.hasNext()).isTrue();
        this.journal.deleteAfter(2L);
        Assertions.assertThat((long)this.journal.getLastIndex()).isEqualTo(2L);
        reader.next();
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
    }

    @Test
    void shouldAppendJournalRecord() {
        SegmentedJournal receiverJournal = SegmentedJournal.builder().withDirectory(this.directory.resolve("data-2").toFile()).withJournalIndexDensity(5).build();
        JournalRecord expected = this.journal.append(10L, this.data);
        receiverJournal.append(expected);
        JournalReader reader = receiverJournal.openReader();
        Assertions.assertThat((boolean)reader.hasNext()).isTrue();
        JournalRecord actual = (JournalRecord)reader.next();
        Assertions.assertThat((Object)expected).isEqualTo((Object)actual);
    }

    @Test
    void shouldNotAppendRecordWithAlreadyAppendedIndex() {
        JournalRecord record = this.journal.append(1L, this.data);
        this.journal.append(this.data);
        Assertions.assertThatThrownBy(() -> this.journal.append(record)).isInstanceOf(JournalException.InvalidIndex.class);
    }

    @Test
    void shouldNotAppendRecordWithGapInIndex() {
        SegmentedJournal receiverJournal = SegmentedJournal.builder().withDirectory(this.directory.resolve("data-2").toFile()).withJournalIndexDensity(5).build();
        this.journal.append(1L, this.data);
        JournalRecord record = this.journal.append(1L, this.data);
        Assertions.assertThatThrownBy(() -> receiverJournal.append(record)).isInstanceOf(JournalException.InvalidIndex.class);
    }

    @Test
    void shouldNotAppendLastRecord() {
        JournalRecord record = this.journal.append(1L, this.data);
        Assertions.assertThatThrownBy(() -> this.journal.append(record)).isInstanceOf(JournalException.InvalidIndex.class);
    }

    @Test
    void shouldNotAppendRecordWithInvalidChecksum() {
        SegmentedJournal receiverJournal = SegmentedJournal.builder().withDirectory(this.directory.resolve("data-2").toFile()).withJournalIndexDensity(5).build();
        JournalRecord record = this.journal.append(1L, this.data);
        TestJournalRecord invalidChecksumRecord = new TestJournalRecord(record.index(), record.asqn(), -1L, record.data());
        Assertions.assertThatThrownBy(() -> receiverJournal.append((JournalRecord)invalidChecksumRecord)).isInstanceOf(JournalException.InvalidChecksum.class);
    }

    @Test
    void shouldReturnFirstIndex() {
        long firstIndex = this.journal.append(this.data).index();
        this.journal.append(this.data);
        Assertions.assertThat((long)this.journal.getFirstIndex()).isEqualTo(firstIndex);
    }

    @Test
    void shouldReturnLastIndex() {
        this.journal.append(this.data);
        long lastIndex = this.journal.append(this.data).index();
        Assertions.assertThat((long)this.journal.getLastIndex()).isEqualTo(lastIndex);
    }

    @Test
    void shouldOpenAndClose() throws Exception {
        Assertions.assertThat((boolean)this.journal.isOpen()).isTrue();
        this.journal.close();
        Assertions.assertThat((boolean)this.journal.isOpen()).isFalse();
    }

    @Test
    void shouldReopenJournalWithExistingRecords() throws Exception {
        this.journal.append(this.data);
        this.journal.append(this.data);
        long lastIndexBeforeClose = this.journal.getLastIndex();
        Assertions.assertThat((long)lastIndexBeforeClose).isEqualTo(2L);
        this.journal.close();
        this.journal = this.openJournal();
        Assertions.assertThat((boolean)this.journal.isOpen()).isTrue();
        Assertions.assertThat((long)this.journal.getLastIndex()).isEqualTo(lastIndexBeforeClose);
    }

    @Test
    void shouldReadReopenedJournal() throws Exception {
        PersistedJournalRecord appendedRecord = JournalTest.copyRecord(this.journal.append(this.data));
        this.journal.close();
        this.journal = this.openJournal();
        JournalReader reader = this.journal.openReader();
        Assertions.assertThat((boolean)this.journal.isOpen()).isTrue();
        Assertions.assertThat((boolean)reader.hasNext()).isTrue();
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)appendedRecord);
    }

    @Test
    void shouldWriteToReopenedJournalAtNextIndex() throws Exception {
        PersistedJournalRecord firstRecord = JournalTest.copyRecord(this.journal.append(this.data));
        this.journal.close();
        this.journal = this.openJournal();
        JournalRecord secondRecord = this.journal.append(this.data);
        Assertions.assertThat((long)secondRecord.index()).isEqualTo(2L);
        JournalReader reader = this.journal.openReader();
        Assertions.assertThat((boolean)reader.hasNext()).isTrue();
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)firstRecord);
        Assertions.assertThat((boolean)reader.hasNext()).isTrue();
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)secondRecord);
    }

    @Test
    void shouldNotReadDeletedEntries() {
        JournalRecord firstRecord = this.journal.append(this.data);
        this.journal.append(this.data);
        this.journal.append(this.data);
        this.journal.deleteAfter(firstRecord.index());
        JournalRecord newSecondRecord = this.journal.append(this.data);
        JournalReader reader = this.journal.openReader();
        Assertions.assertThat((long)newSecondRecord.index()).isEqualTo(2L);
        Assertions.assertThat((boolean)reader.hasNext()).isTrue();
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)firstRecord);
        Assertions.assertThat((boolean)reader.hasNext()).isTrue();
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)newSecondRecord);
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
    }

    @Test
    void shouldInvalidateAllEntries() throws Exception {
        this.data.wrap("000".getBytes(StandardCharsets.UTF_8));
        PersistedJournalRecord firstRecord = JournalTest.copyRecord(this.journal.append(this.data));
        this.journal.append(this.data);
        this.journal.append(this.data);
        this.journal.deleteAfter(firstRecord.index());
        this.data.wrap("111".getBytes(StandardCharsets.UTF_8));
        PersistedJournalRecord secondRecord = JournalTest.copyRecord(this.journal.append(this.data));
        this.journal.close();
        this.journal = this.openJournal();
        JournalReader reader = this.journal.openReader();
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)firstRecord);
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)secondRecord);
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
    }

    @Test
    void shouldDetectCorruptedEntry() throws Exception {
        this.data.wrap("000".getBytes(StandardCharsets.UTF_8));
        this.journal.append(this.data);
        PersistedJournalRecord secondRecord = JournalTest.copyRecord(this.journal.append(this.data));
        File dataFile = Objects.requireNonNull(this.directory.toFile().listFiles())[0];
        File log = Objects.requireNonNull(dataFile.listFiles())[0];
        this.journal.close();
        Assertions.assertThat((boolean)LogCorrupter.corruptRecord(log, secondRecord.index())).isTrue();
        Assertions.assertThatThrownBy(() -> {
            this.journal = this.openJournal(b -> b.withLastWrittenIndex(secondRecord.index()));
        }).isInstanceOf(CorruptedLogException.class);
    }

    @Test
    void shouldDeletePartiallyWrittenEntry() throws Exception {
        this.data.wrap("000".getBytes(StandardCharsets.UTF_8));
        PersistedJournalRecord firstRecord = JournalTest.copyRecord(this.journal.append(this.data));
        PersistedJournalRecord secondRecord = JournalTest.copyRecord(this.journal.append(this.data));
        File dataFile = Objects.requireNonNull(this.directory.toFile().listFiles())[0];
        File log = Objects.requireNonNull(dataFile.listFiles())[0];
        this.journal.close();
        Assertions.assertThat((boolean)LogCorrupter.corruptRecord(log, secondRecord.index())).isTrue();
        this.journal = this.openJournal(b -> b.withLastWrittenIndex(firstRecord.index()));
        this.data.wrap("111".getBytes(StandardCharsets.UTF_8));
        JournalRecord lastRecord = this.journal.append(this.data);
        JournalReader reader = this.journal.openReader();
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)firstRecord);
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)lastRecord);
    }

    static PersistedJournalRecord copyRecord(JournalRecord record) {
        DirectBuffer data = record.data();
        byte[] buffer = new byte[data.capacity()];
        data.getBytes(0, buffer);
        UnsafeBuffer copiedData = new UnsafeBuffer(buffer);
        RecordData copiedRecord = new RecordData(record.index(), record.asqn(), (DirectBuffer)copiedData);
        return new PersistedJournalRecord(new RecordMetadata(record.checksum(), copiedRecord.data().capacity()), copiedRecord);
    }

    private SegmentedJournal openJournal() {
        return this.openJournal(b -> {});
    }

    private SegmentedJournal openJournal(Consumer<SegmentedJournalBuilder> option) {
        SegmentedJournalBuilder builder = SegmentedJournal.builder().withDirectory(this.directory.resolve("data").toFile()).withJournalIndexDensity(5);
        option.accept(builder);
        return builder.build();
    }
}

