/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.journal.file;

import io.camunda.zeebe.journal.JournalReader;
import io.camunda.zeebe.journal.JournalRecord;
import io.camunda.zeebe.journal.file.FrameUtil;
import io.camunda.zeebe.journal.file.IndexInfo;
import io.camunda.zeebe.journal.file.JournalIndex;
import io.camunda.zeebe.journal.file.JournalSegment;
import io.camunda.zeebe.journal.file.JournalSegmentDescriptor;
import io.camunda.zeebe.journal.file.JournalSegmentFile;
import io.camunda.zeebe.journal.file.JournalTest;
import io.camunda.zeebe.journal.file.LogCorrupter;
import io.camunda.zeebe.journal.file.SegmentedJournal;
import io.camunda.zeebe.journal.file.record.CorruptedLogException;
import io.camunda.zeebe.journal.file.record.PersistedJournalRecord;
import io.camunda.zeebe.journal.file.record.RecordData;
import io.camunda.zeebe.journal.file.record.SBESerializer;
import java.io.File;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.Objects;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.agrona.DirectBuffer;
import org.agrona.MutableDirectBuffer;
import org.agrona.concurrent.UnsafeBuffer;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.AssertionsForClassTypes;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

class SegmentedJournalTest {
    private static final String JOURNAL_NAME = "journal";
    @TempDir
    Path directory;
    private final int journalIndexDensity = 1;
    private final DirectBuffer data = new UnsafeBuffer("test".getBytes(StandardCharsets.UTF_8));
    private final int entrySize = this.getSerializedSize(this.data);

    SegmentedJournalTest() {
    }

    @Test
    void shouldDeleteIndexMappingsOnReset() {
        SegmentedJournal journal = this.openJournal(10.0f);
        long asqn = 1L;
        for (int i = 0; i < 2; ++i) {
            journal.append(asqn++, this.data);
        }
        Assertions.assertThat((Object)journal.getJournalIndex().lookup(1L)).isNotNull();
        Assertions.assertThat((Object)journal.getJournalIndex().lookup(2L)).isNotNull();
        journal.reset(journal.getLastIndex());
        Assertions.assertThat((Object)journal.getJournalIndex().lookup(1L)).isNull();
        Assertions.assertThat((Object)journal.getJournalIndex().lookup(2L)).isNull();
    }

    @Test
    void shouldUpdateIndexMappingsOnCompact() {
        int entriesPerSegment = 10;
        long asqn = 1L;
        SegmentedJournal journal = this.openJournal(10.0f);
        for (int i = 0; i < 30; ++i) {
            journal.append(asqn++, this.data);
        }
        Assertions.assertThat((Object)journal.getJournalIndex().lookup(10L)).isNotNull();
        journal.deleteUntil(11L);
        IndexInfo lookup = journal.getJournalIndex().lookup(9L);
        Assertions.assertThat((Object)lookup).isNull();
        Assertions.assertThat((Object)journal.getJournalIndex().lookup(30L)).isNotNull();
    }

    @Test
    void shouldUpdateIndexMappingsOnTruncate() {
        int entriesPerSegment = 10;
        long asqn = 1L;
        SegmentedJournal journal = this.openJournal(10.0f);
        for (int i = 0; i < 2; ++i) {
            journal.append(asqn++, this.data);
        }
        Assertions.assertThat((Object)journal.getJournalIndex().lookup(1L)).isNotNull();
        Assertions.assertThat((long)journal.getJournalIndex().lookup(2L).index()).isEqualTo(2L);
        journal.deleteAfter(1L);
        Assertions.assertThat((Object)journal.getJournalIndex().lookup(1L)).isNotNull();
        Assertions.assertThat((long)journal.getJournalIndex().lookup(2L).index()).isEqualTo(1L);
    }

    @Test
    void shouldCreateNewSegmentIfEntryExceedsBuffer() {
        int i;
        boolean asqn = true;
        SegmentedJournal journal = this.openJournal(1.5f);
        JournalReader reader = journal.openReader();
        for (i = 0; i < 2; ++i) {
            journal.append((long)(1 + i), this.data);
        }
        Assertions.assertThat((Object)journal.getFirstSegment()).isNotEqualTo((Object)journal.getLastSegment());
        for (i = 0; i < 2; ++i) {
            Assertions.assertThat((boolean)reader.hasNext()).isTrue();
            JournalRecord entry = (JournalRecord)reader.next();
            Assertions.assertThat((long)entry.asqn()).isEqualTo((long)(1 + i));
            Assertions.assertThat((Comparable)entry.data()).isEqualTo((Object)this.data);
        }
    }

    @Test
    void shouldNotTruncateIfIndexIsHigherThanLast() {
        int i;
        boolean asqn = true;
        SegmentedJournal journal = this.openJournal(1.0f);
        JournalReader reader = journal.openReader();
        long lastIndex = -1L;
        for (i = 0; i < 2; ++i) {
            lastIndex = journal.append((long)(1 + i), this.data).index();
        }
        journal.deleteAfter(lastIndex);
        for (i = 0; i < 2; ++i) {
            Assertions.assertThat((boolean)reader.hasNext()).isTrue();
            JournalRecord entry = (JournalRecord)reader.next();
            Assertions.assertThat((long)entry.asqn()).isEqualTo((long)(1 + i));
            Assertions.assertThat((Comparable)entry.data()).isEqualTo((Object)this.data);
        }
    }

    @Test
    void shouldNotCompactIfIndexIsLowerThanFirst() {
        boolean asqn = true;
        SegmentedJournal journal = this.openJournal(1.5f);
        JournalReader reader = journal.openReader();
        JournalRecord firstRecord = journal.append(1L, this.data);
        JournalRecord secondRecord = journal.append(2L, this.data);
        journal.deleteUntil(firstRecord.index());
        Assertions.assertThat((boolean)reader.hasNext()).isTrue();
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)firstRecord);
        Assertions.assertThat((boolean)reader.hasNext()).isTrue();
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)secondRecord);
    }

    @Test
    void shouldTruncateNextEntry() {
        SegmentedJournal journal = this.openJournal(2.0f);
        JournalReader reader = journal.openReader();
        JournalRecord firstRecord = journal.append(1L, this.data);
        journal.append(2L, this.data).index();
        journal.append(3L, this.data).index();
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)firstRecord);
        journal.deleteAfter(firstRecord.index());
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
    }

    @Test
    void shouldTruncateReadEntry() {
        SegmentedJournal journal = this.openJournal(2.0f);
        JournalReader reader = journal.openReader();
        long first = journal.append(1L, this.data).index();
        journal.append(2L, this.data).index();
        Assertions.assertThat((boolean)reader.hasNext()).isTrue();
        journal.deleteAfter(first - 1L);
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
        Assertions.assertThat((long)journal.getLastIndex()).isEqualTo(0L);
    }

    @Test
    void shouldTruncateNextSegment() {
        SegmentedJournal journal = this.openJournal(1.0f);
        JournalReader reader = journal.openReader();
        JournalRecord firstRecord = journal.append(1L, this.data);
        journal.append(2L, this.data);
        journal.deleteAfter(firstRecord.index());
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)firstRecord);
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
        Assertions.assertThat((long)journal.getLastIndex()).isEqualTo(firstRecord.index());
    }

    @Test
    void shouldReadSegmentStartAfterMidSegmentTruncate() {
        int entryPerSegment = 2;
        SegmentedJournal journal = this.openJournal(2.0f);
        JournalReader reader = journal.openReader();
        long lastIndex = -1L;
        for (int i = 0; i < 4; ++i) {
            lastIndex = journal.append((long)(i + 1), this.data).index();
        }
        journal.deleteAfter(lastIndex - 1L);
        Assertions.assertThat((long)reader.seek(lastIndex - 1L)).isEqualTo(lastIndex - 1L);
        Assertions.assertThat((long)((JournalRecord)reader.next()).index()).isEqualTo(lastIndex - 1L);
        Assertions.assertThat((long)journal.getLastIndex()).isEqualTo(lastIndex - 1L);
    }

    @Test
    void shouldCompactUpToStartOfSegment() {
        int entryPerSegment = 2;
        SegmentedJournal journal = this.openJournal(2.0f);
        JournalReader reader = journal.openReader();
        long lastIndex = -1L;
        for (int i = 0; i < 4; ++i) {
            lastIndex = journal.append((long)(i + 1), this.data).index();
        }
        Assertions.assertThat((boolean)reader.hasNext()).isTrue();
        journal.deleteUntil(lastIndex);
        Assertions.assertThat((long)journal.getFirstIndex()).isEqualTo(lastIndex - 1L);
        reader.seekToFirst();
        Assertions.assertThat((long)((JournalRecord)reader.next()).index()).isEqualTo(lastIndex - 1L);
    }

    @Test
    void shouldNotCompactTheLastSegmentWhenNonExistingHigherIndex() {
        int entryPerSegment = 2;
        SegmentedJournal journal = this.openJournal(2.0f);
        JournalReader reader = journal.openReader();
        long lastIndex = -1L;
        for (int i = 0; i < 4; ++i) {
            lastIndex = journal.append((long)(i + 1), this.data).index();
        }
        Assertions.assertThat((boolean)reader.hasNext()).isTrue();
        journal.deleteUntil(lastIndex + 1L);
        Assertions.assertThat((long)journal.getFirstIndex()).isEqualTo(lastIndex - 1L);
        reader.seekToFirst();
        Assertions.assertThat((long)((JournalRecord)reader.next()).index()).isEqualTo(lastIndex - 1L);
    }

    @Test
    void shouldReturnCorrectFirstIndexAfterCompaction() {
        int entryPerSegment = 2;
        SegmentedJournal journal = this.openJournal(2.0f);
        long lastIndex = -1L;
        for (int i = 0; i < 4; ++i) {
            lastIndex = journal.append((long)(i + 1), this.data).index();
        }
        journal.deleteUntil(lastIndex);
        Assertions.assertThat((long)journal.getFirstIndex()).isEqualTo(lastIndex - 1L);
    }

    @Test
    void shouldWriteAndReadAfterTruncate() {
        SegmentedJournal journal = this.openJournal(2.0f);
        JournalReader reader = journal.openReader();
        long first = journal.append(1L, this.data).index();
        journal.append(2L, this.data);
        journal.deleteAfter(first - 1L);
        this.data.wrap("new".getBytes());
        JournalRecord lastRecord = journal.append(3L, this.data);
        Assertions.assertThat((long)first).isEqualTo(lastRecord.index());
        Assertions.assertThat((boolean)reader.hasNext()).isTrue();
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)lastRecord);
    }

    @Test
    void shouldAppendEntriesOfDifferentSizesOverSegmentSize() {
        this.data.wrap("1234567890".getBytes(StandardCharsets.UTF_8));
        int entrySize = this.getSerializedSize(this.data);
        SegmentedJournal journal = this.openJournal(1.0f, entrySize);
        JournalReader reader = journal.openReader();
        JournalRecord firstRecord = journal.append((DirectBuffer)new UnsafeBuffer("12345".getBytes()));
        JournalRecord secondRecord = journal.append((DirectBuffer)new UnsafeBuffer("1234567".getBytes()));
        JournalRecord thirdRecord = journal.append((DirectBuffer)new UnsafeBuffer("1234567890".getBytes()));
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)firstRecord);
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)secondRecord);
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)thirdRecord);
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
    }

    @Test
    void shouldUpdateIndexMappingsAfterRestart() {
        int entriesPerSegment = 10;
        long asqn = 1L;
        SegmentedJournal journal = this.openJournal(10.0f);
        for (int i = 0; i < 2; ++i) {
            journal.append(asqn++, this.data);
        }
        JournalIndex indexBeforeClose = journal.getJournalIndex();
        journal.close();
        journal = this.openJournal(10.0f);
        boolean firstIndexedPosition = true;
        int secondIndexedPosition = 2;
        JournalIndex indexAfterRestart = journal.getJournalIndex();
        Assertions.assertThat((long)indexAfterRestart.lookup(1L).index()).isEqualTo(1L);
        Assertions.assertThat((long)indexAfterRestart.lookup(2L).index()).isEqualTo(2L);
        Assertions.assertThat((int)indexAfterRestart.lookup(1L).position()).isEqualTo(indexBeforeClose.lookup(1L).position());
        Assertions.assertThat((int)indexAfterRestart.lookup(2L).position()).isEqualTo(indexBeforeClose.lookup(2L).position());
    }

    @Test
    void shouldHandlePartiallyWrittenDescriptor() throws Exception {
        File dataFile = this.directory.resolve("data").toFile();
        Assertions.assertThat((boolean)dataFile.mkdirs()).isTrue();
        File emptyLog = new File(dataFile, "journal-1.log");
        Assertions.assertThat((boolean)emptyLog.createNewFile()).isTrue();
        SegmentedJournal journal = this.openJournal(10.0f);
        JournalReader reader = journal.openReader();
        JournalRecord record = journal.append(this.data);
        Assertions.assertThat((long)journal.getFirstIndex()).isEqualTo(record.index());
        Assertions.assertThat((long)journal.getLastIndex()).isEqualTo(record.index());
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)record);
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
    }

    @Test
    void shouldHandleCorruptionAtDescriptorWithoutAckedEntries() throws Exception {
        SegmentedJournal journal = this.openJournal(1.0f);
        journal.close();
        File dataFile = this.directory.resolve("data").toFile();
        File logFile = Objects.requireNonNull(dataFile.listFiles(f -> f.getName().endsWith(".log")))[0];
        LogCorrupter.corruptDescriptor(logFile);
        journal = this.openJournal(1.0f);
        JournalReader reader = journal.openReader();
        JournalRecord record = journal.append(this.data);
        Assertions.assertThat((long)journal.getFirstIndex()).isEqualTo(record.index());
        Assertions.assertThat((long)journal.getLastIndex()).isEqualTo(record.index());
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)record);
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
    }

    @Test
    void shouldHandleCorruptionAtDescriptorWithSomeAckedEntries() throws Exception {
        SegmentedJournal journal = this.openJournal(1.0f);
        PersistedJournalRecord firstRecord = JournalTest.copyRecord(journal.append(this.data));
        journal.append(this.data);
        journal.close();
        File dataFile = this.directory.resolve("data").toFile();
        File logFile = Objects.requireNonNull(dataFile.listFiles(f -> f.getName().endsWith("2.log")))[0];
        LogCorrupter.corruptDescriptor(logFile);
        journal = this.openJournal(1.0f);
        JournalReader reader = journal.openReader();
        JournalRecord lastRecord = journal.append(this.data);
        Assertions.assertThat((long)journal.getFirstIndex()).isEqualTo(firstRecord.index());
        Assertions.assertThat((long)journal.getLastIndex()).isEqualTo(lastRecord.index());
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)firstRecord);
        Assertions.assertThat((Object)((JournalRecord)reader.next())).isEqualTo((Object)lastRecord);
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
    }

    @Test
    void shouldDetectCorruptionAtDescriptorWithAckedEntries() throws Exception {
        SegmentedJournal journal = this.openJournal(1.0f);
        long index = journal.append(this.data).index();
        journal.close();
        File dataFile = this.directory.resolve("data").toFile();
        File logFile = Objects.requireNonNull(dataFile.listFiles(f -> f.getName().endsWith(".log")))[0];
        LogCorrupter.corruptDescriptor(logFile);
        AssertionsForClassTypes.assertThatThrownBy(() -> SegmentedJournal.builder().withDirectory(this.directory.resolve("data").toFile()).withMaxSegmentSize(this.entrySize + JournalSegmentDescriptor.getEncodingLength()).withJournalIndexDensity(1).withLastWrittenIndex(index).build()).isInstanceOf(CorruptedLogException.class);
    }

    @Test
    void shouldNotDeleteSegmentFileImmediately() {
        SegmentedJournal journal = this.openJournal(2.0f);
        journal.append(this.data);
        journal.append(this.data);
        JournalReader reader = journal.openReader();
        reader.next();
        journal.reset(100L);
        File logDirectory = this.directory.resolve("data").toFile();
        Assertions.assertThat((File)logDirectory).isDirectoryContaining(file -> JournalSegmentFile.isDeletedSegmentFile((String)JOURNAL_NAME, (String)file.getName())).isDirectoryContaining(file -> JournalSegmentFile.isSegmentFile((String)JOURNAL_NAME, (String)file.getName()));
    }

    @Test
    void shouldNotFailOnResetAndOpeningReaderConcurrently() throws InterruptedException {
        CountDownLatch latch = new CountDownLatch(2);
        SegmentedJournal journal = this.openJournal(2.0f);
        journal.append(this.data);
        journal.append(this.data);
        new Thread(() -> {
            journal.reset(100L);
            latch.countDown();
        }).start();
        new Thread(() -> {
            journal.openReader();
            latch.countDown();
        }).start();
        Assertions.assertThat((boolean)latch.await(1L, TimeUnit.SECONDS)).isTrue();
    }

    @Test
    void shouldNotFailOnDeleteAndOpeningReaderConcurrently() throws InterruptedException {
        CountDownLatch latch = new CountDownLatch(2);
        SegmentedJournal journal = this.openJournal(2.0f);
        for (int i = 0; i < 10; ++i) {
            journal.append(this.data);
        }
        long indexToCompact = journal.append(this.data).index();
        new Thread(() -> {
            journal.deleteUntil(indexToCompact);
            latch.countDown();
        }).start();
        new Thread(() -> {
            journal.openReader();
            latch.countDown();
        }).start();
        Assertions.assertThat((boolean)latch.await(1L, TimeUnit.SECONDS)).isTrue();
    }

    @Test
    void shouldDeleteSegmentFileWhenReaderIsClosed() {
        SegmentedJournal journal = this.openJournal(2.0f);
        journal.append(this.data);
        JournalReader reader = journal.openReader();
        journal.reset(100L);
        reader.close();
        File logDirectory = this.directory.resolve("data").toFile();
        Assertions.assertThat((File)logDirectory).isDirectoryNotContaining(file -> JournalSegmentFile.isDeletedSegmentFile((String)JOURNAL_NAME, (String)file.getName())).isDirectoryContaining(file -> JournalSegmentFile.isSegmentFile((String)JOURNAL_NAME, (String)file.getName()));
    }

    @Test
    void shouldDeleteSegmentFileImmediatelyWhenThereAreNoReaders() {
        SegmentedJournal journal = this.openJournal(2.0f);
        journal.append(this.data);
        journal.reset(100L);
        File logDirectory = this.directory.resolve("data").toFile();
        Assertions.assertThat((File)logDirectory).isDirectoryNotContaining(file -> JournalSegmentFile.isDeletedSegmentFile((String)JOURNAL_NAME, (String)file.getName())).isDirectoryContaining(file -> JournalSegmentFile.isSegmentFile((String)JOURNAL_NAME, (String)file.getName()));
    }

    @Test
    void shouldDeleteFilesMarkedForDeletionsOnLoad() {
        SegmentedJournal journal = this.openJournal(2.0f);
        journal.append(this.data);
        journal.openReader();
        journal.reset(100L);
        try (SegmentedJournal ignored = this.openJournal(2.0f);){
            File logDirectory = this.directory.resolve("data").toFile();
            Assertions.assertThat((File)logDirectory).isDirectoryNotContaining(file -> JournalSegmentFile.isDeletedSegmentFile((String)JOURNAL_NAME, (String)file.getName())).isDirectoryContaining(file -> JournalSegmentFile.isSegmentFile((String)JOURNAL_NAME, (String)file.getName()));
        }
    }

    @Test
    void shouldBeAbleToResetAgainWhileThePreviousFileIsNotDeleted() {
        SegmentedJournal journal = this.openJournal(2.0f);
        journal.append(this.data);
        journal.openReader();
        journal.reset(100L);
        journal.openReader();
        journal.reset(200L);
        File logDirectory = this.directory.resolve("data").toFile();
        Assertions.assertThat((Object[])logDirectory.listFiles(file -> JournalSegmentFile.isDeletedSegmentFile((String)JOURNAL_NAME, (String)file.getName()))).hasSize(2);
        Assertions.assertThat((Object[])logDirectory.listFiles(file -> JournalSegmentFile.isSegmentFile((String)JOURNAL_NAME, (String)file.getName()))).hasSize(1);
    }

    @Test
    void shouldReleaseReadLock() throws InterruptedException {
        CountDownLatch latch = new CountDownLatch(1);
        SegmentedJournal journal = this.openJournal(5.0f);
        JournalSegment segment = journal.getFirstSegment();
        segment.delete();
        Assertions.assertThat((boolean)segment.isOpen()).isFalse();
        AssertionsForClassTypes.assertThatThrownBy(() -> journal.openReader()).withFailMessage("Segment not open", new Object[0]);
        new Thread(() -> {
            journal.reset(100L);
            latch.countDown();
        }).start();
        Assertions.assertThat((boolean)latch.await(5L, TimeUnit.SECONDS)).isTrue();
    }

    private SegmentedJournal openJournal(float entriesPerSegment) {
        return this.openJournal(entriesPerSegment, this.entrySize);
    }

    private SegmentedJournal openJournal(float entriesPerSegment, int entrySize) {
        return SegmentedJournal.builder().withDirectory(this.directory.resolve("data").toFile()).withMaxSegmentSize((int)((float)entrySize * entriesPerSegment) + JournalSegmentDescriptor.getEncodingLength()).withJournalIndexDensity(1).withName(JOURNAL_NAME).build();
    }

    private int getSerializedSize(DirectBuffer data) {
        RecordData record = new RecordData(1L, 1L, data);
        SBESerializer serializer = new SBESerializer();
        ByteBuffer buffer = ByteBuffer.allocate(128);
        return (Integer)serializer.writeData(record, (MutableDirectBuffer)new UnsafeBuffer(buffer), 0).get() + FrameUtil.getLength() + serializer.getMetadataLength();
    }
}

