/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.engine.processing.job;

import io.camunda.zeebe.engine.processing.job.JobBatchCollector;
import io.camunda.zeebe.engine.state.immutable.JobState;
import io.camunda.zeebe.engine.state.immutable.VariableState;
import io.camunda.zeebe.engine.state.mutable.MutableProcessingState;
import io.camunda.zeebe.engine.state.mutable.MutableVariableState;
import io.camunda.zeebe.engine.util.MockTypedRecord;
import io.camunda.zeebe.engine.util.ProcessingStateExtension;
import io.camunda.zeebe.msgpack.value.StringValue;
import io.camunda.zeebe.protocol.impl.record.RecordMetadata;
import io.camunda.zeebe.protocol.impl.record.value.job.JobBatchRecord;
import io.camunda.zeebe.protocol.impl.record.value.job.JobRecord;
import io.camunda.zeebe.protocol.record.RecordType;
import io.camunda.zeebe.protocol.record.RecordValueWithVariables;
import io.camunda.zeebe.protocol.record.RecordValueWithVariablesAssert;
import io.camunda.zeebe.protocol.record.ValueType;
import io.camunda.zeebe.protocol.record.intent.Intent;
import io.camunda.zeebe.protocol.record.intent.JobBatchIntent;
import io.camunda.zeebe.protocol.record.value.JobBatchRecordValue;
import io.camunda.zeebe.protocol.record.value.JobBatchRecordValueAssert;
import io.camunda.zeebe.protocol.record.value.JobRecordValue;
import io.camunda.zeebe.protocol.record.value.JobRecordValueAssert;
import io.camunda.zeebe.stream.api.records.TypedRecord;
import io.camunda.zeebe.test.util.MsgPackUtil;
import io.camunda.zeebe.test.util.asserts.EitherAssert;
import io.camunda.zeebe.util.Either;
import io.camunda.zeebe.util.buffer.BufferUtil;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Predicate;
import org.agrona.DirectBuffer;
import org.agrona.collections.MutableReference;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ThrowingConsumer;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;

@ExtendWith(value={ProcessingStateExtension.class})
final class JobBatchCollectorTest {
    private static final String JOB_TYPE = "job";
    private final RecordLengthEvaluator lengthEvaluator = new RecordLengthEvaluator();
    private MutableProcessingState state;
    private JobBatchCollector collector;

    JobBatchCollectorTest() {
    }

    @BeforeEach
    void beforeEach() {
        this.collector = new JobBatchCollector((JobState)this.state.getJobState(), (VariableState)this.state.getVariableState(), (Predicate)this.lengthEvaluator);
    }

    @Test
    void shouldTruncateBatchIfNoMoreCanBeWritten() {
        long variableScopeKey = this.state.getKeyGenerator().nextKey();
        TypedRecord<JobBatchRecord> record = this.createRecord(new String[0]);
        List<Job> jobs = Arrays.asList(this.createJob(variableScopeKey), this.createJob(variableScopeKey));
        AtomicBoolean toggle = new AtomicBoolean(true);
        this.lengthEvaluator.canWriteEventOfLength = length -> toggle.getAndSet(false);
        Either result = this.collector.collectJobs(record);
        JobBatchRecord batchRecord = (JobBatchRecord)record.getValue();
        ((EitherAssert)EitherAssert.assertThat((Either)result).as("should have activated only one job successfully", new Object[0])).right().isEqualTo((Object)1);
        JobBatchRecordValueAssert.assertThat((JobBatchRecordValue)batchRecord).hasOnlyJobKeys(new Long[]{jobs.get((int)0).key}).isTruncated();
    }

    @Test
    void shouldReturnLargeJobIfFirstJobCannotBeWritten() {
        long variableScopeKey = this.state.getKeyGenerator().nextKey();
        TypedRecord<JobBatchRecord> record = this.createRecord(new String[0]);
        List<Job> jobs = Arrays.asList(this.createJob(variableScopeKey), this.createJob(variableScopeKey));
        this.lengthEvaluator.canWriteEventOfLength = length -> false;
        Either result = this.collector.collectJobs(record);
        JobBatchRecord batchRecord = (JobBatchRecord)record.getValue();
        ((EitherAssert)EitherAssert.assertThat((Either)result).as("should return excessively large job", new Object[0])).left().hasFieldOrPropertyWithValue("key", (Object)jobs.get((int)0).key);
        JobBatchRecordValueAssert.assertThat((JobBatchRecordValue)batchRecord).hasNoJobKeys().hasNoJobs().isTruncated();
    }

    @Test
    void shouldCollectJobsWithVariables() {
        TypedRecord<JobBatchRecord> record = this.createRecord(new String[0]);
        long firstScopeKey = this.state.getKeyGenerator().nextKey();
        long secondScopeKey = this.state.getKeyGenerator().nextKey();
        Map<String, String> firstJobVariables = Map.of("foo", "bar", "baz", "buz");
        Map<String, String> secondJobVariables = Map.of("fizz", "buzz");
        this.createJobWithVariables(firstScopeKey, firstJobVariables);
        this.createJobWithVariables(secondScopeKey, secondJobVariables);
        this.collector.collectJobs(record);
        JobBatchRecord batchRecord = (JobBatchRecord)record.getValue();
        JobBatchRecordValueAssert.assertThat((JobBatchRecordValue)batchRecord).satisfies(new ThrowingConsumer[]{batch -> {
            List activatedJobs = batch.getJobs();
            RecordValueWithVariablesAssert.assertThat((RecordValueWithVariables)((RecordValueWithVariables)activatedJobs.get(0))).hasVariables(firstJobVariables);
            RecordValueWithVariablesAssert.assertThat((RecordValueWithVariables)((RecordValueWithVariables)activatedJobs.get(1))).hasVariables(secondJobVariables);
        }});
    }

    @Test
    void shouldAppendJobKeyToBatchRecord() {
        TypedRecord<JobBatchRecord> record = this.createRecord(new String[0]);
        long scopeKey = this.state.getKeyGenerator().nextKey();
        List<Job> jobs = Arrays.asList(this.createJob(scopeKey), this.createJob(scopeKey));
        this.collector.collectJobs(record);
        JobBatchRecord batchRecord = (JobBatchRecord)record.getValue();
        JobBatchRecordValueAssert.assertThat((JobBatchRecordValue)batchRecord).hasJobKeys(new Long[]{jobs.get((int)0).key, jobs.get((int)1).key});
    }

    @Test
    void shouldActivateUpToMaxJobs() {
        TypedRecord<JobBatchRecord> record = this.createRecord(new String[0]);
        long scopeKey = this.state.getKeyGenerator().nextKey();
        List<Job> jobs = Arrays.asList(this.createJob(scopeKey), this.createJob(scopeKey));
        ((JobBatchRecord)record.getValue()).setMaxJobsToActivate(1);
        Either result = this.collector.collectJobs(record);
        JobBatchRecord batchRecord = (JobBatchRecord)record.getValue();
        ((EitherAssert)EitherAssert.assertThat((Either)result).as("should collect only the first job", new Object[0])).right().isEqualTo((Object)1);
        JobBatchRecordValueAssert.assertThat((JobBatchRecordValue)batchRecord).hasJobKeys(new Long[]{jobs.get((int)0).key}).isNotTruncated();
    }

    @Test
    void shouldSetDeadlineOnActivation() {
        TypedRecord<JobBatchRecord> record = this.createRecord(new String[0]);
        long scopeKey = this.state.getKeyGenerator().nextKey();
        long expectedDeadline = record.getTimestamp() + ((JobBatchRecord)record.getValue()).getTimeout();
        this.createJob(scopeKey);
        this.createJob(scopeKey);
        this.collector.collectJobs(record);
        JobBatchRecord batchRecord = (JobBatchRecord)record.getValue();
        JobBatchRecordValueAssert.assertThat((JobBatchRecordValue)batchRecord).satisfies(new ThrowingConsumer[]{batch -> {
            List activatedJobs = batch.getJobs();
            ((JobRecordValueAssert)JobRecordValueAssert.assertThat((JobRecordValue)((JobRecordValue)activatedJobs.get(0))).as("first activated job has the expected deadline", new Object[0])).hasDeadline(expectedDeadline);
            ((JobRecordValueAssert)JobRecordValueAssert.assertThat((JobRecordValue)((JobRecordValue)activatedJobs.get(1))).as("second activated job has the expected deadline", new Object[0])).hasDeadline(expectedDeadline);
        }});
    }

    @Test
    void shouldSetWorkerOnActivation() {
        TypedRecord<JobBatchRecord> record = this.createRecord(new String[0]);
        long scopeKey = this.state.getKeyGenerator().nextKey();
        String expectedWorker = "foo";
        this.createJob(scopeKey);
        this.createJob(scopeKey);
        ((JobBatchRecord)record.getValue()).setWorker("foo");
        this.collector.collectJobs(record);
        JobBatchRecord batchRecord = (JobBatchRecord)record.getValue();
        JobBatchRecordValueAssert.assertThat((JobBatchRecordValue)batchRecord).satisfies(new ThrowingConsumer[]{batch -> {
            List activatedJobs = batch.getJobs();
            ((JobRecordValueAssert)JobRecordValueAssert.assertThat((JobRecordValue)((JobRecordValue)activatedJobs.get(0))).as("first activated job has the expected worker", new Object[0])).hasWorker("foo");
            ((JobRecordValueAssert)JobRecordValueAssert.assertThat((JobRecordValue)((JobRecordValue)activatedJobs.get(1))).as("second activated job has the expected worker", new Object[0])).hasWorker("foo");
        }});
    }

    @Test
    void shouldFetchOnlyRequestedVariables() {
        TypedRecord<JobBatchRecord> record = this.createRecord(new String[0]);
        long firstScopeKey = this.state.getKeyGenerator().nextKey();
        long secondScopeKey = this.state.getKeyGenerator().nextKey();
        Map<String, String> firstJobVariables = Map.of("foo", "bar", "baz", "buz");
        Map<String, String> secondJobVariables = Map.of("fizz", "buzz");
        this.createJobWithVariables(firstScopeKey, firstJobVariables);
        this.createJobWithVariables(secondScopeKey, secondJobVariables);
        ((StringValue)((JobBatchRecord)record.getValue()).variables().add()).wrap(BufferUtil.wrapString((String)"foo"));
        this.collector.collectJobs(record);
        JobBatchRecord batchRecord = (JobBatchRecord)record.getValue();
        JobBatchRecordValueAssert.assertThat((JobBatchRecordValue)batchRecord).satisfies(new ThrowingConsumer[]{batch -> {
            List activatedJobs = batch.getJobs();
            RecordValueWithVariablesAssert.assertThat((RecordValueWithVariables)((RecordValueWithVariables)activatedJobs.get(0))).hasVariables(Map.of("foo", "bar"));
            RecordValueWithVariablesAssert.assertThat((RecordValueWithVariables)((RecordValueWithVariables)activatedJobs.get(1))).hasVariables(Collections.emptyMap());
        }});
    }

    @Test
    void shouldEstimateLengthCorrectly() {
        TypedRecord<JobBatchRecord> record = this.createRecord(new String[0]);
        long scopeKey = this.state.getKeyGenerator().nextKey();
        Map<String, String> variables = Map.of("foo", "bar");
        MutableReference estimatedLength = new MutableReference();
        int initialLength = record.getLength();
        this.createJobWithVariables(scopeKey, variables);
        this.lengthEvaluator.canWriteEventOfLength = length -> {
            estimatedLength.set(length);
            return true;
        };
        this.collector.collectJobs(record);
        JobRecord activatedJob = (JobRecord)((JobBatchRecord)record.getValue()).getJobs().get(0);
        int expectedLength = initialLength + activatedJob.getLength() + 8192;
        Assertions.assertThat((Integer)((Integer)estimatedLength.ref)).isEqualTo(expectedLength);
    }

    @Test
    public void shouldCollectOnlyCustomTenantJobs() {
        String tenantA = "tenant-a";
        String tenantB = "tenant-b";
        TypedRecord<JobBatchRecord> record = this.createRecord("tenant-a", "tenant-b");
        long firstScopeKey = this.state.getKeyGenerator().nextKey();
        long secondScopeKey = this.state.getKeyGenerator().nextKey();
        this.createJob(firstScopeKey, "tenant-a");
        this.createJob(secondScopeKey, "tenant-b");
        this.collector.collectJobs(record);
        JobBatchRecord batchRecord = (JobBatchRecord)record.getValue();
        JobBatchRecordValueAssert.assertThat((JobBatchRecordValue)batchRecord).satisfies(new ThrowingConsumer[]{batch -> {
            List activatedJobs = batch.getJobs();
            Assertions.assertThat((List)activatedJobs).hasSize(2);
            Assertions.assertThat(activatedJobs.stream().map(job -> job.getTenantId()).toList()).containsExactlyInAnyOrder((Object[])new String[]{"tenant-a", "tenant-b"});
        }});
    }

    @Test
    public void shouldCollectOnlyAuthorizedTenantJobs() {
        String tenantA = "tenant-a";
        String tenantB = "tenant-b";
        TypedRecord<JobBatchRecord> record = this.createRecord("tenant-a");
        long firstScopeKey = this.state.getKeyGenerator().nextKey();
        long secondScopeKey = this.state.getKeyGenerator().nextKey();
        this.createJob(firstScopeKey, "tenant-a");
        this.createJob(secondScopeKey, "tenant-b");
        this.collector.collectJobs(record);
        JobBatchRecord batchRecord = (JobBatchRecord)record.getValue();
        JobBatchRecordValueAssert.assertThat((JobBatchRecordValue)batchRecord).satisfies(new ThrowingConsumer[]{batch -> {
            List activatedJobs = batch.getJobs();
            Assertions.assertThat((List)activatedJobs).hasSize(1);
            JobRecordValueAssert.assertThat((JobRecordValue)((JobRecordValue)activatedJobs.get(0))).hasTenantId("tenant-a");
        }});
    }

    private TypedRecord<JobBatchRecord> createRecord(String ... tenantIds) {
        RecordMetadata metadata = new RecordMetadata().recordType(RecordType.COMMAND).intent((Intent)JobBatchIntent.ACTIVATE).valueType(ValueType.JOB_BATCH);
        JobBatchRecord batchRecord = new JobBatchRecord().setTimeout(Duration.ofSeconds(10L).toMillis()).setMaxJobsToActivate(10).setType(JOB_TYPE).setWorker("test");
        List<String> tenantIdsList = tenantIds.length > 0 ? List.of(tenantIds) : List.of("<default>");
        batchRecord.setTenantIds(tenantIdsList);
        return new MockTypedRecord<JobBatchRecord>(this.state.getKeyGenerator().nextKey(), metadata, batchRecord);
    }

    private Job createJob(long variableScopeKey) {
        return this.createJob(variableScopeKey, "<default>");
    }

    private Job createJob(long variableScopeKey, String tenantId) {
        JobRecord jobRecord = new JobRecord().setBpmnProcessId("process").setElementId("element").setElementInstanceKey(variableScopeKey).setType(JOB_TYPE).setTenantId(tenantId);
        long jobKey = this.state.getKeyGenerator().nextKey();
        this.state.getJobState().create(jobKey, jobRecord);
        return new Job(jobKey, jobRecord);
    }

    private void createJobWithVariables(long variableScopeKey, Map<String, String> variables) {
        this.setVariables(variableScopeKey, variables);
        this.createJob(variableScopeKey);
    }

    private void setVariables(long variableScopeKey, Map<String, String> variables) {
        MutableVariableState variableState = this.state.getVariableState();
        variables.forEach((key, value) -> variableState.setVariableLocal(variableScopeKey, variableScopeKey, variableScopeKey, BufferUtil.wrapString((String)key), this.packString((String)value)));
    }

    private DirectBuffer packString(String value) {
        return MsgPackUtil.encodeMsgPack(b -> b.packString(value));
    }

    private static final class RecordLengthEvaluator
    implements Predicate<Integer> {
        private Predicate<Integer> canWriteEventOfLength = length -> true;

        private RecordLengthEvaluator() {
        }

        @Override
        public boolean test(Integer length) {
            return this.canWriteEventOfLength.test(length);
        }
    }

    private record Job(long key, JobRecord job) {
    }
}

