/*
 * Decompiled with CFR 0.152.
 */
package io.continual.iam.impl.s3;

import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import io.continual.iam.exceptions.IamBadRequestException;
import io.continual.iam.exceptions.IamIdentityDoesNotExist;
import io.continual.iam.exceptions.IamSvcException;
import io.continual.iam.identity.ApiKey;
import io.continual.iam.impl.common.CommonJsonApiKey;
import io.continual.iam.impl.common.CommonJsonDb;
import io.continual.iam.impl.common.CommonJsonGroup;
import io.continual.iam.impl.common.CommonJsonIdentity;
import io.continual.iam.impl.common.jwt.JwtProducer;
import io.continual.util.collections.LruCache;
import io.continual.util.data.StreamTools;
import io.continual.util.data.json.CommentedJsonTokener;
import io.continual.util.data.json.JsonUtil;
import io.continual.util.data.json.JsonVisitor;
import io.continual.util.time.Clock;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3IamDb
extends CommonJsonDb<CommonJsonIdentity, CommonJsonGroup> {
    private final AmazonS3Client fDb;
    private final String fBucketId;
    private final String fPrefix;
    private final LruCache<String, JSONObject> fCache;
    private static final long kMaxCacheAgeMs = 60000L;
    private static final Logger log = LoggerFactory.getLogger(S3IamDb.class);

    protected S3IamDb(String s3ApiKey, String s3PrivateKey, String bucket, String prefix, CommonJsonDb.AclFactory aclFactory, JwtProducer jwtIssuer) {
        super(aclFactory, jwtIssuer);
        this.fDb = new AmazonS3Client((AWSCredentials)new S3Creds(s3ApiKey, s3PrivateKey));
        this.fBucketId = bucket;
        this.fPrefix = prefix;
        this.fCache = new LruCache(1024L);
    }

    protected void findOrCreateBucket() throws IamSvcException {
        try {
            if (!this.fDb.doesBucketExist(this.fBucketId)) {
                this.fDb.createBucket(this.fBucketId);
            }
        }
        catch (AmazonClientException x) {
            throw new IamSvcException((Throwable)x);
        }
    }

    public void close() {
        this.fDb.shutdown();
    }

    public Map<String, CommonJsonIdentity> loadAllUsers() throws IamSvcException {
        HashMap<String, CommonJsonIdentity> result = new HashMap<String, CommonJsonIdentity>();
        for (String userId : this.getAllUsers()) {
            result.put(userId, this.loadUser(userId));
        }
        return result;
    }

    public Collection<String> getAllUsers() throws IamSvcException {
        String prefix = this.concatPathParts(this.getPrefix(), "users/");
        LinkedList<String> result = new LinkedList<String>();
        for (String key : this.loadKeysBelow(prefix)) {
            String localPart = key.substring(prefix.length());
            if (localPart.length() <= 0) continue;
            result.add(localPart);
        }
        return result;
    }

    public Collection<String> getAllGroups() throws IamSvcException {
        String prefix = this.concatPathParts(this.getPrefix(), "groups/");
        LinkedList<String> result = new LinkedList<String>();
        for (String key : this.loadKeysBelow(prefix)) {
            String localPart = key.substring(prefix.length());
            if (localPart.length() <= 0) continue;
            result.add(localPart);
        }
        return result;
    }

    public List<String> findUsers(String startingWith) throws IamSvcException {
        String sysPrefix = this.concatPathParts(this.getPrefix(), "users");
        String prefix = this.concatPathParts(sysPrefix, startingWith);
        List<String> matches = this.loadKeysBelow(prefix);
        LinkedList<String> result = new LinkedList<String>();
        for (String match : matches) {
            result.add(match.substring(sysPrefix.length() + 1));
        }
        return result;
    }

    public void sweepExpiredTags() throws IamSvcException {
        TreeSet<String> keys = new TreeSet<String>();
        try {
            String prefix = this.makeByTagId("");
            ListObjectsRequest listObjectRequest = new ListObjectsRequest().withBucketName(this.fBucketId).withPrefix(prefix);
            ObjectListing objects = this.fDb.listObjects(listObjectRequest);
            do {
                for (S3ObjectSummary objectSummary : objects.getObjectSummaries()) {
                    String tagId = objectSummary.getKey().substring(prefix.length());
                    keys.add(tagId);
                }
            } while ((objects = this.fDb.listNextBatchOfObjects(objects)).isTruncated());
        }
        catch (AmazonS3Exception x) {
            throw new IamSvcException((Throwable)x);
        }
        for (String key : keys) {
            this.loadTagObject(key, false);
        }
    }

    String getPrefix() {
        if (this.fPrefix == null || this.fPrefix.length() == 0) {
            return "";
        }
        return this.fPrefix + "/";
    }

    String concatPathParts(String ... parts) {
        StringBuilder sb = new StringBuilder();
        String last = null;
        for (String part : parts) {
            if (last != null && !last.endsWith("/")) {
                sb.append("/");
            }
            if (part.startsWith("/")) {
                part = part.substring(1);
            }
            sb.append(part);
            last = part;
        }
        return sb.toString();
    }

    String makeUserId(String userId) {
        return this.concatPathParts(this.getPrefix(), "users/", userId);
    }

    String makeGroupId(String groupId) {
        return this.concatPathParts(this.getPrefix(), "groups/", groupId);
    }

    String makeByApiKeyId(String apiKeyId) {
        return this.concatPathParts(this.getPrefix(), "apikeys/byKey/", apiKeyId);
    }

    String makeByUserApiKeyId(String userId, String apiKeyId) {
        return this.concatPathParts(this.getPrefix(), "apikeys/byUser/", userId, apiKeyId);
    }

    String makeUserApiKeyFolderId(String userId) {
        return this.concatPathParts(this.getPrefix(), "apikeys/byUser/", userId);
    }

    String makeAclId(String aclId) {
        return this.concatPathParts(this.getPrefix(), "acls/", aclId);
    }

    String makeByTagId(String tagId) {
        return this.concatPathParts(this.getPrefix(), "tags/byTag/", tagId);
    }

    String makeByUserTagId(String userId, String type) {
        return this.concatPathParts(this.getPrefix(), "tags/byUser/", userId, type);
    }

    String makeByAliasId(String alias) {
        return this.concatPathParts(this.getPrefix(), "aliases/byKey/", alias);
    }

    String makeByUserAliasId(String userId) {
        return this.concatPathParts(this.getPrefix(), "aliases/byUser/", userId);
    }

    String makeJwtTokenId(String token) {
        return this.concatPathParts(this.getPrefix(), "invalidJwts/", token);
    }

    private InputStream load(String key) throws IamSvcException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        if (this.loadTo(key, baos)) {
            return new ByteArrayInputStream(baos.toByteArray());
        }
        return null;
    }

    private JSONObject loadObject(String key) throws IamSvcException {
        long startMs = Clock.now();
        try {
            JSONObject cached = (JSONObject)this.fCache.get((Object)key, 60000L);
            if (cached != null) {
                long durMs = Clock.now() - startMs;
                if (log.isDebugEnabled()) {
                    log.debug("S3IamDb.loadObject ( " + key + " ): from cache, " + durMs + " ms");
                }
                return cached;
            }
            InputStream is = this.load(key);
            if (is == null) {
                return null;
            }
            JSONObject result = new JSONObject((JSONTokener)new CommentedJsonTokener(is));
            this.fCache.put((Object)key, (Object)JsonUtil.clone((JSONObject)result));
            long durMs = Clock.now() - startMs;
            if (log.isDebugEnabled()) {
                log.debug("S3IamDb.loadObject ( " + key + " ): from S3, " + durMs + " ms");
            }
            return result;
        }
        catch (JSONException e) {
            throw new IamSvcException((Throwable)e);
        }
    }

    private boolean loadTo(String key, OutputStream os) throws IamSvcException {
        S3Object object = null;
        try {
            object = this.fDb.getObject(new GetObjectRequest(this.fBucketId, key));
            S3ObjectInputStream is = object.getObjectContent();
            StreamTools.copyStream((InputStream)is, (OutputStream)os);
            boolean bl = true;
            return bl;
        }
        catch (AmazonServiceException x) {
            if (404 == x.getStatusCode()) {
                boolean bl = false;
                return bl;
            }
            throw new IamSvcException((Throwable)x);
        }
        catch (AmazonClientException x) {
            throw new IamSvcException((Throwable)x);
        }
        catch (IOException x) {
            throw new IamSvcException((Throwable)x);
        }
        finally {
            if (object != null) {
                try {
                    object.close();
                }
                catch (IOException e) {
                    throw new IamSvcException((Throwable)e);
                }
            }
        }
    }

    List<String> loadKeysBelow(String key) throws IamSvcException {
        LinkedList<String> result = new LinkedList<String>();
        try {
            ObjectListing objectListing;
            ListObjectsRequest listObjectsRequest = new ListObjectsRequest().withBucketName(this.fBucketId).withPrefix(key);
            do {
                objectListing = this.fDb.listObjects(listObjectsRequest);
                for (S3ObjectSummary objectSummary : objectListing.getObjectSummaries()) {
                    result.add(objectSummary.getKey());
                }
                listObjectsRequest.setMarker(objectListing.getNextMarker());
            } while (objectListing.isTruncated());
        }
        catch (AmazonClientException e) {
            throw new IamSvcException((Throwable)e);
        }
        return result;
    }

    void storeObject(String key, JSONObject o) throws IamSvcException {
        try {
            this.fCache.put((Object)key, (Object)JsonUtil.clone((JSONObject)o));
            String data = o.toString();
            ByteArrayInputStream is = new ByteArrayInputStream(data.getBytes("UTF-8"));
            long length = data.length();
            ObjectMetadata om = new ObjectMetadata();
            om.setContentLength(length);
            om.setContentType("application/json");
            this.fDb.putObject(new PutObjectRequest(this.fBucketId, key, (InputStream)is, om));
        }
        catch (AmazonS3Exception x) {
            throw new IamSvcException((Throwable)x);
        }
        catch (UnsupportedEncodingException e) {
            throw new IamSvcException((Throwable)e);
        }
    }

    private void deleteObject(String key) throws IamSvcException {
        try {
            this.fCache.drop((Object)key);
            this.fDb.deleteObject(this.fBucketId, key);
        }
        catch (AmazonS3Exception x) {
            throw new IamSvcException((Throwable)x);
        }
    }

    protected JSONObject createNewUser(String id) {
        return CommonJsonIdentity.initializeIdentity();
    }

    protected JSONObject loadUserObject(String id) throws IamSvcException {
        return this.loadObject(this.makeUserId(id));
    }

    protected void storeUserObject(String id, JSONObject data) throws IamSvcException {
        this.storeObject(this.makeUserId(id), data);
    }

    protected void deleteUserObject(String id) throws IamSvcException {
        this.deleteObject(this.makeUserId(id));
    }

    protected CommonJsonIdentity instantiateIdentity(String id, JSONObject data) {
        return new CommonJsonIdentity(id, data, (CommonJsonDb)this);
    }

    protected JSONObject createNewGroup(String id, String groupDesc) {
        return CommonJsonGroup.initializeGroup((String)groupDesc);
    }

    protected JSONObject loadGroupObject(String id) throws IamSvcException {
        return this.loadObject(this.makeGroupId(id));
    }

    protected void storeGroupObject(String id, JSONObject data) throws IamSvcException {
        this.storeObject(this.makeGroupId(id), data);
    }

    protected void deleteGroupObject(String id) throws IamSvcException {
        this.deleteObject(this.makeGroupId(id));
    }

    protected CommonJsonGroup instantiateGroup(String id, JSONObject data) {
        return new CommonJsonGroup(id, data, (CommonJsonDb)this);
    }

    protected JSONObject createApiKeyObject(String userId, String apiKey, String apiSecret) {
        return CommonJsonApiKey.initialize((String)apiSecret, (String)userId);
    }

    protected JSONObject loadApiKeyObject(String id) throws IamSvcException {
        return this.loadObject(this.makeByApiKeyId(id));
    }

    protected void storeApiKeyObject(String id, JSONObject apiKeyObject) throws IamSvcException, IamBadRequestException {
        TreeSet existing;
        String userId = apiKeyObject.optString("userId", null);
        if (userId == null) {
            throw new IamBadRequestException("no user specified for api key");
        }
        JSONObject user = this.loadUserObject(userId);
        if (user == null) {
            throw new IamIdentityDoesNotExist(userId);
        }
        this.storeObject(this.makeByApiKeyId(id), apiKeyObject);
        JSONArray userApiKeys = user.optJSONArray("apiKeys");
        if (userApiKeys == null) {
            userApiKeys = new JSONArray();
            user.put("apiKeys", (Object)userApiKeys);
        }
        if (!(existing = new TreeSet(JsonVisitor.arrayToList((JSONArray)userApiKeys))).contains(id)) {
            userApiKeys.put((Object)id);
            this.storeUserObject(userId, user);
        }
    }

    protected void deleteApiKeyObject(String id) throws IamSvcException {
        String userId;
        JSONObject user;
        JSONArray userApiKeys;
        JSONObject apiKey = this.loadApiKeyObject(id);
        if (apiKey != null && (userApiKeys = (user = this.loadUserObject(userId = apiKey.getString("userId"))).optJSONArray("apiKeys")) != null && JsonUtil.removeStringFromArray((JSONArray)userApiKeys, (String)id)) {
            this.storeUserObject(userId, user);
        }
        this.deleteObject(this.makeByApiKeyId(id));
    }

    protected ApiKey instantiateApiKey(String id, JSONObject data) {
        return new CommonJsonApiKey(id, data);
    }

    protected Collection<String> loadApiKeysForUser(String userId) throws IamSvcException, IamIdentityDoesNotExist {
        JSONObject user = this.loadUserObject(userId);
        if (user == null) {
            throw new IamIdentityDoesNotExist(userId);
        }
        JSONArray userApiKeys = user.optJSONArray("apiKeys");
        if (userApiKeys != null) {
            return JsonVisitor.arrayToList((JSONArray)userApiKeys);
        }
        return new LinkedList<String>();
    }

    protected JSONObject loadAclObject(String id) throws IamSvcException {
        return this.loadObject(this.makeAclId(id));
    }

    protected void storeAclObject(String id, JSONObject data) throws IamSvcException {
        this.storeObject(this.makeAclId(id), data);
    }

    protected void deleteAclObject(String id) throws IamSvcException {
        this.deleteObject(this.makeAclId(id));
    }

    protected JSONObject loadTagObject(String id, boolean expiredOk) throws IamSvcException {
        JSONObject entry = this.loadObject(this.makeByTagId(id));
        if (entry == null) {
            return null;
        }
        long expireEpoch = entry.getLong("expireEpoch");
        if (expireEpoch < Clock.now() / 1000L && !expiredOk) {
            String userId = entry.optString("userId", null);
            String tagType = entry.optString("tagType", entry.optString("type", null));
            if (userId == null || tagType == null) {
                log.warn("Tag " + id + " is damaged.");
            } else {
                this.deleteTagObject(id, userId, tagType);
                log.info("Tag " + id + " (" + userId + "/" + tagType + ") deleted.");
            }
            return null;
        }
        return entry;
    }

    protected JSONObject loadTagObject(String userId, String appTagType, boolean expiredOk) throws IamSvcException {
        JSONObject entry = this.loadObject(this.makeByUserTagId(userId, appTagType));
        if (entry == null) {
            return null;
        }
        long expireEpoch = entry.getLong("expireEpoch");
        if (expireEpoch < Clock.now() / 1000L && !expiredOk) {
            this.removeMatchingTag(userId, appTagType);
            return null;
        }
        return entry;
    }

    protected void storeTagObject(String id, String userId, String appTagType, JSONObject data) throws IamSvcException {
        this.storeObject(this.makeByTagId(id), data);
        this.storeObject(this.makeByUserTagId(userId, appTagType), data);
    }

    protected void deleteTagObject(String id, String userId, String appTagType) throws IamSvcException {
        this.deleteObject(this.makeByTagId(id));
        this.deleteObject(this.makeByUserTagId(userId, appTagType));
    }

    protected JSONObject loadAliasObject(String id) throws IamSvcException {
        return this.loadObject(this.makeByAliasId(id));
    }

    protected void storeAliasObject(String id, JSONObject aliasObject) throws IamSvcException, IamBadRequestException {
        TreeSet existing;
        String userId = aliasObject.optString("userId", null);
        if (userId == null) {
            throw new IamBadRequestException("no user specified for alias");
        }
        JSONObject user = this.loadUserObject(userId);
        if (user == null) {
            throw new IamIdentityDoesNotExist(userId);
        }
        this.storeObject(this.makeByAliasId(id), aliasObject);
        JSONArray userAliases = user.optJSONArray("aliases");
        if (userAliases == null) {
            userAliases = new JSONArray();
            user.put("aliases", (Object)userAliases);
        }
        if (!(existing = new TreeSet(JsonVisitor.arrayToList((JSONArray)userAliases))).contains(id)) {
            userAliases.put((Object)id);
            this.storeUserObject(userId, user);
        }
    }

    protected void deleteAliasObject(String id) throws IamSvcException {
        String userId;
        JSONObject user;
        JSONArray userAliases;
        JSONObject alias = this.loadAliasObject(id);
        if (alias != null && (userAliases = (user = this.loadUserObject(userId = alias.getString("userId"))).optJSONArray("aliases")) != null && JsonUtil.removeStringFromArray((JSONArray)userAliases, (String)id)) {
            this.storeUserObject(userId, user);
        }
        this.deleteObject(this.makeByAliasId(id));
    }

    protected Collection<String> loadAliasesForUser(String userId) throws IamSvcException, IamIdentityDoesNotExist {
        JSONObject user = this.loadUserObject(userId);
        if (user == null) {
            throw new IamIdentityDoesNotExist(userId);
        }
        JSONArray userAliases = user.optJSONArray("aliases");
        if (userAliases != null) {
            return JsonVisitor.arrayToList((JSONArray)userAliases);
        }
        return new LinkedList<String>();
    }

    protected void storeInvalidJwtToken(String token) throws IamSvcException {
        this.storeObject(this.makeJwtTokenId(token), new JSONObject());
    }

    protected boolean isInvalidJwtToken(String token) throws IamSvcException {
        return null != this.loadObject(this.makeJwtTokenId(token));
    }

    private static class S3Creds
    implements AWSCredentials {
        private final String fAccessKey;
        private final String fPrivateKey;

        public S3Creds(String key, String secret) {
            this.fAccessKey = key;
            this.fPrivateKey = secret;
        }

        public String getAWSAccessKeyId() {
            return this.fAccessKey;
        }

        public String getAWSSecretKey() {
            return this.fPrivateKey;
        }
    }

    public static class Builder {
        private String apiKey;
        private String privateKey;
        private String bucket;
        private String prefix;
        private CommonJsonDb.AclFactory aclFactory;
        private boolean create = false;
        private JwtProducer jwtProducer = null;

        public Builder withAccessKey(String key) {
            this.apiKey = key;
            return this;
        }

        public Builder withSecretKey(String key) {
            this.privateKey = key;
            return this;
        }

        public Builder withBucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public Builder withPathPrefix(String pathPrefix) {
            this.prefix = pathPrefix;
            return this;
        }

        public Builder createBucketIfReqd() {
            this.create = true;
            return this;
        }

        public Builder usingAclFactory(CommonJsonDb.AclFactory af) {
            this.aclFactory = af;
            return this;
        }

        public Builder withJwtProducer(JwtProducer p) {
            this.jwtProducer = p;
            return this;
        }

        public S3IamDb build() throws IamSvcException {
            S3IamDb db = new S3IamDb(this.apiKey, this.privateKey, this.bucket, this.prefix, this.aclFactory, this.jwtProducer);
            if (this.create) {
                db.findOrCreateBucket();
            }
            return db;
        }
    }
}

