package cronapi.workflow;

import cronapi.CronapiMetaData;
import org.camunda.bpm.engine.AuthorizationException;
import org.camunda.bpm.engine.ProcessEngineException;
import org.camunda.bpm.engine.ProcessEngines;
import org.camunda.bpm.engine.TaskService;
import org.camunda.bpm.engine.authorization.Permissions;
import org.camunda.bpm.engine.authorization.Resources;
import org.camunda.bpm.engine.impl.context.Context;
import org.camunda.bpm.engine.task.*;
import org.camunda.bpm.engine.variable.VariableMap;
import org.camunda.bpm.engine.variable.value.SerializableValue;
import org.camunda.bpm.engine.variable.value.TypedValue;

import java.io.InputStream;
import java.util.Collection;
import java.util.List;
import java.util.Map;


/**
 * Serviço para operações dos Human Tasks do Bpmn ...
 *
 * @author Ricardo Caldas
 * @version 1.0
 * @since 2019-06-26
 */

@CronapiMetaData(categoryName = "Workflow Task Operations",
        categoryTags = {"Workflow", "Task", "Operations"})
public class WorkflowTaskOperations {

    private static TaskService getTaskService() {
        return ProcessEngines.getDefaultProcessEngine().getTaskService();
    }

    /**
     * create a new task with a user defined task id
     *
     * @param taskId
     */
    @CronapiMetaData(
            name = "{{newTask}}",
            description = "{{newTaskDescription}}"
    )
    public static Task newTask(String taskId) {
        return getTaskService().newTask(taskId);
    }

    /**
     * Saves the given task to the persistent data store. If the task is already
     * present in the persistent store, it is updated.
     * After a new task has been saved, the task instance passed into this method
     * is updated with the id of the newly created task.
     *
     * @param task the task, cannot be null.
     * @throws AuthorizationException If the task is already present and the user has no {@link Permissions#UPDATE} permission
     *                                on {@link Resources#TASK} or no {@link Permissions#UPDATE_TASK} permission on
     *                                {@link Resources#PROCESS_DEFINITION}.
     *                                Or if the task is not present and the user has no {@link Permissions#CREATE} permission
     *                                on {@link Resources#TASK}.
     */
    @CronapiMetaData(
            name = "{{saveTask}}",
            description = "{{saveTaskDescription}}"
    )
    public static void saveTask(Task task) {
        getTaskService().saveTask(task);
    }

    /**
     * Deletes the given task.
     *
     * @param taskId  The id of the task that will be deleted, cannot be null. If no task
     *                exists with the given taskId, the operation is ignored.
     * @param cascade If cascade is true, also the historic information related to this task is deleted.
     * @throws ProcessEngineException when an error occurs while deleting the task or in case the task is part
     *                                of a running process or case instance.
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE} permission on {@link Resources#TASK}.
     */
    @CronapiMetaData(
            name = "{{deleteTask}}",
            description = "{{deleteTaskDescription}}"
    )
    public static void deleteTask(String taskId, boolean cascade) {
        getTaskService().deleteTask(taskId, cascade);
    }

    /**
     * Deletes all tasks of the given collection.
     *
     * @param taskIds The id's of the tasks that will be deleted, cannot be null. All
     *                id's in the list that don't have an existing task will be ignored.
     * @param cascade If cascade is true, also the historic information related to this task is deleted.
     * @throws ProcessEngineException when an error occurs while deleting the tasks or in case one of the tasks
     *                                is part of a running process or case instance.
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE} permission on {@link Resources#TASK}.
     */
    @CronapiMetaData(
            name = "{{deleteTasks}}",
            description = "{{deleteTasksDescription}}"
    )
    public static void deleteTasks(Collection<String> taskIds, boolean cascade) {
        getTaskService().deleteTasks(taskIds, cascade);
    }

    /**
     * Claim responsibility for a task:
     * the given user is made {@link Task#getAssignee() assignee} for the task.
     * The difference with {@link #setAssignee(String, String)} is that here
     * a check is done if the task already has a user assigned to it.
     * No check is done whether the user is known by the identity component.
     *
     * @param taskId task to claim, cannot be null.
     * @param userId user that claims the task. When userId is null the task is unclaimed,
     *               assigned to no one.
     * @throws ProcessEngineException when the task doesn't exist or when the task is already claimed by another user.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{claim}}",
            description = "{{claimDescription}}"
    )
    public static void claim(String taskId, String userId) {
        getTaskService().claim(taskId, userId);
    }

    /**
     * Marks a task as done and continues process execution.
     * <p>
     * This method is typically called by a task list user interface
     * after a task form has been submitted by the
     * {@link Task#getAssignee() assignee}.
     *
     * @param taskId the id of the task to complete, cannot be null.
     * @throws ProcessEngineException when no task exists with the given id or when this task is {@link DelegationState#PENDING} delegation.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{complete}}",
            description = "{{completeDescription}}"
    )
    public static void complete(String taskId) {
        getTaskService().complete(taskId);
    }

    /**
     * Delegates the task to another user.
     * <p>
     * This means that the {@link Task#getAssignee() assignee} is set
     * and the {@link Task#getDelegationState() delegation state} is set to
     * {@link DelegationState#PENDING}.
     * If no owner is set on the task, the owner is set to the current
     * {@link Task#getAssignee() assignee} of the task.
     * The new assignee must use {@link TaskService#resolveTask(String)}
     * to report back to the owner.
     * Only the owner can {@link TaskService#complete(String) complete} the task.
     *
     * @param taskId The id of the task that will be delegated.
     * @param userId The id of the user that will be set as assignee.
     * @throws ProcessEngineException when no task exists with the given id.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{delegateTask}}",
            description = "{{delegateTaskDescription}}"
    )
    public static void delegateTask(String taskId, String userId) {
        getTaskService().delegateTask(taskId, userId);
    }

    /**
     * Marks that the {@link Task#getAssignee() assignee} is done with the task
     * {@link TaskService#delegateTask(String, String) delegated}
     * to her and that it can be sent back to the {@link Task#getOwner() owner}
     * with the provided variables.
     * Can only be called when this task is {@link DelegationState#PENDING} delegation.
     * After this method returns, the {@link Task#getDelegationState() delegation state}
     * is set to {@link DelegationState#RESOLVED} and the task can be
     * {@link TaskService#complete(String) completed}.
     *
     * @param taskId
     * @param variables
     * @throws ProcessEngineException when no task exists with the given id.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{resolveTask}}",
            description = "{{resolveTaskDescription}}"
    )
    public static void resolveTask(String taskId, Map<String, Object> variables) {
        getTaskService().resolveTask(taskId, variables);
    }

    /**
     * Marks a task as done and continues process execution.
     * <p>
     * This method is typically called by a task list user interface
     * after a task form has been submitted by the
     * {@link Task#getAssignee() assignee}
     * and the required task parameters have been provided.
     *
     * @param taskId    the id of the task to complete, cannot be null.
     * @param variables task parameters. May be null or empty.
     * @throws ProcessEngineException when no task exists with the given id.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{complete}}",
            description = "{{completeDescription}}"
    )
    public static void complete(String taskId, Map<String, Object> variables) {
        getTaskService().complete(taskId, variables);
    }

    /**
     * Changes the assignee of the given task to the given userId.
     * No check is done whether the user is known by the identity component.
     *
     * @param taskId id of the task, cannot be null.
     * @param userId id of the user to use as assignee.
     * @throws ProcessEngineException when the task or user doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{setAssignee}}",
            description = "{{setAssigneeDescription}}"
    )
    public static void setAssignee(String taskId, String userId) {
        getTaskService().setAssignee(taskId, userId);
    }

    /**
     * Transfers ownership of this task to another user.
     * No check is done whether the user is known by the identity component.
     *
     * @param taskId id of the task, cannot be null.
     * @param userId of the person that is receiving ownership.
     * @throws ProcessEngineException when the task or user doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{setOwner}}",
            description = "{{setOwnerDescription}}"
    )
    public static void setOwner(String taskId, String userId) {
        getTaskService().setOwner(taskId, userId);
    }

    /**
     * Retrieves the {@link IdentityLink}s associated with the given task.
     * Such an {@link IdentityLink} informs how a certain identity (eg. group or user)
     * is associated with a certain task (eg. as candidate, assignee, etc.)
     *
     * @param taskId
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{getIdentityLinksForTask}}",
            description = "{{getIdentityLinksForTaskDescription}}"
    )
    public static List<IdentityLink> getIdentityLinksForTask(String taskId) {
        return getTaskService().getIdentityLinksForTask(taskId);
    }

    /**
     * Convenience shorthand for {@link #addUserIdentityLink(String, String, String)}; with type {@link IdentityLinkType#CANDIDATE}
     *
     * @param taskId id of the task, cannot be null.
     * @param userId id of the user to use as candidate, cannot be null.
     * @throws ProcessEngineException when the task or user doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{addCandidateUser}}",
            description = "{{addCandidateUserDescription}}"
    )
    public static void addCandidateUser(String taskId, String userId) {
        getTaskService().addCandidateUser(taskId, userId);
    }

    /**
     * Convenience shorthand for {@link #addGroupIdentityLink(String, String, String)}; with type {@link IdentityLinkType#CANDIDATE}
     *
     * @param taskId  id of the task, cannot be null.
     * @param groupId id of the group to use as candidate, cannot be null.
     * @throws ProcessEngineException when the task or group doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{addCandidateGroup}}",
            description = "{{addCandidateGroupDescription}}"
    )
    public static void addCandidateGroup(String taskId, String groupId) {
        getTaskService().addCandidateGroup(taskId, groupId);
    }

    /**
     * Involves a user with a task. The type of identity link is defined by the
     * given identityLinkType.
     *
     * @param taskId           id of the task, cannot be null.
     * @param userId           id of the user involve, cannot be null.
     * @param identityLinkType type of identityLink, cannot be null (@see {@link IdentityLinkType}).
     * @throws ProcessEngineException when the task or user doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{addUserIdentityLink}}",
            description = "{{addUserIdentityLinkDescription}}"
    )
    public static void addUserIdentityLink(String taskId, String userId, String identityLinkType) {
        getTaskService().addUserIdentityLink(taskId, userId, identityLinkType);
    }

    /**
     * Involves a group with a task. The type of identityLink is defined by the
     * given identityLink.
     *
     * @param taskId           id of the task, cannot be null.
     * @param groupId          id of the group to involve, cannot be null.
     * @param identityLinkType type of identity, cannot be null (@see {@link IdentityLinkType}).
     * @throws ProcessEngineException when the task or group doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{addGroupIdentityLink}}",
            description = "{{addGroupIdentityLinkDescription}}"
    )
    public static void addGroupIdentityLink(String taskId, String groupId, String identityLinkType) {
        getTaskService().addGroupIdentityLink(taskId, groupId, identityLinkType);
    }

    /**
     * Convenience shorthand for {@link #deleteUserIdentityLink(String, String, String)}; with type {@link IdentityLinkType#CANDIDATE}
     *
     * @param taskId id of the task, cannot be null.
     * @param userId id of the user to use as candidate, cannot be null.
     * @throws ProcessEngineException when the task or user doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{deleteCandidateUser}}",
            description = "{{deleteCandidateUserDescription}}"
    )
    public static void deleteCandidateUser(String taskId, String userId) {
        getTaskService().deleteCandidateUser(taskId, userId);
    }

    /**
     * Convenience shorthand for {@link #deleteGroupIdentityLink(String, String, String)}; with type {@link IdentityLinkType#CANDIDATE}
     *
     * @param taskId  id of the task, cannot be null.
     * @param groupId id of the group to use as candidate, cannot be null.
     * @throws ProcessEngineException when the task or group doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{deleteCandidateGroup}}",
            description = "{{deleteCandidateGroupDescription}}"
    )
    public static void deleteCandidateGroup(String taskId, String groupId) {
        getTaskService().deleteCandidateGroup(taskId, groupId);
    }

    /**
     * Removes the association between a user and a task for the given identityLinkType.
     *
     * @param taskId           id of the task, cannot be null.
     * @param userId           id of the user involve, cannot be null.
     * @param identityLinkType type of identityLink, cannot be null (@see {@link IdentityLinkType}).
     * @throws ProcessEngineException when the task or user doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{deleteUserIdentityLink}}",
            description = "{{deleteUserIdentityLinkDescription}}"
    )
    public static void deleteUserIdentityLink(String taskId, String userId, String identityLinkType) {
        getTaskService().deleteUserIdentityLink(taskId, userId, identityLinkType);
    }

    /**
     * Removes the association between a group and a task for the given identityLinkType.
     *
     * @param taskId           id of the task, cannot be null.
     * @param groupId          id of the group to involve, cannot be null.
     * @param identityLinkType type of identity, cannot be null (@see {@link IdentityLinkType}).
     * @throws ProcessEngineException when the task or group doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{deleteGroupIdentityLink}}",
            description = "{{deleteGroupIdentityLinkDescription}}"
    )
    public static void deleteGroupIdentityLink(String taskId, String groupId, String identityLinkType) {
        getTaskService().deleteGroupIdentityLink(taskId, groupId, identityLinkType);
    }

    /**
     * Changes the priority of the task.
     * <p>
     * Authorization: actual owner / business admin
     *
     * @param taskId   id of the task, cannot be null.
     * @param priority the new priority for the task.
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{setPriority}}",
            description = "{{setPriorityDescription}}"
    )
    public static void setPriority(String taskId, int priority) {
        getTaskService().setPriority(taskId, priority);
    }

    /**
     * Returns a new {@link TaskQuery} that can be used to dynamically query tasks.
     */
    @CronapiMetaData(
            name = "{{createTaskQuery}}",
            description = "{{createTaskQueryDescription}}"
    )
    public static TaskQuery createTaskQuery(String taskID) {
        return getTaskService().createTaskQuery().taskId(taskID);
    }

    /**
     * Returns a new
     */
    @CronapiMetaData(
            name = "{{createNativeTaskQuery}}",
            description = "{{createNativeTaskQueryDescription}}"
    )
    public static NativeTaskQuery createNativeTaskQuery() {
        return getTaskService().createNativeTaskQuery();
    }

    /**
     * Set variable on a task. If the variable is not already existing, it will be created in the
     * most outer scope.  This means the process instance in case this task is related to an
     * execution.
     *
     * @param taskId
     * @param variableName
     * @param value
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{setVariable}}",
            description = "{{setVariableDescription}}"
    )
    public static void setVariable(String taskId, String variableName, Object value) {
        getTaskService().setVariable(taskId, variableName, value);
    }

    /**
     * Set variables on a task. If the variable is not already existing, it will be created in the
     * most outer scope.  This means the process instance in case this task is related to an
     * execution.
     *
     * @param taskId
     * @param variables
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{setVariables}}",
            description = "{{setVariablesDescription}}"
    )
    public static void setVariables(String taskId, Map<String, ?> variables) {
        getTaskService().setVariables(taskId, variables);
    }

    /**
     * Set variable on a task. If the variable is not already existing, it will be created in the
     * task.
     *
     * @param taskId
     * @param variableName
     * @param value
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{setVariableLocal}}",
            description = "{{setVariableLocalDescription}}"
    )
    public static void setVariableLocal(String taskId, String variableName, Object value) {
        getTaskService().setVariableLocal(taskId, variableName, value);
    }

    /**
     * Set variables on a task. If the variable is not already existing, it will be created in the
     * task.
     *
     * @param taskId
     * @param variables
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{setVariablesLocal}}",
            description = "{{setVariablesLocalDescription}}"
    )
    public static void setVariablesLocal(String taskId, Map<String, ?> variables) {
        getTaskService().setVariablesLocal(taskId, variables);
    }

    /**
     * Get a variables and search in the task scope and if available also the execution scopes.
     *
     * @param taskId
     * @param variableName
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{getVariable}}",
            description = "{{getVariableDescription}}"
    )
    public static Object getVariable(String taskId, String variableName) {
        return getTaskService().getVariable(taskId, variableName);
    }

    /**
     * Get a variables and search in the task scope and if available also the execution scopes.
     *
     * @param taskId       the id of the task
     * @param variableName the name of the variable to fetch
     * @return the TypedValue for the variable or 'null' in case no such variable exists.
     * @throws ClassCastException     in case the value is not of the requested type
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     * @since 7.2
     */
    @CronapiMetaData(
            name = "{{getVariableTyped}}",
            description = "{{getVariableTypedDescription}}"
    )
    public static <T extends TypedValue> T getVariableTyped(String taskId, String variableName) {
        return getTaskService().getVariableTyped(taskId, variableName);
    }

    /**
     * Get a variables and search in the task scope and if available also the execution scopes.
     *
     * @param taskId           the id of the task
     * @param variableName     the name of the variable to fetch
     * @param deserializeValue if false a, {@link SerializableValue} will not be deserialized.
     * @return the TypedValue for the variable or 'null' in case no such variable exists.
     * @throws ClassCastException     in case the value is not of the requested type
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     * @since 7.2
     */
    @CronapiMetaData(
            name = "{{getVariableTyped}}",
            description = "{{getVariableTypedDescription}}"
    )
    public static <T extends TypedValue> T getVariableTyped(String taskId, String variableName, boolean deserializeValue) {
        return getTaskService().getVariableTyped(taskId, variableName, deserializeValue);
    }

    /**
     * Get a variables and only search in the task scope.
     *
     * @param taskId
     * @param variableName
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{getVariableLocal}}",
            description = "{{getVariableLocalDescription}}"
    )
    public static Object getVariableLocal(String taskId, String variableName) {
        return getTaskService().getVariableLocal(taskId, variableName);
    }

    /**
     * Get a variables and only search in the task scope.
     *
     * @param taskId           the id of the task
     * @param variableName     the name of the variable to fetch
     * @param deserializeValue if false a, {@link SerializableValue} will not be deserialized.
     * @return the TypedValue for the variable or 'null' in case no such variable exists.
     * @throws ClassCastException     in case the value is not of the requested type
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     * @since 7.2
     */
    @CronapiMetaData(
            name = "{{getVariableLocalTyped}}",
            description = "{{getVariableLocalTypedDescription}}"
    )
    public static <T extends TypedValue> T getVariableLocalTyped(String taskId, String variableName, boolean deserializeValue) {
        return getTaskService().getVariableLocalTyped(taskId, variableName, deserializeValue);
    }

    /**
     * Get all variables and search in the task scope and if available also the execution scopes.
     * If you have many variables and you only need a few, consider using {@link #getVariables(String, Collection)}
     * for better performance.
     *
     * @param taskId
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{getVariables}}",
            description = "{{getVariablesDescription}}"
    )
    public static Map<String, Object> getVariables(String taskId) {
        return getTaskService().getVariables(taskId);
    }

    /**
     * Get all variables and search in the task scope and if available also the execution scopes.
     * If you have many variables and you only need a few, consider using {@link #getVariables(String, Collection)}
     * for better performance.
     *
     * @param taskId            the id of the task
     * @param deserializeValues if false, {@link SerializableValue SerializableValues} will not be deserialized.
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     * @since 7.2
     */
    @CronapiMetaData(
            name = "{{getVariablesTyped}}",
            description = "{{getVariablesTypedDescription}}"
    )
    public static VariableMap getVariablesTyped(String taskId, boolean deserializeValues) {
        return getTaskService().getVariablesTyped(taskId, deserializeValues);
    }

    /**
     * Get all variables and search only in the task scope.
     * If you have many task local variables and you only need a few, consider using {@link #getVariablesLocal(String, Collection)}
     * for better performance.
     *
     * @param taskId
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{getVariablesLocal}}",
            description = "{{getVariablesLocalDescription}}"
    )
    public static Map<String, Object> getVariablesLocal(String taskId) {
        return getTaskService().getVariablesLocal(taskId);
    }

    /**
     * Get all variables and search only in the task scope.
     * If you have many task local variables and you only need a few, consider using {@link #getVariablesLocal(String, Collection)}
     * for better performance.
     *
     * @param taskId            the id of the task
     * @param deserializeValues if false, {@link SerializableValue SerializableValues} will not be deserialized.
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     * @since 7.2
     */
    @CronapiMetaData(
            name = "{{getVariablesLocalTyped}}",
            description = "{{getVariablesLocalTypedDescription}}"
    )
    public static VariableMap getVariablesLocalTyped(String taskId, boolean deserializeValues) {
        return getTaskService().getVariablesLocalTyped(taskId, deserializeValues);
    }

    /**
     * Get values for all given variableNames
     *
     * @param taskId
     * @param variableNames
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{getVariables}}",
            description = "{{getVariablesDescription}}"
    )
    public static Map<String, Object> getVariables(String taskId, Collection<String> variableNames) {
        return getTaskService().getVariables(taskId, variableNames);
    }

    /**
     * Get values for all given variableName
     *
     * @param taskId            the id of the task
     * @param variableNames     only fetch variables whose names are in the collection.
     * @param deserializeValues if false, {@link SerializableValue SerializableValues} will not be deserialized.
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     * @since 7.2
     */
    @CronapiMetaData(
            name = "{{getVariablesTyped}}",
            description = "{{getVariablesTypedDescription}}"
    )
    public static VariableMap getVariablesTyped(String taskId, Collection<String> variableNames, boolean deserializeValues) {
        return getTaskService().getVariablesTyped(taskId, variableNames, deserializeValues);
    }

    /**
     * Get a variable on a task
     *
     * @param taskId
     * @param variableNames
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{getVariablesLocal}}",
            description = "{{getVariablesLocalDescription}}"
    )
    public static Map<String, Object> getVariablesLocal(String taskId, Collection<String> variableNames) {
        return getTaskService().getVariablesLocal(taskId, variableNames);
    }

    /**
     * Get values for all given variableName. Only search in the local task scope.
     *
     * @param taskId            the id of the task
     * @param variableNames     only fetch variables whose names are in the collection.
     * @param deserializeValues if false, {@link SerializableValue SerializableValues} will not be deserialized.
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     * @since 7.2
     */
    @CronapiMetaData(
            name = "{{getVariablesLocalTyped}}",
            description = "{{getVariablesLocalTypedDescription}}"
    )
    public static VariableMap getVariablesLocalTyped(String taskId, Collection<String> variableNames, boolean deserializeValues) {
        return getTaskService().getVariablesLocalTyped(taskId, variableNames, deserializeValues);
    }

    /**
     * Removes the variable from the task.
     * When the variable does not exist, nothing happens.
     *
     * @param taskId
     * @param variableName
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{removeVariable}}",
            description = "{{removeVariableDescription}}"
    )
    public static void removeVariable(String taskId, String variableName) {
        getTaskService().removeVariableLocal(taskId, variableName);
    }

    /**
     * Removes the variable from the task (not considering parent scopes).
     * When the variable does not exist, nothing happens.
     *
     * @param taskId
     * @param variableName
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{removeVariableLocal}}",
            description = "{{removeVariableLocalDescription}}"
    )
    public static void removeVariableLocal(String taskId, String variableName) {
        getTaskService().removeVariableLocal(taskId, variableName);
    }

    /**
     * Removes all variables in the given collection from the task.
     * Non existing variable names are simply ignored.
     *
     * @param taskId
     * @param variableNames
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{removeVariables}}",
            description = "{{removeVariablesDescription}}"
    )
    public static void removeVariables(String taskId, Collection<String> variableNames) {
        getTaskService().removeVariables(taskId, variableNames);
    }

    /**
     * Removes all variables in the given collection from the task (not considering parent scopes).
     * Non existing variable names are simply ignored.
     *
     * @param taskId
     * @param variableNames
     * @throws ProcessEngineException when the task doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}
     *                                (if the task is part of a running process instance).
     */
    @CronapiMetaData(
            name = "{{removeVariablesLocal}}",
            description = "{{removeVariablesLocalDescription}}"
    )
    public static void removeVariablesLocal(String taskId, Collection<String> variableNames) {
        getTaskService().removeVariablesLocal(taskId, variableNames);
    }

    /**
     * Creates a comment to a task and/or process instance and returns the comment.
     *
     * @param taskId
     * @param processInstanceId
     * @param message
     */
    @CronapiMetaData(
            name = "{{createComment}}",
            description = "{{createCommentDescription}}"
    )
    public static Comment createComment(String taskId, String processInstanceId, String message) {
        return getTaskService().createComment(taskId, processInstanceId, message);
    }

    /**
     * The comments related to the given task.
     *
     * @param taskId
     */
    @CronapiMetaData(
            name = "{{getTaskComments}}",
            description = "{{getTaskCommentsDescription}}"
    )
    public static List<Comment> getTaskComments(String taskId) {
        return getTaskService().getTaskComments(taskId);
    }

    /**
     * Retrieve a particular task comment
     *
     * @param taskId
     * @param commentId
     */
    @CronapiMetaData(
            name = "{{getTaskComment}}",
            description = "{{getTaskCommentDescription}}"
    )
    public static Comment getTaskComment(String taskId, String commentId) {
        return getTaskService().getTaskComment(taskId, commentId);
    }

    /**
     * The comments related to the given process instance.
     *
     * @param processInstanceId
     */
    @CronapiMetaData(
            name = "{{getProcessInstanceComments}}",
            description = "{{getProcessInstanceCommentsDescription}}"
    )
    public static List<Comment> getProcessInstanceComments(String processInstanceId) {
        return getTaskService().getProcessInstanceComments(processInstanceId);
    }

    /**
     * Add a new attachment to a task and/or a process instance and use an input stream to provide the content
     * please use method in runtime service to operate on process instance.
     * <p>
     * Either taskId or processInstanceId has to be provided
     *
     * @param attachmentType        - name of the attachment, can be null
     * @param taskId                - task that should have an attachment
     * @param processInstanceId     - id of a process to use if task id is null
     * @param attachmentName        - name of the attachment, can be null
     * @param attachmentDescription - full text description, can be null
     * @param content               - byte array with content of attachment
     */
    @CronapiMetaData(
            name = "{{createAttachment}}",
            description = "{{createAttachmentDescription}}"
    )
    public static Attachment createAttachment(String attachmentType, String taskId, String processInstanceId, String attachmentName, String attachmentDescription, InputStream content) {
        return getTaskService().createAttachment(attachmentType, taskId, processInstanceId, attachmentName, attachmentDescription, content);
    }

    /**
     * Update the name and decription of an attachment
     *
     * @param attachment
     */
    @CronapiMetaData(
            name = "{{saveAttachment}}",
            description = "{{saveAttachmentDescription}}"
    )
    public static void saveAttachment(Attachment attachment) {
        getTaskService().saveAttachment(attachment);
    }

    /**
     * Retrieve a particular attachment
     *
     * @param attachmentId
     */
    @CronapiMetaData(
            name = "{{getAttachment}}",
            description = "{{getAttachmentDescription}}"
    )
    public static Attachment getAttachment(String attachmentId) {
        return getTaskService().getAttachment(attachmentId);
    }

    /**
     * Retrieve a particular attachment to the given task id and attachment id
     *
     * @param taskId
     * @param attachmentId
     */
    @CronapiMetaData(
            name = "{{getTaskAttachment}}",
            description = "{{getTaskAttachmentDescription}}"
    )
    public static Attachment getTaskAttachment(String taskId, String attachmentId) {
        return getTaskService().getTaskAttachment(taskId, attachmentId);
    }

    /**
     * Retrieve stream content of a particular attachment
     *
     * @param attachmentId
     */
    @CronapiMetaData(
            name = "{{getAttachmentContent}}",
            description = "{{getAttachmentContentDescription}}"
    )
    public static InputStream getAttachmentContent(String attachmentId) {
        return getTaskService().getAttachmentContent(attachmentId);
    }

    /**
     * Retrieve stream content of a particular attachment to the given task id and attachment id
     *
     * @param taskId
     * @param attachmentId
     */
    @CronapiMetaData(
            name = "{{getTaskAttachmentContent}}",
            description = "{{getTaskAttachmentContentDescription}}"
    )
    public static InputStream getTaskAttachmentContent(String taskId, String attachmentId) {
        return getTaskService().getTaskAttachmentContent(taskId, attachmentId);
    }

    /**
     * The list of attachments associated to a task
     *
     * @param taskId
     */
    @CronapiMetaData(
            name = "{{getTaskAttachments}}",
            description = "{{getTaskAttachmentsDescription}}"
    )
    public static List<Attachment> getTaskAttachments(String taskId) {
        return getTaskService().getTaskAttachments(taskId);
    }

    /**
     * The list of attachments associated to a process instance
     *
     * @param processInstanceId
     */
    @CronapiMetaData(
            name = "{{getProcessInstanceAttachments}}",
            description = "{{getProcessInstanceAttachmentsDescription}}"
    )
    public static List<Attachment> getProcessInstanceAttachments(String processInstanceId) {
        return getTaskService().getProcessInstanceAttachments(processInstanceId);
    }

    /**
     * Delete an attachment
     *
     * @param attachmentId
     */
    @CronapiMetaData(
            name = "{{deleteAttachment}}",
            description = "{{deleteAttachmentDescription}}"
    )
    public static void deleteAttachment(String attachmentId) {
        getTaskService().deleteAttachment(attachmentId);
    }

    /**
     * Delete an attachment to the given task id and attachment id
     *
     * @param taskId
     * @param attachmentId
     */
    @CronapiMetaData(
            name = "{{deleteTaskAttachment}}",
            description = "{{deleteTaskAttachmentDescription}}"
    )
    public static void deleteTaskAttachment(String taskId, String attachmentId) {
        getTaskService().deleteTaskAttachment(taskId, attachmentId);
    }

    /**
     * The list of subtasks for this parent task
     *
     * @param parentTaskId
     */
    @CronapiMetaData(
            name = "{{getSubTasks}}",
            description = "{{getSubTasksDescription}}"
    )
    public static List<Task> getSubTasks(String parentTaskId) {
        return getTaskService().getSubTasks(parentTaskId);
    }

    /**
     * Instantiate a task report
     */
    @CronapiMetaData(
            name = "{{createTaskReport}}",
            description = "{{createTaskReportDescription}}"
    )
    public static TaskReport createTaskReport() {
        return getTaskService().createTaskReport();
    }
}