package cronapi.workflow;

import cronapi.CronapiMetaData;
import org.camunda.bpm.application.ProcessApplicationReference;
import org.camunda.bpm.engine.*;
import org.camunda.bpm.engine.authorization.Permissions;
import org.camunda.bpm.engine.authorization.Resources;
import org.camunda.bpm.engine.delegate.ExecutionListener;
import org.camunda.bpm.engine.exception.NotFoundException;
import org.camunda.bpm.engine.exception.NotValidException;
import org.camunda.bpm.engine.impl.context.Context;
import org.camunda.bpm.engine.repository.*;
import org.camunda.bpm.engine.task.IdentityLink;
import org.camunda.bpm.model.bpmn.BpmnModelInstance;
import org.camunda.bpm.model.cmmn.CmmnModelInstance;
import org.camunda.bpm.model.dmn.DmnModelInstance;

import java.io.InputStream;
import java.util.Date;
import java.util.List;

@CronapiMetaData(categoryName = "Workflow Repository Operations",
        categoryTags = {"Workflow", "Repository", "Operations"})
public class WorkflowRepositoryOperations {


    private static RepositoryService getRepositoryService() {
        return ProcessEngines.getDefaultProcessEngine().getRepositoryService();
    }

    /**
     * Starts creating a new deployment
     * @return DeploymentBuilder
     */
    @CronapiMetaData(
            name = "{{createDeployment}}",
            description = "{{createDeploymentDescription}}"
    )
    public static DeploymentBuilder createDeployment() {
        return getRepositoryService().createDeployment();
    }

    /**
     * Starts creating a new {@link ProcessApplicationDeployment}.
     *
     * @param processApplication ProcessApplicationReference
     * @return ProcessApplicationDeploymentBuilder
     * @see ProcessApplicationDeploymentBuilder
     */
    @CronapiMetaData(
            name = "{{createDeployment}}",
            description = "{{createDeploymentDescription}}"
    )
    public static ProcessApplicationDeploymentBuilder createDeployment(ProcessApplicationReference processApplication) {
        return getRepositoryService().createDeployment(processApplication);
    }

    /**
     * Deletes the given deployment.
     *
     * @param deploymentId id of the deployment, cannot be null.
     * @throws RuntimeException       If there are still runtime or history process instances or jobs.
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE} permission on {@link Resources#DEPLOYMENT}.
     */
    @CronapiMetaData(
            name = "{{deleteDeployment}}",
            description = "{{deleteDeploymentDescription}}"
    )
    public static void deleteDeployment(String deploymentId) {
        getRepositoryService().deleteDeployment(deploymentId);
    }

    /**
     * Deletes the given deployment and cascade deletion to process instances,
     * history process instances and jobs.
     *
     * @param deploymentId id of the deployment, cannot be null.
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE} permission on {@link Resources#DEPLOYMENT}.
     * @deprecated use {@link #deleteDeployment(String, boolean)}. This methods may be deleted from 5.3.
     */
    @CronapiMetaData(
            name = "{{deleteDeploymentCascade}}",
            description = "{{deleteDeploymentCascadeDescription}}"
    )
    public static void deleteDeploymentCascade(String deploymentId) {
        getRepositoryService().deleteDeploymentCascade(deploymentId);
    }

    /**
     * Deletes the given deployment and cascade deletion to process instances,
     * history process instances and jobs.
     *
     * @param deploymentId id of the deployment, cannot be null.
     * @param cascade boolean
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE} permission on {@link Resources#DEPLOYMENT}.
     */
    @CronapiMetaData(
            name = "{{deleteDeployment}}",
            description = "{{deleteDeploymentDescription}}"
    )
    public static void deleteDeployment(String deploymentId, boolean cascade) {
        getRepositoryService().deleteDeployment(deploymentId, cascade);
    }

    /**
     * Deletes the given deployment and cascade deletion to process instances,
     * history process instances and jobs.
     *
     * @param deploymentId        id of the deployment, cannot be null.
     * @param cascade             if set to true, all process instances (including) history are deleted
     * @param skipCustomListeners if true, only the built-in {@link ExecutionListener}s
     *                            are notified with the {@link ExecutionListener#EVENTNAME_END} event.
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE} permission on {@link Resources#DEPLOYMENT}.
     */
    @CronapiMetaData(
            name = "{{deleteDeployment}}",
            description = "{{deleteDeploymentDescription}}"
    )
    public static void deleteDeployment(String deploymentId, boolean cascade, boolean skipCustomListeners) {
        getRepositoryService().deleteDeployment(deploymentId, cascade, skipCustomListeners);
    }

    /**
     * Deletes the given deployment and cascade deletion to process instances,
     * history process instances and jobs.
     *
     * @param deploymentId        id of the deployment, cannot be null.
     * @param cascade             if set to true, all process instances (including) history are deleted
     * @param skipCustomListeners if true, only the built-in {@link ExecutionListener}s
     *                            are notified with the {@link ExecutionListener#EVENTNAME_END} event.
     * @param skipIoMappings      specifies whether input/output mappings for tasks should be invoked
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE} permission on {@link Resources#DEPLOYMENT}.
     */
    @CronapiMetaData(
            name = "{{deleteDeployment}}",
            description = "{{deleteDeploymentDescription}}"
    )
    public static void deleteDeployment(String deploymentId, boolean cascade, boolean skipCustomListeners, boolean skipIoMappings) {
        getRepositoryService().deleteDeployment(deploymentId, cascade, skipCustomListeners, skipIoMappings);
    }

    /**
     * Deletes the process definition which belongs to the given process definition id.
     * Same behavior as {@link RepositoryService#deleteProcessDefinition(String, boolean, boolean)}
     * Both boolean parameters of this method are per default false. The deletion is
     * in this case not cascading.
     *
     * @param processDefinitionId the id, which corresponds to the process definition
     * @throws ProcessEngineException If the process definition does not exist
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE} permission on {@link Resources#PROCESS_DEFINITION}.
     * @see RepositoryService#deleteProcessDefinition(String, boolean, boolean)
     */
    @CronapiMetaData(
            name = "{{deleteProcessDefinition}}",
            description = "{{deleteProcessDefinitionDescription}}"
    )
    public static void deleteProcessDefinition(String processDefinitionId) {
        getRepositoryService().deleteProcessDefinition(processDefinitionId);
    }

    /**
     * Deletes the process definition which belongs to the given process definition id.
     * Cascades the deletion if the cascade is set to true.
     * Same behavior as {@link RepositoryService#deleteProcessDefinition(String, boolean, boolean)}
     * The skipCustomListeners parameter is per default false. The custom listeners are called
     * if the cascading flag is set to true and the process instances are deleted.
     *
     * @param processDefinitionId the id, which corresponds to the process definition
     * @param cascade             if set to true, all process instances (including) history are deleted
     * @throws ProcessEngineException If the process definition does not exist
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE} permission on {@link Resources#PROCESS_DEFINITION}.
     * @see RepositoryService#deleteProcessDefinition(String, boolean, boolean)
     */
    @CronapiMetaData(
            name = "{{deleteProcessDefinition}}",
            description = "{{deleteProcessDefinitionDescription}}"
    )
    public static void deleteProcessDefinition(String processDefinitionId, boolean cascade) {
        getRepositoryService().deleteProcessDefinition(processDefinitionId, cascade);
    }

    /**
     * Deletes the process definition which belongs to the given process definition id.
     * Cascades the deletion if the cascade is set to true the custom listener
     * can be skipped if the third parameter is set to true.
     *
     * @param processDefinitionId the id, which corresponds to the process definition
     * @param cascade             if set to true, all process instances (including) history are deleted
     * @param skipCustomListeners if true, only the built-in {@link ExecutionListener}s
     *                            are notified with the {@link ExecutionListener#EVENTNAME_END} event.
     *                            Is only used if cascade set to true.
     * @throws ProcessEngineException If the process definition does not exist
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{deleteProcessDefinition}}",
            description = "{{deleteProcessDefinitionDescription}}"
    )
    public static void deleteProcessDefinition(String processDefinitionId, boolean cascade, boolean skipCustomListeners) {
        getRepositoryService().deleteProcessDefinition(processDefinitionId, cascade, skipCustomListeners);
    }

    /**
     * Deletes the process definition which belongs to the given process definition id.
     * Cascades the deletion if the cascade is set to true, the custom listener can be skipped if
     * the third parameter is set to true, io mappings can be skipped if the forth parameter is set to true.
     *
     * @param processDefinitionId the id, which corresponds to the process definition
     * @param cascade             if set to true, all process instances (including) history are deleted
     * @param skipCustomListeners if true, only the built-in {@link ExecutionListener}s
     *                            are notified with the {@link ExecutionListener#EVENTNAME_END} event.
     *                            Is only used if cascade set to true.
     * @param skipIoMappings      Specifies whether input/output mappings for tasks should be invoked
     * @throws ProcessEngineException If the process definition does not exist
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{deleteProcessDefinition}}",
            description = "{{deleteProcessDefinitionDescription}}"
    )
    public static void deleteProcessDefinition(String processDefinitionId, boolean cascade, boolean skipCustomListeners, boolean skipIoMappings) {
        getRepositoryService().deleteProcessDefinition(processDefinitionId, cascade, skipCustomListeners, skipIoMappings);
    }

    /**
     * Fluent builder to delete process definitions.
     *
     * @return the builder to delete process definitions
     */
    @CronapiMetaData(
            name = "{{deleteProcessDefinitions}}",
            description = "{{deleteProcessDefinitionsDescription}}"
    )
    public static DeleteProcessDefinitionsSelectBuilder deleteProcessDefinitions() {
        return getRepositoryService().deleteProcessDefinitions();
    }

    /**
     * Retrieves a list of deployment resource names for the given deployment,
     * ordered alphabetically.
     *
     * @param deploymentId id of the deployment, cannot be null.
     * @return Lista
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#DEPLOYMENT}.
     */
    @CronapiMetaData(
            name = "{{getDeploymentResourceNames}}",
            description = "{{getDeploymentResourceNamesDescription}}"
    )
    public static List<String> getDeploymentResourceNames(String deploymentId) {
        return getRepositoryService().getDeploymentResourceNames(deploymentId);
    }

    /**
     * Retrieves a list of deployment resources for the given deployment,
     * ordered alphabetically by name.
     *
     * @param deploymentId id of the deployment, cannot be null.
     * @return Lista
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#DEPLOYMENT}.
     */
    @CronapiMetaData(
            name = "{{getDeploymentResources}}",
            description = "{{getDeploymentResourcesDescription}}"
    )
    public static List<Resource> getDeploymentResources(String deploymentId) {
        return getRepositoryService().getDeploymentResources(deploymentId);
    }

    /**
     * Gives access to a deployment resource through a stream of bytes.
     *
     * @param deploymentId id of the deployment, cannot be null.
     * @param resourceName name of the resource, cannot be null.
     * @return InputStream
     * @throws ProcessEngineException When the resource doesn't exist in the given deployment or when no deployment exists
     *                                for the given deploymentId.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#DEPLOYMENT}.
     */
    @CronapiMetaData(
            name = "{{getResourceAsStream}}",
            description = "{{getResourceAsStreamDescription}}"
    )
    public static InputStream getResourceAsStream(String deploymentId, String resourceName) {
        return getRepositoryService().getResourceAsStream(deploymentId, resourceName);
    }

    /**
     * Gives access to a deployment resource through a stream of bytes.
     *
     * @param deploymentId id of the deployment, cannot be null.
     * @param resourceId   id of the resource, cannot be null.
     * @return InputStream
     * @throws ProcessEngineException When the resource doesn't exist in the given deployment or when no deployment exists
     *                                for the given deploymentId.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#DEPLOYMENT}.
     */
    @CronapiMetaData(
            name = "{{getResourceAsStreamById}}",
            description = "{{getResourceAsStreamByIdDescription}}"
    )
    public static InputStream getResourceAsStreamById(String deploymentId, String resourceId) {
        return getRepositoryService().getResourceAsStreamById(deploymentId, resourceId);
    }

    /**
     * Query process definitions.
     * @return ProcessDefinitionQuery
     */
    @CronapiMetaData(
            name = "{{createProcessDefinitionQuery}}",
            description = "{{createProcessDefinitionQueryDescription}}"
    )
    public static ProcessDefinitionQuery createProcessDefinitionQuery() {
        return getRepositoryService().createProcessDefinitionQuery();
    }

    /**
     * Query case definitions.
     * @return CaseDefinitionQuery
     */
    @CronapiMetaData(
            name = "{{createCaseDefinitionQuery}}",
            description = "{{createCaseDefinitionQueryDescription}}"
    )
    public static CaseDefinitionQuery createCaseDefinitionQuery() {
        return getRepositoryService().createCaseDefinitionQuery();
    }

    /**
     * Query decision definitions.
     * @return DecisionDefinitionQuery
     */
    @CronapiMetaData(
            name = "{{createDecisionDefinitionQuery}}",
            description = "{{createDecisionDefinitionQueryDescription}}"
    )
    public static DecisionDefinitionQuery createDecisionDefinitionQuery() {
        return getRepositoryService().createDecisionDefinitionQuery();
    }

    /**
     * Query decision requirements definition.
     * @return DecisionRequirementsDefinitionQuery
     */
    @CronapiMetaData(
            name = "{{createDecisionRequirementsDefinitionQuery}}",
            description = "{{createDecisionRequirementsDefinitionQueryDescription}}"
    )
    public static DecisionRequirementsDefinitionQuery createDecisionRequirementsDefinitionQuery() {
        return getRepositoryService().createDecisionRequirementsDefinitionQuery();
    }

    /**
     * Query process definitions.
     * @return DeploymentQuery
     */
    @CronapiMetaData(
            name = "{{createDeploymentQuery}}",
            description = "{{createDeploymentQueryDescription}}"
    )
    public static DeploymentQuery createDeploymentQuery() {
        return getRepositoryService().createDeploymentQuery();
    }

    /**
     * Suspends the process definition with the given id.
     * <p>
     * If a process definition is in state suspended, it will not be possible to start new process instances
     * based on the process definition.
     *
     * <strong>Note: all the process instances of the process definition will still be active
     * (ie. not suspended)!</strong>
     *
     * <p>Note: for more complex suspend commands use {@link #updateProcessDefinitionSuspensionState()}.
     *
     * @param processDefinitionId String
     * @throws ProcessEngineException If no such processDefinition can be found.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{suspendProcessDefinitionById}}",
            description = "{{suspendProcessDefinitionByIdDescription}}"
    )
    public static void suspendProcessDefinitionById(String processDefinitionId) {
        getRepositoryService().suspendProcessDefinitionById(processDefinitionId);
    }

    /**
     * Suspends the process definition with the given id.
     * <p>
     * If a process definition is in state suspended, it will not be possible to start new process instances
     * based on the process definition.
     *
     * <p>Note: for more complex suspend commands use {@link #updateProcessDefinitionSuspensionState()}.
     *
     * @param processDefinitionId String
     * @param suspendProcessInstances If true, all the process instances of the provided process definition
     *                                will be suspended too.
     * @param suspensionDate          The date on which the process definition will be suspended. If null, the
     *                                process definition is suspended immediately.
     *                                Note: The job executor needs to be active to use this!
     * @throws ProcessEngineException If no such processDefinition can be found.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#PROCESS_DEFINITION}
     *                                and if <code>suspendProcessInstances</code> is set to <code>true</code> and the user have no
     *                                {@link Permissions#UPDATE} permission on {@link Resources#PROCESS_INSTANCE} or no
     *                                {@link Permissions#UPDATE_INSTANCE} permission on {@link Resources#PROCESS_DEFINITION}.
     * @see RuntimeService#suspendProcessInstanceById(String)
     */
    @CronapiMetaData(
            name = "{{suspendProcessDefinitionById}}",
            description = "{{suspendProcessDefinitionByIdDescription}}"
    )
    public static void suspendProcessDefinitionById(String processDefinitionId, boolean suspendProcessInstances, Date suspensionDate) {
        getRepositoryService().suspendProcessDefinitionById(processDefinitionId, suspendProcessInstances, suspensionDate);
    }

    /**
     * Suspends the <strong>all</strong> process definitions with the given key (= id in the bpmn20.xml file).
     * <p>
     * If a process definition is in state suspended, it will not be possible to start new process instances
     * based on the process definition.
     *
     * <strong>Note: all the process instances of the process definition will still be active
     * (ie. not suspended)!</strong>
     *
     * <p>Note: for more complex suspend commands use {@link #updateProcessDefinitionSuspensionState()}.
     *
     * @param processDefinitionKey String
     * @throws ProcessEngineException If no such processDefinition can be found.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{suspendProcessDefinitionByKey}}",
            description = "{{suspendProcessDefinitionByKeyDescription}}"
    )
    public static void suspendProcessDefinitionByKey(String processDefinitionKey) {
        getRepositoryService().suspendProcessDefinitionByKey(processDefinitionKey);
    }

    /**
     * Suspends the <strong>all</strong> process definitions with the given key (= id in the bpmn20.xml file).
     * <p>
     * If a process definition is in state suspended, it will not be possible to start new process instances
     * based on the process definition.
     *
     * <p>Note: for more complex suspend commands use {@link #updateProcessDefinitionSuspensionState()}.
     *
     * @param processDefinitionKey String
     * @param suspendProcessInstances If true, all the process instances of the provided process definition
     *                                will be suspended too.
     * @param suspensionDate          The date on which the process definition will be suspended. If null, the
     *                                process definition is suspended immediately.
     *                                Note: The job executor needs to be active to use this!
     * @throws ProcessEngineException If no such processDefinition can be found.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#PROCESS_DEFINITION}
     *                                and if <code>suspendProcessInstances</code> is set to <code>true</code> and the user have no
     *                                {@link Permissions#UPDATE} permission on {@link Resources#PROCESS_INSTANCE} or no
     *                                {@link Permissions#UPDATE_INSTANCE} permission on {@link Resources#PROCESS_DEFINITION}.
     * @see RuntimeService#suspendProcessInstanceById(String)
     */
    @CronapiMetaData(
            name = "{{suspendProcessDefinitionByKey}}",
            description = "{{suspendProcessDefinitionByKeyDescription}}"
    )
    public static void suspendProcessDefinitionByKey(String processDefinitionKey, boolean suspendProcessInstances, Date suspensionDate) {
        getRepositoryService().suspendProcessDefinitionByKey(processDefinitionKey, suspendProcessInstances, suspensionDate);
    }

    /**
     * Activates the process definition with the given id.
     *
     * <p>Note: for more complex activate commands use {@link #updateProcessDefinitionSuspensionState()}.
     *
     * @param processDefinitionId String
     * @throws ProcessEngineException If no such processDefinition can be found or if the process definition is already in state active.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{activateProcessDefinitionById}}",
            description = "{{activateProcessDefinitionByIdDescription}}"
    )
    public static void activateProcessDefinitionById(String processDefinitionId) {
        getRepositoryService().activateProcessDefinitionById(processDefinitionId);
    }

    /**
     * Activates the process definition with the given id.
     *
     * <p>Note: for more complex activate commands use {@link #updateProcessDefinitionSuspensionState()}.
     *
     * @param processDefinitionId String
     * @param activateProcessInstances boolean
     * @param activationDate           The date on which the process definition will be activated. If null, the
     *                                 process definition is suspended immediately.
     *                                 Note: The job executor needs to be active to use this!
     * @throws ProcessEngineException If no such processDefinition can be found.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#PROCESS_DEFINITION}
     *                                and if <code>activateProcessInstances</code> is set to <code>true</code> and the user have no
     *                                {@link Permissions#UPDATE} permission on {@link Resources#PROCESS_INSTANCE} or no
     *                                {@link Permissions#UPDATE_INSTANCE} permission on {@link Resources#PROCESS_DEFINITION}.
     * @see RuntimeService#activateProcessInstanceById(String)
     */
    @CronapiMetaData(
            name = "{{activateProcessDefinitionById}}",
            description = "{{activateProcessDefinitionByIdDescription}}"
    )
    public static void activateProcessDefinitionById(String processDefinitionId, boolean activateProcessInstances, Date activationDate) {
        getRepositoryService().activateProcessDefinitionById(processDefinitionId, activateProcessInstances, activationDate);
    }

    /**
     * Activates the process definition with the given key (=id in the bpmn20.xml file).
     *
     * <p>Note: for more complex activate commands use {@link #updateProcessDefinitionSuspensionState()}.
     *
     * @param processDefinitionKey String
     * @throws ProcessEngineException If no such processDefinition can be found.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{activateProcessDefinitionByKey}}",
            description = "{{activateProcessDefinitionByKeyDescription}}"
    )
    public static void activateProcessDefinitionByKey(String processDefinitionKey) {
        getRepositoryService().activateProcessDefinitionByKey(processDefinitionKey);
    }

    /**
     * Activates the process definition with the given key (=id in the bpmn20.xml file).
     *
     * <p>Note: for more complex activate commands use {@link #updateProcessDefinitionSuspensionState()}.
     *
     * @param processDefinitionKey String
     * @param activateProcessInstances boolean
     * @param activationDate           The date on which the process definition will be activated. If null, the
     *                                 process definition is suspended immediately.
     *                                 Note: The job executor needs to be active to use this!
     * @throws ProcessEngineException If no such processDefinition can be found.
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#PROCESS_DEFINITION}
     *                                and if <code>activateProcessInstances</code> is set to <code>true</code> and the user have no
     *                                {@link Permissions#UPDATE} permission on {@link Resources#PROCESS_INSTANCE} or no
     *                                {@link Permissions#UPDATE_INSTANCE} permission on {@link Resources#PROCESS_DEFINITION}.
     * @see RuntimeService#activateProcessInstanceById(String)
     */
    @CronapiMetaData(
            name = "{{activateProcessDefinitionByKey}}",
            description = "{{activateProcessDefinitionByKeyDescription}}"
    )
    public static void activateProcessDefinitionByKey(String processDefinitionKey, boolean activateProcessInstances, Date activationDate) {
        getRepositoryService().activateProcessDefinitionByKey(processDefinitionKey, activateProcessInstances, activationDate);
    }

    /**
     * Activate or suspend process definitions using a fluent builder. Specify the
     * definitions by calling one of the <i>by</i> methods, like
     * <i>byProcessDefinitionId</i>. To update the suspension state call
     * {@link UpdateProcessDefinitionSuspensionStateBuilder#activate()} or
     * {@link UpdateProcessDefinitionSuspensionStateBuilder#suspend()}.
     *
     * @return the builder to update the suspension state
     */
    @CronapiMetaData(
            name = "{{updateProcessDefinitionSuspensionState}}",
            description = "{{updateProcessDefinitionSuspensionStateDescription}}"
    )
    public static UpdateProcessDefinitionSuspensionStateSelectBuilder updateProcessDefinitionSuspensionState() {
        return getRepositoryService().updateProcessDefinitionSuspensionState();
    }

    /**
     * Updates time to live of process definition. The field is used within history cleanup process.
     *
     * @param processDefinitionId String
     * @param historyTimeToLive Integerv
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{updateProcessDefinitionHistoryTimeToLive}}",
            description = "{{updateProcessDefinitionHistoryTimeToLiveDescription}}"
    )
    public static void updateProcessDefinitionHistoryTimeToLive(String processDefinitionId, Integer historyTimeToLive) {
        getRepositoryService().updateCaseDefinitionHistoryTimeToLive(processDefinitionId, historyTimeToLive);
    }

    /**
     * Updates time to live of decision definition. The field is used within history cleanup process.
     *
     * @param decisionDefinitionId String
     * @param historyTimeToLive Integer
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#DECISION_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{updateDecisionDefinitionHistoryTimeToLive}}",
            description = "{{updateDecisionDefinitionHistoryTimeToLiveDescription}}"
    )
    public static void updateDecisionDefinitionHistoryTimeToLive(String decisionDefinitionId, Integer historyTimeToLive) {
        getRepositoryService().updateCaseDefinitionHistoryTimeToLive(decisionDefinitionId, historyTimeToLive);
    }

    /**
     * Updates time to live of case definition. The field is used within history cleanup process.
     *
     * @param caseDefinitionId String
     * @param historyTimeToLive Integer
     */
    @CronapiMetaData(
            name = "{{updateCaseDefinitionHistoryTimeToLive}}",
            description = "{{updateCaseDefinitionHistoryTimeToLiveDescription}}"
    )
    public static void updateCaseDefinitionHistoryTimeToLive(String caseDefinitionId, Integer historyTimeToLive) {
        getRepositoryService().updateCaseDefinitionHistoryTimeToLive(caseDefinitionId, historyTimeToLive);
    }

    /**
     * Gives access to a deployed process model, e.g., a BPMN 2.0 XML file,
     * through a stream of bytes.
     *
     * @param processDefinitionId id of a {@link ProcessDefinition}, cannot be null.
     * @return InputStream
     * @throws ProcessEngineException when the process model doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getProcessModel}}",
            description = "{{getProcessModelDescription}}"
    )
    public static InputStream getProcessModel(String processDefinitionId) {
        return getRepositoryService().getProcessModel(processDefinitionId);
    }

    /**
     * Gives access to a deployed process diagram, e.g., a PNG image, through a
     * stream of bytes.
     *
     * @param processDefinitionId id of a {@link ProcessDefinition}, cannot be null.
     * @return null when the diagram resource name of a {@link ProcessDefinition} is null.
     * @throws ProcessEngineException when the process diagram doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getProcessDiagram}}",
            description = "{{getProcessDiagramDescription}}"
    )
    public static InputStream getProcessDiagram(String processDefinitionId) {
        return getRepositoryService().getProcessDiagram(processDefinitionId);
    }

    /**
     * Returns the {@link ProcessDefinition} including all BPMN information like additional
     * Properties (e.g. documentation).
     *
     * @param processDefinitionId String
     * @return ProcessDefinition
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getProcessDefinition}}",
            description = "{{getProcessDefinitionDescription}}"
    )
    public static ProcessDefinition getProcessDefinition(String processDefinitionId) {
        return getRepositoryService().getProcessDefinition(processDefinitionId);
    }

    /**
     * Provides positions and dimensions of elements in a process diagram as
     * provided by {@link RepositoryService#getProcessDiagram(String)}.
     * <p>
     * This method requires a process model and a diagram image to be deployed.
     *
     * @param processDefinitionId id of a {@link ProcessDefinition}, cannot be null.
     * @return null when the input stream of a process diagram is null.
     * @throws ProcessEngineException When the process model or diagram doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getProcessDiagramLayout}}",
            description = "{{getProcessDiagramLayoutDescription}}"
    )
    public static DiagramLayout getProcessDiagramLayout(String processDefinitionId) {
        return getRepositoryService().getProcessDiagramLayout(processDefinitionId);
    }

    /**
     * Returns the {@link BpmnModelInstance} for the given processDefinitionId.
     *
     * @param processDefinitionId the id of the Process Definition for which the {@link BpmnModelInstance}
     *                            should be retrieved.
     * @return the {@link BpmnModelInstance}
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getBpmnModelInstance}}",
            description = "{{getBpmnModelInstanceDescription}}"
    )
    public static BpmnModelInstance getBpmnModelInstance(String processDefinitionId) {
        return getRepositoryService().getBpmnModelInstance(processDefinitionId);
    }

    /**
     * Returns the {@link CmmnModelInstance} for the given caseDefinitionId.
     *
     * @param caseDefinitionId the id of the Case Definition for which the {@link CmmnModelInstance}
     *                         should be retrieved.
     * @return the {@link CmmnModelInstance}
     * @throws NotValidException      when the given case definition id or deployment id or resource name is null
     * @throws NotFoundException      when no CMMN model instance or deployment resource is found for the given
     *                                case definition id
     * @throws ProcessEngineException when an internal exception happens during the execution
     *                                of the command.
     */
    @CronapiMetaData(
            name = "{{getCmmnModelInstance}}",
            description = "{{getCmmnModelInstanceDescription}}"
    )
    public static CmmnModelInstance getCmmnModelInstance(String caseDefinitionId) {
        return getRepositoryService().getCmmnModelInstance(caseDefinitionId);
    }

    /**
     * Returns the {@link DmnModelInstance} for the given decisionDefinitionId.
     *
     * @param decisionDefinitionId the id of the Decision Definition for which the {@link DmnModelInstance}
     *                             should be retrieved.
     * @return the {@link DmnModelInstance}
     * @throws NotValidException      when the given decision definition id or deployment id or resource name is null
     * @throws NotFoundException      when no DMN model instance or deployment resource is found for the given
     *                                decision definition id
     * @throws ProcessEngineException when an internal exception happens during the execution of the command.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#DECISION_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getDmnModelInstance}}",
            description = "{{getDmnModelInstanceDescription}}"
    )
    public static DmnModelInstance getDmnModelInstance(String decisionDefinitionId) {
        return getRepositoryService().getDmnModelInstance(decisionDefinitionId);
    }

    /**
     * Authorizes a candidate user for a process definition.
     *
     * @param processDefinitionId id of the process definition, cannot be null.
     * @param userId              id of the user involve, cannot be null.
     * @throws ProcessEngineException When the process definition or user doesn't exist.
     * @deprecated Use authorization mechanism instead.
     */
    @CronapiMetaData(
            name = "{{addCandidateStarterUser}}",
            description = "{{addCandidateStarterUserDescription}}"
    )
    public static void addCandidateStarterUser(String processDefinitionId, String userId) {
        getRepositoryService().addCandidateStarterUser(processDefinitionId, userId);
    }

    /**
     * Authorizes a candidate group for a process definition.
     *
     * @param processDefinitionId id of the process definition, cannot be null.
     * @param groupId             id of the group involve, cannot be null.
     * @throws ProcessEngineException When the process definition or group doesn't exist.
     * @deprecated Use authorization mechanism instead.
     */
    @CronapiMetaData(
            name = "{{addCandidateStarterGroup}}",
            description = "{{addCandidateStarterGroupDescription}}"
    )
    public static void addCandidateStarterGroup(String processDefinitionId, String groupId) {
        getRepositoryService().addCandidateStarterGroup(processDefinitionId, groupId);
    }

    /**
     * Removes the authorization of a candidate user for a process definition.
     *
     * @param processDefinitionId id of the process definition, cannot be null.
     * @param userId              id of the user involve, cannot be null.
     * @throws ProcessEngineException When the process definition or user doesn't exist.
     * @deprecated Use authorization mechanism instead.
     */
    @CronapiMetaData(
            name = "{{deleteCandidateStarterUser}}",
            description = "{{deleteCandidateStarterUserDescription}}"
    )
    public static void deleteCandidateStarterUser(String processDefinitionId, String userId) {
        getRepositoryService().deleteCandidateStarterUser(processDefinitionId, userId);
    }

    /**
     * Removes the authorization of a candidate group for a process definition.
     *
     * @param processDefinitionId id of the process definition, cannot be null.
     * @param groupId             id of the group involve, cannot be null.
     * @throws ProcessEngineException When the process definition or group doesn't exist.
     * @deprecated Use authorization mechanism instead.
     */
    @CronapiMetaData(
            name = "{{deleteCandidateStarterGroup}}",
            description = "{{deleteCandidateStarterGroupDescription}}"
    )
    public static void deleteCandidateStarterGroup(String processDefinitionId, String groupId) {
        getRepositoryService().deleteCandidateStarterUser(processDefinitionId, groupId);
    }

    /**
     * Retrieves the {@link IdentityLink}s associated with the given process definition.
     * Such an {@link IdentityLink} informs how a certain identity (eg. group or user)
     * is authorized for a certain process definition
     *
     * @param processDefinitionId String
     * @return Lista
     * @deprecated Use authorization mechanism instead.
     */
    @CronapiMetaData(
            name = "{{getIdentityLinksForProcessDefinition}}",
            description = "{{getIdentityLinksForProcessDefinitionDescription}}"
    )
    public static List<IdentityLink> getIdentityLinksForProcessDefinition(String processDefinitionId) {
        return getRepositoryService().getIdentityLinksForProcessDefinition(processDefinitionId);
    }

    /**
     * Returns the {@link CaseDefinition}.
     *
     * @param caseDefinitionId String
     * @return CaseDefinition
     * @throws NotValidException      when the given case definition id is null
     * @throws NotFoundException      when no case definition is found for the given case definition id
     * @throws ProcessEngineException when an internal exception happens during the execution
     *                                of the command.
     */
    @CronapiMetaData(
            name = "{{getCaseDefinition}}",
            description = "{{getCaseDefinitionDescription}}"
    )
    public static CaseDefinition getCaseDefinition(String caseDefinitionId) {
        return getRepositoryService().getCaseDefinition(caseDefinitionId);
    }

    /**
     * Gives access to a deployed case model, e.g., a CMMN 1.0 XML file,
     * through a stream of bytes.
     *
     * @param caseDefinitionId id of a {@link CaseDefinition}, cannot be null.
     * @return InputStream
     * @throws NotValidException      when the given case definition id or deployment id or resource name is null
     * @throws NotFoundException      when no case definition or deployment resource is found for the given case definition id
     * @throws ProcessEngineException when an internal exception happens during the execution of the command
     */
    @CronapiMetaData(
            name = "{{getCaseModel}}",
            description = "{{getCaseModelDescription}}"
    )
    public static InputStream getCaseModel(String caseDefinitionId) {
        return getRepositoryService().getCaseModel(caseDefinitionId);
    }

    /**
     * Gives access to a deployed case diagram, e.g., a PNG image, through a
     * stream of bytes.
     *
     * @param caseDefinitionId id of a {@link CaseDefinition}, cannot be null.
     * @return null when the diagram resource name of a {@link CaseDefinition} is null.
     * @throws ProcessEngineException when the process diagram doesn't exist.
     */
    @CronapiMetaData(
            name = "{{getCaseDiagram}}",
            description = "{{getCaseDiagramDescription}}"
    )
    public static InputStream getCaseDiagram(String caseDefinitionId) {
        return getRepositoryService().getCaseDiagram(caseDefinitionId);
    }

    /**
     * Returns the {@link DecisionDefinition}.
     *
     * @param decisionDefinitionId String
     * @return DecisionDefinition
     * @throws NotValidException      when the given decision definition id is null
     * @throws NotFoundException      when no decision definition is found for the given decision definition id
     * @throws ProcessEngineException when an internal exception happens during the execution of the command.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#DECISION_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getDecisionDefinition}}",
            description = "{{getDecisionDefinitionDescription}}"
    )
    public static DecisionDefinition getDecisionDefinition(String decisionDefinitionId) {
        return getRepositoryService().getDecisionDefinition(decisionDefinitionId);
    }

    /**
     * Returns the {@link DecisionRequirementsDefinition}.
     *
     * @param decisionRequirementsDefinitionId String
     * @return DecisionRequirementsDefinition
     * @throws NotValidException      when the given decision requirements definition id is null
     * @throws NotFoundException      when no decision requirements definition is found for the given decision requirements definition id
     * @throws ProcessEngineException when an internal exception happens during the execution of the command.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#DECISION_REQUIREMENTS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getDecisionRequirementsDefinition}}",
            description = "{{getDecisionRequirementsDefinitionDescription}}"
    )
    public static DecisionRequirementsDefinition getDecisionRequirementsDefinition(String decisionRequirementsDefinitionId) {
        return getRepositoryService().getDecisionRequirementsDefinition(decisionRequirementsDefinitionId);
    }

    /**
     * Gives access to a deployed decision model, e.g., a DMN 1.1 XML file,
     * through a stream of bytes.
     *
     * @param decisionDefinitionId id of a {@link DecisionDefinition}, cannot be null.
     * @return InputStream
     * @throws NotValidException      when the given decision definition id or deployment id or resource name is null
     * @throws NotFoundException      when no decision definition or deployment resource is found for the given decision definition id
     * @throws ProcessEngineException when an internal exception happens during the execution of the command
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#DECISION_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getDecisionModel}}",
            description = "{{getDecisionModelDescription}}"
    )
    public static InputStream getDecisionModel(String decisionDefinitionId) {
        return getRepositoryService().getDecisionModel(decisionDefinitionId);
    }

    /**
     * Gives access to a deployed decision requirements model, e.g., a DMN 1.1 XML file,
     * through a stream of bytes.
     *
     * @param decisionRequirementsDefinitionId id of a {@link DecisionRequirementsDefinition}, cannot be null.
     * @return InputStream
     * @throws NotValidException      when the given decision requirements definition id or deployment id or resource name is null
     * @throws NotFoundException      when no decision requirements definition or deployment resource is found for the given decision requirements definition id
     * @throws ProcessEngineException when an internal exception happens during the execution of the command
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#DECISION_REQUIREMENTS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getDecisionRequirementsModel}}",
            description = "{{getDecisionRequirementsModelDescription}}"
    )
    public static InputStream getDecisionRequirementsModel(String decisionRequirementsDefinitionId) {
        return getRepositoryService().getDecisionModel(decisionRequirementsDefinitionId);
    }

    /**
     * Gives access to a deployed decision diagram, e.g., a PNG image, through a
     * stream of bytes.
     *
     * @param decisionDefinitionId id of a {@link DecisionDefinition}, cannot be null.
     * @return null when the diagram resource name of a {@link DecisionDefinition} is null.
     * @throws ProcessEngineException when the decision diagram doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#DECISION_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getDecisionDiagram}}",
            description = "{{getDecisionDiagramDescription}}"
    )
    public static InputStream getDecisionDiagram(String decisionDefinitionId) {
        return getRepositoryService().getDecisionDiagram(decisionDefinitionId);
    }

    /**
     * Gives access to a deployed decision requirements diagram, e.g., a PNG image, through a
     * stream of bytes.
     *
     * @param decisionRequirementsDefinitionId id of a {@link DecisionRequirementsDefinition}, cannot be null.
     * @return null when the diagram resource name of a {@link DecisionRequirementsDefinition} is null.
     * @throws ProcessEngineException when the decision requirements diagram doesn't exist.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#DECISION_REQUIREMENTS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getDecisionRequirementsDiagram}}",
            description = "{{getDecisionRequirementsDiagramDescription}}"
    )
    public static InputStream getDecisionRequirementsDiagram(String decisionRequirementsDefinitionId) {
        return getRepositoryService().getDecisionRequirementsDiagram(decisionRequirementsDefinitionId);
    }
}
