package io.dangernoodle.github.repo.setup;

import java.io.IOException;
import java.util.Map;
import java.util.Set;

import org.kohsuke.github.GHLabel;
import org.kohsuke.github.GHRepository;

import io.dangernoodle.github.GithubClient;
import io.dangernoodle.github.repo.setup.settings.GithubRepositorySettings;
import io.dangernoodle.github.repo.setup.settings.GithubRepositorySettings.Color;
import io.dangernoodle.github.utils.GithubUtilities;
import io.dangernoodle.scm.ScmException;
import io.dangernoodle.scm.setup.RepositorySetupException;


public class GithubCreateLabels extends GithubRepositorySetupStep
{
    public GithubCreateLabels(GithubClient client)
    {
        super(client);
    }

    @Override
    protected void execute(String organization, String repository, GithubRepositorySettings settings) throws ScmException
    {
        GHRepository ghRepository = GithubUtilities.getRepository(client, organization, repository);

        Map<String, Color> labels = settings.getLabels();
        Set<GHLabel> existing = loadExistingLabels(ghRepository);

        for (String name : labels.keySet())
        {
            GHLabel label = findLabel(name, existing);
            String color = labels.get(name).toString();

            if (label == null)
            {
                logger.debug("creating label [{}] color [{}]", name, color);
                createLabel(name, color, ghRepository);
            }
            else if (!label.getColor().equals(color))
            {
                logger.warn("label [{}] color [{}] does not match [{}]", name, color, label.getColor());
            }
        }
    }

    private void createLabel(String label, String color, GHRepository repository) throws RepositorySetupException
    {
        try
        {
            repository.createLabel(label, color);
        }
        catch (IOException e)
        {
            logger.error("failed to create label [{}] for repository [{}]", label, repository.getName(), e);
            throw new RepositorySetupException(e, "failed to create label [%s] for repository [%s]", label, repository.getName());
        }
    }

    private GHLabel findLabel(String name, Set<GHLabel> existing)
    {
        return existing.stream()
                       .filter(label -> label.getName().equals(name))
                       .findFirst()
                       .orElse(null);
    }

    private Set<GHLabel> loadExistingLabels(GHRepository repository) throws RepositorySetupException
    {
        try
        {
            return repository.listLabels().asSet();
        }
        catch (IOException e)
        {
            logger.error("failed to load existing labels for [{}]", repository.getName(), e);
            throw new RepositorySetupException(e, "failed to load existing labels for [%s]", repository.getName());
        }
    }
}
