package io.dangernoodle.github.repo.setup;

import java.io.IOException;
import java.util.Collection;
import java.util.Map;

import org.kohsuke.github.GHBranch;
import org.kohsuke.github.GHBranchProtectionBuilder;
import org.kohsuke.github.GHRepository;

import io.dangernoodle.github.GithubClient;
import io.dangernoodle.github.repo.setup.settings.GithubRepositorySettings;
import io.dangernoodle.github.repo.setup.settings.GithubRepositorySettings.Protection;
import io.dangernoodle.github.repo.setup.settings.GithubRepositorySettings.Protection.PushAccess;
import io.dangernoodle.github.utils.GithubUtilities;
import io.dangernoodle.scm.ScmException;
import io.dangernoodle.scm.setup.RepositorySetupException;


public class GithubEnableBranchProtection extends GithubRepositorySetupStep
{
    public GithubEnableBranchProtection(GithubClient client)
    {
        super(client);
    }

    @Override
    protected void execute(String organization, String repository, GithubRepositorySettings settings) throws ScmException
    {
        GHRepository ghRepository = GithubUtilities.getRepository(client, organization, repository);

        for (Map.Entry<String, Protection> entry : settings.getProtections().entrySet())
        {
            String branch = entry.getKey();
            GHBranch ghBranch = getBranch(branch, ghRepository);

            logger.debug("enabling 'branch protections' for branch [{}]", branch);
            enableProtection(branch, organization, repository, entry.getValue(), ghBranch.enableProtection());
        }
    }

    private void enableProtection(String branch, String organization, String repository, Protection protection,
            GHBranchProtectionBuilder builder)
        throws RepositorySetupException
    {
        try
        {
            enableReviews(branch, protection, builder);
            enableStatusChecks(branch, protection, builder);
            restrictPushAccess(branch, organization, protection, builder);

            builder.enable();
        }
        catch (IOException e)
        {
            logger.error("failed to enable protections on branch [{}] in repository [{}]", branch, repository, e);
            throw new RepositorySetupException(e, "failed to retrieve branch [%s] from repository [%s]", branch, repository);
        }
    }

    private void enableReviews(String branch, Protection protection, GHBranchProtectionBuilder builder)
    {
        if (protection.isRequireReviews())
        {
            logger.trace("requiring reviews for branch [{}]", branch);
            builder.requireReviews();

            builder.dismissStaleReviews();
            
//            if (protection.isRequireAdminReviews())
//            {
//                logger.trace("enabling 'reviews required for admins' for branch [{}]", branch);
//                builder.requireReviewsForAdmins();
//            }
        }
    }

    private void enableStatusChecks(String branch, Protection protection, GHBranchProtectionBuilder builder)
    {
        if (protection.isRequireUpToDate())
        {
            logger.trace("requiring branch [{}] must up-to-date", branch);
            builder.requireBranchIsUpToDate();
        }

        Collection<String> checks = protection.getRequiredStatusChecks();
        if (!checks.isEmpty())
        {
            logger.trace("requiring status check(s) [{}] for branch", branch, checks);
            builder.addRequiredChecks(checks);

//            if (protection.isRequireStatusChecksForAdmins())
//            {
//                logger.trace("enabling 'require status checks for admins' for branch [{}]", branch);
//                builder.requireStatusChecksForAdmins();
//            }
        }
    }

    private GHBranch getBranch(String branch, GHRepository repository) throws RepositorySetupException
    {
        // TODO: create branch if it doesn't exist...
        try
        {
            return repository.getBranch(branch);
        }
        catch (IOException e)
        {
            logger.error("failed to retrieve branch [{}] from repository [{}]", branch, repository.getName(), e);
            throw new RepositorySetupException(e, "failed to retrieve branch [%s] from repository [%s]", branch, repository.getName());
        }
    }

    private void restrictPushAccess(String branch, String organization, Protection protection, GHBranchProtectionBuilder builder)
        throws IOException
    {
        PushAccess pushAccess = protection.getPushAccess();
        if (pushAccess != null)
        {
            builder.restrictPushAccess();
            logger.trace("enabling push access restrictions against branch [{}]", branch);

            for (String team : pushAccess.getTeams())
            {
                logger.trace("enabling push access for team [{}]", team);
                builder.teamPushAccess(client.getTeam(organization, team));
            }

            for (String user : pushAccess.getUsers())
            {
                logger.trace("enabling push access for user [{}]", user);
                builder.userPushAccess(client.getUser(user));
            }
        }
    }
}
