package io.dangernoodle.github.repo.setup;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.dangernoodle.Project;
import io.dangernoodle.github.GithubClient;
import io.dangernoodle.github.repo.setup.settings.GithubRepositorySettings;
import io.dangernoodle.github.repo.setup.settings.GithubRepositorySettings.Color;
import io.dangernoodle.github.repo.setup.settings.GithubRepositorySettings.Permission;
import io.dangernoodle.github.repo.setup.settings.GithubRepositorySettings.Protection;
import io.dangernoodle.scm.ScmException;
import io.dangernoodle.scm.setup.RepositorySetupWorkflow;


public class GithubRepositoryWorkflow implements RepositorySetupWorkflow
{
    private static final Logger logger = LoggerFactory.getLogger(GithubRepositoryWorkflow.class);

    private final GithubClient client;

    public GithubRepositoryWorkflow(GithubClient client)
    {
        this.client = client;
    }

    @Override
    public void execute(Project project) throws ScmException
    {
        GithubRepositorySettings settings = project.getRepositorySettings();

        if (settings == null)
        {
            logger.warn("no repositoru settings found for [{}], aboring workflow!", project.getName());
            return;
        }

        Collection<GithubRepositorySetupStep> steps = new ArrayList<>();
        steps.add(new GithubFindOrCreateRepository(client));

        addCreateLabels(settings.getLabels(), steps);
        addTeamAccess(settings.getTeams(), steps);
        addEnableProtections(settings.getProtections(), steps);

        for (GithubRepositorySetupStep step : steps)
        {
            logger.trace("executing step [{}]", step.getClass().getName());
            step.execute(project);
        }
    }

    private void addTeamAccess(Map<String, Permission> teams, Collection<GithubRepositorySetupStep> steps)
    {
        if (!teams.isEmpty())
        {
            steps.add(new GithubAssignTeams(client));
        }
    }

    private void addEnableProtections(Map<String, Protection> protections, Collection<GithubRepositorySetupStep> steps)
    {
        if (protections != null)
        {
            steps.add(new GithubEnableBranchProtection(client));
        }
    }

    private void addCreateLabels(Map<String, Color> labels, Collection<GithubRepositorySetupStep> steps)
    {
        if (!labels.isEmpty())
        {
            steps.add(new GithubCreateLabels(client));
        }
    }
}
