/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.http.policy;

import com.azure.core.http.HttpHeader;
import com.azure.core.http.HttpHeaders;
import com.azure.core.http.HttpPipelineCallContext;
import com.azure.core.http.HttpPipelineNextPolicy;
import com.azure.core.http.HttpRequest;
import com.azure.core.http.HttpResponse;
import com.azure.core.http.policy.HttpLogDetailLevel;
import com.azure.core.http.policy.HttpLogOptions;
import com.azure.core.http.policy.HttpPipelinePolicy;
import com.azure.core.http.policy.HttpRequestLogger;
import com.azure.core.http.policy.HttpRequestLoggingContext;
import com.azure.core.http.policy.HttpResponseLogger;
import com.azure.core.http.policy.HttpResponseLoggingContext;
import com.azure.core.implementation.AccessibleByteArrayOutputStream;
import com.azure.core.implementation.ImplUtils;
import com.azure.core.implementation.http.HttpPipelineCallContextHelper;
import com.azure.core.implementation.jackson.ObjectMapperShim;
import com.azure.core.util.Context;
import com.azure.core.util.CoreUtils;
import com.azure.core.util.FluxUtil;
import com.azure.core.util.UrlBuilder;
import com.azure.core.util.logging.ClientLogger;
import com.azure.core.util.logging.LogLevel;
import com.fasterxml.jackson.databind.JsonNode;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Collections;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public class HttpLoggingPolicy
implements HttpPipelinePolicy {
    private static final ObjectMapperShim PRETTY_PRINTER = ObjectMapperShim.createPrettyPrintMapper();
    private static final int MAX_BODY_LOG_SIZE = 16384;
    private static final String REDACTED_PLACEHOLDER = "REDACTED";
    private static final int LOGGER_CACHE_MAX_SIZE = 1000;
    private static final Map<String, ClientLogger> CALLER_METHOD_LOGGER_CACHE = new ConcurrentHashMap<String, ClientLogger>();
    private static final ClientLogger LOGGER = new ClientLogger(HttpLoggingPolicy.class);
    private final HttpLogDetailLevel httpLogDetailLevel;
    private final Set<String> allowedHeaderNames;
    private final Set<String> allowedQueryParameterNames;
    private final boolean prettyPrintBody;
    private final HttpRequestLogger requestLogger;
    private final HttpResponseLogger responseLogger;
    public static final String RETRY_COUNT_CONTEXT = "requestRetryCount";

    public HttpLoggingPolicy(HttpLogOptions httpLogOptions) {
        if (httpLogOptions == null) {
            this.httpLogDetailLevel = HttpLogDetailLevel.NONE;
            this.allowedHeaderNames = Collections.emptySet();
            this.allowedQueryParameterNames = Collections.emptySet();
            this.prettyPrintBody = false;
            this.requestLogger = new DefaultHttpRequestLogger();
            this.responseLogger = new DefaultHttpResponseLogger();
        } else {
            this.httpLogDetailLevel = httpLogOptions.getLogLevel();
            this.allowedHeaderNames = httpLogOptions.getAllowedHeaderNames().stream().map(headerName -> headerName.toLowerCase(Locale.ROOT)).collect(Collectors.toSet());
            this.allowedQueryParameterNames = httpLogOptions.getAllowedQueryParamNames().stream().map(queryParamName -> queryParamName.toLowerCase(Locale.ROOT)).collect(Collectors.toSet());
            this.prettyPrintBody = httpLogOptions.isPrettyPrintBody();
            this.requestLogger = httpLogOptions.getRequestLogger() == null ? new DefaultHttpRequestLogger() : httpLogOptions.getRequestLogger();
            this.responseLogger = httpLogOptions.getResponseLogger() == null ? new DefaultHttpResponseLogger() : httpLogOptions.getResponseLogger();
        }
    }

    @Override
    public Mono<HttpResponse> process(HttpPipelineCallContext context, HttpPipelineNextPolicy next) {
        if (this.httpLogDetailLevel == HttpLogDetailLevel.NONE) {
            return next.process();
        }
        ClientLogger logger = HttpLoggingPolicy.getOrCreateMethodLogger((String)context.getData("caller-method").orElse(""));
        long startNs = System.nanoTime();
        return this.requestLogger.logRequest(logger, this.getRequestLoggingOptions(context)).then(next.process()).flatMap(response -> this.responseLogger.logResponse(logger, this.getResponseLoggingOptions((HttpResponse)response, startNs, context))).doOnError(throwable -> logger.warning("<-- HTTP FAILED: ", throwable));
    }

    private HttpRequestLoggingContext getRequestLoggingOptions(HttpPipelineCallContext callContext) {
        return new HttpRequestLoggingContext(callContext.getHttpRequest(), HttpPipelineCallContextHelper.getContext(callContext), HttpLoggingPolicy.getRequestRetryCount(HttpPipelineCallContextHelper.getContext(callContext)));
    }

    private HttpResponseLoggingContext getResponseLoggingOptions(HttpResponse httpResponse, long startNs, HttpPipelineCallContext callContext) {
        return new HttpResponseLoggingContext(httpResponse, Duration.ofNanos(System.nanoTime() - startNs), HttpPipelineCallContextHelper.getContext(callContext), HttpLoggingPolicy.getRequestRetryCount(HttpPipelineCallContextHelper.getContext(callContext)));
    }

    private static void logMessage(ClientLogger logger, LogLevel logLevel, StringBuilder logMessageBuilder) {
        switch (logLevel) {
            case VERBOSE: {
                logger.verbose(logMessageBuilder.toString());
                break;
            }
            case INFORMATIONAL: {
                logger.info(logMessageBuilder.toString());
                break;
            }
            case WARNING: {
                logger.warning(logMessageBuilder.toString());
                break;
            }
            case ERROR: {
                logger.error(logMessageBuilder.toString());
                break;
            }
        }
    }

    private static String getRedactedUrl(URL url, Set<String> allowedQueryParameterNames) {
        UrlBuilder builder = UrlBuilder.parse(url);
        String allowedQueryString = HttpLoggingPolicy.getAllowedQueryString(url.getQuery(), allowedQueryParameterNames);
        return builder.clearQuery().setQuery(allowedQueryString).toString();
    }

    private static String getAllowedQueryString(String queryString, Set<String> allowedQueryParameterNames) {
        String[] queryParams;
        if (CoreUtils.isNullOrEmpty(queryString)) {
            return "";
        }
        StringBuilder queryStringBuilder = new StringBuilder();
        for (String queryParam : queryParams = queryString.split("&")) {
            String[] queryPair;
            if (queryStringBuilder.length() > 0) {
                queryStringBuilder.append("&");
            }
            if ((queryPair = queryParam.split("=", 2)).length == 2) {
                String queryName = queryPair[0];
                if (allowedQueryParameterNames.contains(queryName.toLowerCase(Locale.ROOT))) {
                    queryStringBuilder.append(queryParam);
                    continue;
                }
                queryStringBuilder.append(queryPair[0]).append("=").append(REDACTED_PLACEHOLDER);
                continue;
            }
            queryStringBuilder.append(queryParam);
        }
        return queryStringBuilder.toString();
    }

    private static void addHeadersToLogMessage(Set<String> allowedHeaderNames, HttpHeaders headers, StringBuilder sb) {
        for (HttpHeader header : headers) {
            String headerName = header.getName();
            sb.append(headerName).append(":");
            if (allowedHeaderNames.contains(headerName.toLowerCase(Locale.ROOT))) {
                sb.append(header.getValue());
            } else {
                sb.append(REDACTED_PLACEHOLDER);
            }
            sb.append(System.lineSeparator());
        }
    }

    private static String prettyPrintIfNeeded(ClientLogger logger, boolean prettyPrintBody, String contentType, String body) {
        String result = body;
        if (prettyPrintBody && contentType != null && (contentType.startsWith("application/json") || contentType.startsWith("text/json"))) {
            try {
                JsonNode deserialized = PRETTY_PRINTER.readTree(body);
                result = PRETTY_PRINTER.writeValueAsString(deserialized);
            }
            catch (Exception e) {
                logger.warning("Failed to pretty print JSON: {}", e.getMessage());
            }
        }
        return result;
    }

    private static long getContentLength(ClientLogger logger, HttpHeaders headers) {
        long contentLength = 0L;
        String contentLengthString = headers.getValue("Content-Length");
        if (CoreUtils.isNullOrEmpty(contentLengthString)) {
            return contentLength;
        }
        try {
            contentLength = Long.parseLong(contentLengthString);
        }
        catch (NullPointerException | NumberFormatException e) {
            logger.warning("Could not parse the HTTP header content-length: '{}'.", headers.getValue("content-length"), e);
        }
        return contentLength;
    }

    private static boolean shouldBodyBeLogged(String contentTypeHeader, long contentLength) {
        return !"application/octet-stream".equalsIgnoreCase(contentTypeHeader) && contentLength != 0L && contentLength < 16384L;
    }

    private static Integer getRequestRetryCount(Context context) {
        Object rawRetryCount = context.getData(RETRY_COUNT_CONTEXT).orElse(null);
        if (rawRetryCount == null) {
            return null;
        }
        try {
            return Integer.valueOf(rawRetryCount.toString());
        }
        catch (NumberFormatException ex) {
            LOGGER.warning("Could not parse the request retry count: '{}'.", rawRetryCount);
            return null;
        }
    }

    private static ClientLogger getOrCreateMethodLogger(String methodName) {
        if (CALLER_METHOD_LOGGER_CACHE.size() > 1000) {
            CALLER_METHOD_LOGGER_CACHE.clear();
        }
        return CALLER_METHOD_LOGGER_CACHE.computeIfAbsent(methodName, ClientLogger::new);
    }

    private final class DefaultHttpRequestLogger
    implements HttpRequestLogger {
        private DefaultHttpRequestLogger() {
        }

        @Override
        public Mono<Void> logRequest(ClientLogger logger, HttpRequestLoggingContext loggingOptions) {
            long contentLength;
            LogLevel logLevel = this.getLogLevel(loggingOptions);
            if (!logger.canLogAtLevel(logLevel)) {
                return Mono.empty();
            }
            HttpRequest request = loggingOptions.getHttpRequest();
            StringBuilder requestLogMessage = new StringBuilder();
            if (HttpLoggingPolicy.this.httpLogDetailLevel.shouldLogUrl()) {
                requestLogMessage.append("--> ").append((Object)request.getHttpMethod()).append(" ").append(HttpLoggingPolicy.getRedactedUrl(request.getUrl(), HttpLoggingPolicy.this.allowedQueryParameterNames)).append(System.lineSeparator());
                Integer retryCount = loggingOptions.getTryCount();
                if (retryCount != null) {
                    requestLogMessage.append("Try count: ").append(retryCount).append(System.lineSeparator());
                }
            }
            if (HttpLoggingPolicy.this.httpLogDetailLevel.shouldLogHeaders() && logger.canLogAtLevel(LogLevel.VERBOSE)) {
                HttpLoggingPolicy.addHeadersToLogMessage(HttpLoggingPolicy.this.allowedHeaderNames, request.getHeaders(), requestLogMessage);
            }
            if (!HttpLoggingPolicy.this.httpLogDetailLevel.shouldLogBody()) {
                HttpLoggingPolicy.logMessage(logger, logLevel, requestLogMessage);
                return Mono.empty();
            }
            if (request.getBody() == null) {
                requestLogMessage.append("(empty body)").append(System.lineSeparator()).append("--> END ").append((Object)request.getHttpMethod()).append(System.lineSeparator());
                HttpLoggingPolicy.logMessage(logger, logLevel, requestLogMessage);
                return Mono.empty();
            }
            String contentType = request.getHeaders().getValue("Content-Type");
            if (HttpLoggingPolicy.shouldBodyBeLogged(contentType, contentLength = HttpLoggingPolicy.getContentLength(logger, request.getHeaders()))) {
                AccessibleByteArrayOutputStream stream = new AccessibleByteArrayOutputStream((int)contentLength);
                request.setBody((Flux<ByteBuffer>)request.getBody().doOnNext(byteBuffer -> {
                    try {
                        ImplUtils.writeByteBufferToStream(byteBuffer.duplicate(), stream);
                    }
                    catch (IOException ex) {
                        throw LOGGER.logExceptionAsError(new UncheckedIOException(ex));
                    }
                }).doFinally(ignored -> {
                    requestLogMessage.append(contentLength).append("-byte body:").append(System.lineSeparator()).append(HttpLoggingPolicy.prettyPrintIfNeeded(logger, HttpLoggingPolicy.this.prettyPrintBody, contentType, new String(stream.toByteArray(), 0, stream.count(), StandardCharsets.UTF_8))).append(System.lineSeparator()).append("--> END ").append((Object)request.getHttpMethod()).append(System.lineSeparator());
                    HttpLoggingPolicy.logMessage(logger, logLevel, requestLogMessage);
                }));
            } else {
                requestLogMessage.append(contentLength).append("-byte body: (content not logged)").append(System.lineSeparator()).append("--> END ").append((Object)request.getHttpMethod()).append(System.lineSeparator());
                HttpLoggingPolicy.logMessage(logger, logLevel, requestLogMessage);
            }
            return Mono.empty();
        }
    }

    private final class DefaultHttpResponseLogger
    implements HttpResponseLogger {
        private DefaultHttpResponseLogger() {
        }

        @Override
        public Mono<HttpResponse> logResponse(ClientLogger logger, HttpResponseLoggingContext loggingOptions) {
            long contentLength;
            LogLevel logLevel = this.getLogLevel(loggingOptions);
            HttpResponse response = loggingOptions.getHttpResponse();
            if (!logger.canLogAtLevel(logLevel)) {
                return Mono.just((Object)response);
            }
            String contentLengthString = response.getHeaderValue("Content-Length");
            String bodySize = CoreUtils.isNullOrEmpty(contentLengthString) ? "unknown-length body" : contentLengthString + "-byte body";
            StringBuilder responseLogMessage = new StringBuilder();
            if (HttpLoggingPolicy.this.httpLogDetailLevel.shouldLogUrl()) {
                responseLogMessage.append("<-- ").append(response.getStatusCode()).append(" ").append(HttpLoggingPolicy.getRedactedUrl(response.getRequest().getUrl(), HttpLoggingPolicy.this.allowedQueryParameterNames)).append(" (").append(loggingOptions.getResponseDuration().toMillis()).append(" ms, ").append(bodySize).append(")").append(System.lineSeparator());
            }
            if (HttpLoggingPolicy.this.httpLogDetailLevel.shouldLogHeaders() && logger.canLogAtLevel(LogLevel.VERBOSE)) {
                HttpLoggingPolicy.addHeadersToLogMessage(HttpLoggingPolicy.this.allowedHeaderNames, response.getHeaders(), responseLogMessage);
            }
            if (!HttpLoggingPolicy.this.httpLogDetailLevel.shouldLogBody()) {
                responseLogMessage.append("<-- END HTTP");
                HttpLoggingPolicy.logMessage(logger, logLevel, responseLogMessage);
                return Mono.justOrEmpty((Object)response);
            }
            String contentTypeHeader = response.getHeaderValue("Content-Type");
            if (HttpLoggingPolicy.shouldBodyBeLogged(contentTypeHeader, contentLength = HttpLoggingPolicy.getContentLength(logger, response.getHeaders()))) {
                return Mono.just((Object)new LoggingHttpResponse(response, responseLogMessage, logger, logLevel, (int)contentLength, contentTypeHeader, HttpLoggingPolicy.this.prettyPrintBody));
            }
            responseLogMessage.append("(body content not logged)").append(System.lineSeparator()).append("<-- END HTTP");
            HttpLoggingPolicy.logMessage(logger, logLevel, responseLogMessage);
            return Mono.just((Object)response);
        }
    }

    private static final class LoggingHttpResponse
    extends HttpResponse {
        private final HttpResponse actualResponse;
        private final StringBuilder responseLogMessage;
        private final int contentLength;
        private final ClientLogger logger;
        private final boolean prettyPrintBody;
        private final String contentTypeHeader;
        private final LogLevel logLevel;

        private LoggingHttpResponse(HttpResponse actualResponse, StringBuilder responseLogMessage, ClientLogger logger, LogLevel logLevel, int contentLength, String contentTypeHeader, boolean prettyPrintBody) {
            super(actualResponse.getRequest());
            this.actualResponse = actualResponse;
            this.responseLogMessage = responseLogMessage;
            this.logger = logger;
            this.logLevel = logLevel;
            this.contentLength = contentLength;
            this.contentTypeHeader = contentTypeHeader;
            this.prettyPrintBody = prettyPrintBody;
        }

        @Override
        public int getStatusCode() {
            return this.actualResponse.getStatusCode();
        }

        @Override
        public String getHeaderValue(String name) {
            return this.actualResponse.getHeaderValue(name);
        }

        @Override
        public HttpHeaders getHeaders() {
            return this.actualResponse.getHeaders();
        }

        @Override
        public Flux<ByteBuffer> getBody() {
            AccessibleByteArrayOutputStream stream = new AccessibleByteArrayOutputStream(this.contentLength);
            return this.actualResponse.getBody().doOnNext(byteBuffer -> {
                try {
                    ImplUtils.writeByteBufferToStream(byteBuffer.duplicate(), stream);
                }
                catch (IOException ex) {
                    throw LOGGER.logExceptionAsError(new UncheckedIOException(ex));
                }
            }).doFinally(ignored -> {
                this.responseLogMessage.append("Response body:").append(System.lineSeparator()).append(HttpLoggingPolicy.prettyPrintIfNeeded(this.logger, this.prettyPrintBody, this.contentTypeHeader, new String(stream.toByteArray(), 0, stream.count(), StandardCharsets.UTF_8))).append(System.lineSeparator()).append("<-- END HTTP");
                HttpLoggingPolicy.logMessage(this.logger, this.logLevel, this.responseLogMessage);
            });
        }

        @Override
        public Mono<byte[]> getBodyAsByteArray() {
            return FluxUtil.collectBytesFromNetworkResponse(this.getBody(), this.actualResponse.getHeaders());
        }

        @Override
        public Mono<String> getBodyAsString() {
            return this.getBodyAsByteArray().map(String::new);
        }

        @Override
        public Mono<String> getBodyAsString(Charset charset) {
            return this.getBodyAsByteArray().map(bytes -> new String((byte[])bytes, charset));
        }
    }
}

