/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.server.http.jwt;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.debezium.DebeziumException;
import io.debezium.server.http.Authenticator;
import io.debezium.server.http.jwt.JWTAuthorizationInitialRequest;
import io.debezium.server.http.jwt.JWTAuthorizationRefreshRequest;
import io.debezium.server.http.jwt.JWTAuthorizationResponse;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JWTAuthenticator
implements Authenticator {
    private static final Logger LOGGER = LoggerFactory.getLogger(JWTAuthenticator.class);
    private static final double EXPIRATION_DURATION_MULTIPLIER = 0.9;
    private final String username;
    private final String password;
    private final long tokenExpirationDuration;
    private final long refreshTokenExpirationDuration;
    private String jwtToken;
    private String jwtRefreshToken;
    private final HttpClient client;
    private final HttpRequest.Builder authRequestBuilder;
    private final HttpRequest.Builder refreshRequestBuilder;
    private final ObjectMapper mapper;
    private AuthenticationState authenticationState;
    private DateTime expirationDateTime;

    JWTAuthenticator(URI authUri, URI refreshUri, String username, String password, long tokenExpirationDuration, long refreshTokenExpirationDuration, Duration httpTimeoutDuration) {
        this.username = username;
        this.password = password;
        this.tokenExpirationDuration = tokenExpirationDuration;
        this.refreshTokenExpirationDuration = refreshTokenExpirationDuration;
        this.mapper = new ObjectMapper();
        this.client = HttpClient.newHttpClient();
        this.authRequestBuilder = HttpRequest.newBuilder(authUri).timeout(httpTimeoutDuration);
        this.authRequestBuilder.setHeader("content-type", "application/json");
        this.refreshRequestBuilder = HttpRequest.newBuilder(refreshUri).timeout(httpTimeoutDuration);
        this.refreshRequestBuilder.setHeader("content-type", "application/json");
        this.authenticationState = AuthenticationState.NOT_AUTHENTICATED;
        this.expirationDateTime = DateTime.now().minusDays(1);
    }

    void setAuthenticationState(AuthenticationState state) {
        this.authenticationState = state;
    }

    void setJwtToken(String token) {
        this.jwtToken = token;
    }

    void setJwtRefreshToken(String token) {
        this.jwtRefreshToken = token;
    }

    HttpRequest generateInitialAuthenticationRequest() {
        JWTAuthorizationInitialRequest payload = new JWTAuthorizationInitialRequest(this.username, this.password, this.tokenExpirationDuration, this.refreshTokenExpirationDuration);
        StringWriter payloadWriter = new StringWriter();
        try {
            this.mapper.writeValue((Writer)payloadWriter, (Object)payload);
        }
        catch (IOException e) {
            throw new DebeziumException("Could not serialize JWTAuthorizationRequest object to JSON.", (Throwable)e);
        }
        String payloadJSON = payloadWriter.toString();
        HttpRequest.Builder builder = this.authRequestBuilder.POST(HttpRequest.BodyPublishers.ofString(payloadJSON));
        return builder.build();
    }

    private void checkAuthenticationExpired() {
        if (this.authenticationState == AuthenticationState.ACTIVE && this.expirationDateTime.isBeforeNow()) {
            this.authenticationState = AuthenticationState.EXPIRED;
        }
    }

    HttpRequest generateRefreshAuthenticationRequest() {
        this.checkAuthenticationExpired();
        if (this.authenticationState == AuthenticationState.NOT_AUTHENTICATED || this.authenticationState == AuthenticationState.FAILED_AUTHENTICATION) {
            throw new DebeziumException("Must perform initial authentication successfully before attempting to refresh authentication");
        }
        JWTAuthorizationRefreshRequest payload = new JWTAuthorizationRefreshRequest(this.jwtRefreshToken, this.tokenExpirationDuration, this.refreshTokenExpirationDuration);
        StringWriter payloadWriter = new StringWriter();
        try {
            this.mapper.writeValue((Writer)payloadWriter, (Object)payload);
        }
        catch (IOException e) {
            throw new DebeziumException("Could not serialize JWTAuthorizationRequest object to JSON.", (Throwable)e);
        }
        String payloadJSON = payloadWriter.toString();
        HttpRequest.Builder builder = this.authRequestBuilder.POST(HttpRequest.BodyPublishers.ofString(payloadJSON));
        return builder.build();
    }

    @Override
    public void setAuthorizationHeader(HttpRequest.Builder httpRequestBuilder) {
        this.checkAuthenticationExpired();
        if (this.authenticationState == AuthenticationState.NOT_AUTHENTICATED || this.authenticationState == AuthenticationState.FAILED_AUTHENTICATION) {
            throw new DebeziumException("Must successfully authenticate against JWT endpoint before you can add the authorization information to the HTTP header.");
        }
        if (this.authenticationState == AuthenticationState.EXPIRED) {
            throw new DebeziumException("JWT authentication is expired. Must renew authentication before you can add the authorization information to the HTTP header.");
        }
        httpRequestBuilder.setHeader("Authorization", "Bearer: " + this.jwtToken);
    }

    @Override
    public boolean authenticate() throws InterruptedException {
        HttpResponse<String> r;
        HttpRequest request;
        this.checkAuthenticationExpired();
        if (this.authenticationState == AuthenticationState.ACTIVE) {
            return true;
        }
        if (this.authenticationState == AuthenticationState.NOT_AUTHENTICATED || this.authenticationState == AuthenticationState.FAILED_AUTHENTICATION) {
            request = this.generateInitialAuthenticationRequest();
        } else if (this.authenticationState == AuthenticationState.EXPIRED) {
            request = this.generateRefreshAuthenticationRequest();
        } else {
            throw new DebeziumException("Reached invalid authentication state.");
        }
        try {
            r = this.client.send(request, HttpResponse.BodyHandlers.ofString());
        }
        catch (IOException e) {
            throw new DebeziumException("Failed to send authentication request", (Throwable)e);
        }
        if (r.statusCode() == 200) {
            JWTAuthorizationResponse response;
            String responseBody = r.body();
            try {
                response = (JWTAuthorizationResponse)this.mapper.readValue(responseBody, JWTAuthorizationResponse.class);
            }
            catch (IOException e) {
                throw new DebeziumException("Could not deserialize JWT authorization response.", (Throwable)e);
            }
            this.jwtToken = response.getJwt();
            this.jwtRefreshToken = response.getJwtRefreshToken();
            long expirationDuration = (long)(0.9 * (double)response.getExpiresIn());
            this.expirationDateTime = DateTime.now().plus(expirationDuration);
            this.authenticationState = AuthenticationState.ACTIVE;
            return true;
        }
        this.authenticationState = AuthenticationState.FAILED_AUTHENTICATION;
        LOGGER.error("JWT Authentication failure. Check credentials.");
        return false;
    }

    static enum AuthenticationState {
        NOT_AUTHENTICATED,
        FAILED_AUTHENTICATION,
        ACTIVE,
        EXPIRED;

    }
}

