package io.adbrix.sdk.data.net;

import android.net.TrafficStats;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.HttpURLConnection;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import javax.net.ssl.HttpsURLConnection;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.IEventSender;
import io.adbrix.sdk.data.SdkVersion;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.IApiModel;
import io.adbrix.sdk.utils.Base64;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;
import io.adbrix.sdk.utils.HttpConnectionUtils;
import io.adbrix.sdk.utils.IoUtils;

public class ApiConnection implements IApiConnection {

    private DataRegistry dataRegistry;
    private int responseCode;
    private StringBuilder responseStringBuilder;
    private String dynamicHashKey;
    private JSONObject body;
    private String urlString;
    private ApiConnection.RequestMethod requestMethod;
    private Map<String, String> header = new HashMap<>();

    public ApiConnection(DataRegistry dataRegistry) {
        this.dataRegistry = dataRegistry;
    }

    @Override
    public int getResponseCode() {
        return responseCode;
    }

    @Override
    public String getResponseString() {
        return responseStringBuilder.toString();
    }

    @Override
    public IApiConnection header(Map<String, String> header) {
        this.header = header;
        return this;
    }

    @Override
    public IApiConnection get(String urlString) {
        this.urlString = urlString;
        this.requestMethod = RequestMethod.GET;
        return this;
    }

    @Override
    public IApiConnection get(IApiModel apiModel) {
        this.body = new JSONObject();
        try {
            this.body = apiModel.getJson();
        } catch (JSONException e) {
            AbxLog.w(e, true);
        }
        this.urlString = apiModel.getUrlString();
        this.requestMethod = RequestMethod.GET;
        return this;
    }

    @Override
    public IApiConnection post(IApiModel apiModel) {
        this.body = new JSONObject();
        try {
            this.body = apiModel.getJson();
        } catch (JSONException e) {
            AbxLog.w(e, true);
        }
        this.urlString = apiModel.getUrlString();
        this.requestMethod = RequestMethod.POST;
        return this;
    }

    @Override
    public IApiConnection delete(IApiModel apiModel) {
        this.body = new JSONObject();
        try {
            this.body = apiModel.getJson();
        } catch (JSONException e) {
            AbxLog.w(e, true);
        }
        this.urlString = apiModel.getUrlString();
        this.requestMethod = RequestMethod.DELETE;
        return this;
    }

    @Override
    public String request() throws Exception {
        validate();
        HttpURLConnection connection = null;
        try {
            connection = HttpConnectionUtils.createConnection(urlString, requestMethod, dataRegistry);
            setExtraHeader(connection);

            if (requestMethod == RequestMethod.POST || requestMethod == RequestMethod.DELETE) {
                setHeader(connection);
                writeBody(connection);
            }

            response(connection);

        } catch (IOException e) {
            makeErrorResponse(e);
//            AbxLog.w("["+requestMethod+"] "+urlString,e, true);
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
        }

        if (responseStringBuilder != null) {
            return responseStringBuilder.toString();
        } else return null;
    }


    private void validate() throws Exception {
        StringBuilder errorMessageBuilder = new StringBuilder();
        if (urlString == null) {
            errorMessageBuilder.append("urlString must not be null.\n");
        }
        if (requestMethod == null) {
            errorMessageBuilder.append("requestMethod must not be null.\n");
        }
        if (requestMethod == RequestMethod.POST && body == null) {
            errorMessageBuilder.append("body must not be null with POST method.\n");
        }
        if (header == null)
            header = new HashMap<>();

        if (errorMessageBuilder.length() > 0)
            throw new Exception(errorMessageBuilder.toString());
    }

    private void setExtraHeader(HttpURLConnection connection) {
        for (Map.Entry<String, String> entry : header.entrySet()) {
            connection.setRequestProperty(entry.getKey(), entry.getValue());
        }
    }

    private void setHeader(HttpURLConnection connection) {
        String userAgent = String.format("IGAWorks/%s (Android %s; U; %s)",
                SdkVersion.SDK_VERSION,
                dataRegistry.safeGetString(DataRegistryKey.STRING_OS, null),
                dataRegistry.safeGetString(DataRegistryKey.STRING_MODEL, null)
        );

        connection.setRequestProperty("User-Agent", userAgent);

        if (urlString.contains("opengdpr_requests") ||
                urlString.contains("inappmessage") ||
                urlString.toLowerCase(Locale.ENGLISH).contains("actionhistory") ||
                urlString.contains("users/profile") ||
                urlString.contains("users/ci")
        ) {
            connection.setRequestProperty("Content-Type", "application/json; charset=utf-8");
        } else {
            connection.setRequestProperty("Content-Type", "application/octet-stream; charset=utf-8");
        }

        connection.setRequestProperty("abx-auth-time", getAbxAuthTime(body));
        connection.setRequestProperty("abx-auth-cs", getAbxAuthCs(body));
        connection.setDoInput(true);
        connection.setDoOutput(true);
        connection.setRequestProperty("Accept-Charset", "UTF-8");
    }

    private void writeBody(HttpURLConnection connection) throws IOException {
        OutputStream outputStream = connection.getOutputStream();
        OutputStreamWriter outputStreamWriter = new OutputStreamWriter(outputStream, "UTF-8");
        BufferedWriter bufferedWriter = new BufferedWriter(outputStreamWriter);

        if (urlString.contains("opengdpr_requests") ||
                urlString.contains("inappmessage") ||
                urlString.toLowerCase(Locale.ENGLISH).contains("actionhistory") ||
                urlString.contains("users/profile") ||
                urlString.contains("users/ci")
        ) {
            bufferedWriter.write(body.toString());
        } else {
            bufferedWriter.write(Base64.encode(body.toString()));
        }
        bufferedWriter.flush();
        IoUtils.close(outputStream);
        IoUtils.close(outputStreamWriter);
        IoUtils.close(bufferedWriter);
    }

    private void response(HttpURLConnection connection) throws IOException {
        InputStream inputStream;
        responseStringBuilder = new StringBuilder();
        responseCode = connection.getResponseCode();

        if (200 <= responseCode && responseCode <= 299) {
            inputStream = connection.getInputStream();
        } else {
            inputStream = connection.getErrorStream();
        }
        InputStreamReader inputStreamReader = new InputStreamReader(inputStream);
        BufferedReader bufferedReader = new BufferedReader(inputStreamReader);

        String currentLine;

        while ((currentLine = bufferedReader.readLine()) != null)
            responseStringBuilder.append(currentLine);
        IoUtils.close(inputStream);
        IoUtils.close(inputStreamReader);
        IoUtils.close(bufferedReader);
    }

    @Override
    public boolean isHttpOK() {
        return HttpsURLConnection.HTTP_OK <= responseCode && responseCode < 300;
    }

    @Override
    public boolean isWrongAppkey() {
        return responseCode == 404 || (responseCode > 500 && responseCode < 600);
    }

    @Override
    public boolean isInvalidAppkey() {
        return responseCode == 400;
    }

    @Override
    public String getUrl() {
        return this.urlString;
    }

    @Override
    public JSONObject getBody() {
        return this.body;
    }

    @Override
    public RequestMethod getRequestMethod() {
        return this.requestMethod;
    }

    private String getAbxAuthTime(JSONObject jsonObject) {
        Date currentTime = null;
        SimpleDateFormat sdfKST = CoreUtils.createDateFormat(CoreConstants.DB_DATE_FORMAT);

        try {
            if (jsonObject.has("common")) {
                JSONObject common = jsonObject.getJSONObject("common");
                if (common.has("request_datetime")) {
                    Object requestDatetime = common.get("request_datetime");
                    if (requestDatetime instanceof String) {
                        currentTime = sdfKST.parse((String) requestDatetime);
                    }
                }
            }
        } catch (JSONException | ParseException e) {
            AbxLog.w("getAbxAuthTime"+e, true);
        } finally {
            if (currentTime == null) {
                currentTime = new Date();
            }
        }

        setDynamicHashKey(currentTime);
        return sdfKST.format(currentTime);
    }

    private void setDynamicHashKey(Date currentTime) {
        DateFormat h = CoreUtils.createDateFormat("HH");
        String secretKey = dataRegistry.safeGetString(DataRegistryKey.STRING_SECRETKEY, "");
        char[] epK = {
                (char) 99, (char) 50, (char) 100, (char) 108,
                (char) 97, (char) 51, (char) 74, (char) 122,
                (char) 98, (char) 88, (char) 82, (char) 105,
                (char) 89, (char) 81, (char) 61, (char) 61
        };
        // Default DSK v6
        int index = (Integer.parseInt(h.format(currentTime)) * 49 + 6909 * 6) % 32;
        if (urlString.contains("inappmessage")) { // DSK v7
            index = (Integer.parseInt(h.format(currentTime)) * 467 + 8537 * 4) % 32;
        }
        String fullHashKey = secretKey + Base64.decode(String.valueOf(epK));
        dynamicHashKey = fullHashKey.substring(index) + fullHashKey.substring(0, index);
    }

    private String getAbxAuthCs(JSONObject jsonObject) {
        // first encrypt by hashkey, second encrypt by dsk
        return CoreUtils.hmacSha256(dynamicHashKey, getDynamicHashKeyMD5(Base64.encode(jsonObject.toString())));
    }

    private String getDynamicHashKeyMD5(String string) {
        try {
            // Create MD5 Hash
            MessageDigest digest = java.security.MessageDigest.getInstance("MD5");
            digest.update(string.getBytes());
            byte[] messageDigest = digest.digest();

            // Create Hex String
            StringBuffer hexString = new StringBuffer();
            for (int i = 0; i < messageDigest.length; i++) {
                hexString.append(String.format("%02X", 0xFF & messageDigest[i]));
            }

            return hexString.toString();
        } catch (NoSuchAlgorithmException e) {
            AbxLog.w("getDynamicHashKeyMD5"+e, true);
        }
        return string;
    }

    public String getUrlString() {
        return urlString;
    }

    private void makeErrorResponse(Exception e){
        responseStringBuilder = new StringBuilder();
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put("method",requestMethod);
            jsonObject.put("url", urlString);
            jsonObject.put("exception", e.toString());
        } catch (JSONException ex) {
            ex.printStackTrace();
        }
        responseStringBuilder.append(jsonObject.toString());
        responseCode = IEventSender.RESPONSE_CODE_ERROR;
    }
}
