package io.adbrix.sdk.ui.inappmessage;

import android.app.Activity;
import android.content.Context;
import android.os.Build;
import android.os.Bundle;
import android.os.ResultReceiver;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;
import android.view.Window;
import android.view.inputmethod.InputMethodManager;
import android.widget.FrameLayout;

import com.igaworks.v2.core.AdBrixRm;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.domain.model.DfnInAppMessage;
import io.adbrix.sdk.domain.model.IAMEnums;
import io.adbrix.sdk.ui.inappmessage.webview.InAppMessageWebView;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.DisplayUtils;

public class DefaultInAppMessageViewWrapper {
    private final DfnInAppMessage inAppMessage;
    private InAppMessageWebView contentView;
    private final InAppMessageViewListener inAppMessageViewListener;
    private View previouslyFocusedView = null;

    public DefaultInAppMessageViewWrapper(DfnInAppMessage inAppMessage, InAppMessageViewListener inAppMessageViewListener) {
        this.inAppMessage = inAppMessage;
        this.inAppMessageViewListener = inAppMessageViewListener;
    }

    public void setContentView(InAppMessageWebView view){
        this.contentView = view;
    }

    public DfnInAppMessage getInAppMessage() {
        return inAppMessage;
    }

    public void open(Activity activity, DfnInAppMessage inAppMessage, String commonJS) {
        if(CommonUtils.isNull(this.contentView)){
            AbxLog.d("view is null", true);
            InAppMessageManager.getInstance().setIsInProgress(false);
            return;
        }

        final ViewGroup parentViewGroup = getParentViewGroup(activity);
        if(CommonUtils.isNull(parentViewGroup)){
            AbxLog.d("parentViewGroup is null", true);
            InAppMessageManager.getInstance().setIsInProgress(false);
            return;
        }
        AbxLog.d("parentViewGroup : "  + parentViewGroup.getChildCount(), true);
        this.previouslyFocusedView = activity.getCurrentFocus();
        final int parentViewGroupHeight = parentViewGroup.getHeight();
        if(parentViewGroupHeight == 0){
            ViewTreeObserver viewTreeObserver = parentViewGroup.getViewTreeObserver();
            if(viewTreeObserver.isAlive()){
                viewTreeObserver.addOnGlobalLayoutListener(
                        new ViewTreeObserver.OnGlobalLayoutListener() {
                            @Override
                            public void onGlobalLayout() {
                                if(CommonUtils.isNull(parentViewGroup)){
                                    AbxLog.d("parentViewGroup is null", true);
                                    InAppMessageManager.getInstance().setIsInProgress(false);
                                    return;
                                }
                                if(CommonUtils.isNull(contentView)){
                                    AbxLog.d("contentView is null", true);
                                    InAppMessageManager.getInstance().setIsInProgress(false);
                                    return;
                                }
                                AbxLog.d("Detected root view height of " + parentViewGroup.getHeight() + " in onGlobalLayout", true);
                                parentViewGroup.removeView(contentView);
                                addInAppMessageViewToViewGroup(activity, parentViewGroup, contentView, inAppMessage, commonJS);
                                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                                    parentViewGroup.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                                } else{
                                    parentViewGroup.getViewTreeObserver().removeGlobalOnLayoutListener(this);
                                }
                            }
                        });
            }
        } else {
            AbxLog.d("Detected root view height of " +parentViewGroup.getHeight() + " in normal", true);
            addInAppMessageViewToViewGroup(activity, parentViewGroup, contentView, inAppMessage, commonJS);
        }
    }

    private void addInAppMessageViewToViewGroup(Activity activity,
                                                ViewGroup parentViewGroup,
                                                View view,
                                                DfnInAppMessage inAppMessage,
                                                String commonJS){
        AbxLog.d("adding inAppMessageView to parentViewGroup", true);
        try{
            parentViewGroup.removeView(view);
            if(CommonUtils.notNull(view.getParent())){
                AbxLog.d("view's parent is removed. because view already has a parent.", true);
                ((ViewGroup)view.getParent()).removeView((ViewGroup)view);
            }
            if(CommonUtils.notNull(view.getParent())){
                AbxLog.d("view already has a parent", true);
                InAppMessageManager.getInstance().setIsInProgress(false);
                return;
            }
            if(InAppMessageManager.getInstance().isShowing()){
                AbxLog.d("InAppMessage is already shown", true);
                InAppMessageManager.getInstance().setIsInProgress(false);
                return;
            }
            view.setVisibility(View.INVISIBLE);
            parentViewGroup.addView(view, getLayoutParams(activity));
            parentViewGroup.post(new Runnable() {
                @Override
                public void run() {
                    if(CommonUtils.isNull(view)){
                        AbxLog.d("InAppMessage view is null", true);
                        InAppMessageManager.getInstance().setIsInProgress(false);
                        InAppMessageManager.getInstance().setShowing(false);
                        return;
                    }
                    if(CommonUtils.isNull(parentViewGroup)){
                        AbxLog.d("parentViewGroup is null", true);
                        InAppMessageManager.getInstance().setIsInProgress(false);
                        InAppMessageManager.getInstance().setShowing(false);
                        return;
                    }
                    if(CommonUtils.isNull(activity)){
                        AbxLog.d("activity is null", true);
                        InAppMessageManager.getInstance().setIsInProgress(false);
                        InAppMessageManager.getInstance().setShowing(false);
                        return;
                    }
                    if(CommonUtils.isNull(contentView)){
                        AbxLog.d("contentView is null", true);
                        InAppMessageManager.getInstance().setIsInProgress(false);
                        InAppMessageManager.getInstance().setShowing(false);
                        return;
                    }
                    AbxLog.d("add view: "+view+"\nfrom parent: "+parentViewGroup+"\n"+activity, true);
                    if(AdBrixRm.customInAppMessageParentView != null){
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                            view.setZ(24);
                        }
                    }
                    view.invalidate();
                    ViewCompat.requestApplyInsets(parentViewGroup);
                    ViewCompat.setOnApplyWindowInsetsListener(parentViewGroup, new OnApplyWindowInsetsListener() {
                        @Override
                        public WindowInsetsCompat onApplyWindowInsets(View v, WindowInsetsCompat insets) {
                            if(CommonUtils.isNull(insets)){
                                return insets;
                            }
                            applyWindowInsets(view, insets, inAppMessage);
                            return insets;
                        }
                    });
                    view.requestLayout();
                    String initHtml = String.format(InAppMessageWebView.INIT_HTML, commonJS);
                    contentView.loadDataWithBaseURL(null, initHtml, "text/html", "UTF-8", null);
                }
            });
        } catch(Exception e){
            AbxLog.e( "addInAppMessageViewToViewGroup failed. ", e, true);
            InAppMessageManager.getInstance().setIsInProgress(false);
            return;
        }
    }

    private ViewGroup.LayoutParams getLayoutParams(Activity activity){
        FrameLayout.LayoutParams layoutParams = new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT);
        if(CommonUtils.isNull(inAppMessage)){
            AbxLog.d("getLayoutParams inAppMessage is null", true);
            return layoutParams;
        }
        if(CommonUtils.isNull(inAppMessage.getType())){
            AbxLog.d("getLayoutParams inAppMessageType is null", true);
            return layoutParams;
        }
        if(InAppMessageFactoryContainer.STICKY_BANNER.equals(inAppMessage.getType())){
            layoutParams = new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, (int) DisplayUtils.dpToPx(activity, 98.0f));
            layoutParams.gravity = Gravity.CENTER_HORIZONTAL;
            layoutParams.rightMargin = (int) DisplayUtils.dpToPx(activity, 8.0f);
            layoutParams.leftMargin = (int) DisplayUtils.dpToPx(activity, 8.0f);
            layoutParams.topMargin = (int) DisplayUtils.dpToPx(activity, 8.0f);
            layoutParams = setStickyBannerLayoutGravity(layoutParams);
        }
        return layoutParams;
    }

    private FrameLayout.LayoutParams setStickyBannerLayoutGravity(FrameLayout.LayoutParams layoutParams){
        layoutParams.gravity = Gravity.TOP;
        String stickyBannerAlign = inAppMessage.getPosition();
        if(CommonUtils.isNullOrEmpty(stickyBannerAlign)){
            return layoutParams;
        }
        if(stickyBannerAlign.equals("top")){
            layoutParams.gravity = Gravity.TOP;
        } else{
            layoutParams.gravity = Gravity.BOTTOM;
        }
        AbxLog.d("getLayoutParams inAppMessageType stickyBannerAlign : " + stickyBannerAlign, true);
        return layoutParams;
    }

    /**
     * 2022.11.08 bobos
     * 자동으로 최상위 view를 찾는 코드
     * @param activity
     * @return
     */
    private ViewGroup getParentViewGroup(Activity activity){
        if(AdBrixRm.customInAppMessageParentView != null){
            AbxLog.d("using customInAppMessageViewGroup.", true);
            return AdBrixRm.customInAppMessageParentView;
        }
        Window window = activity.getWindow();
        if(CommonUtils.isNull(window)){
            AbxLog.d("parentWindow is null", true);
            return null;
        }
        View decorView = window.getDecorView();
        if(CommonUtils.isNull(decorView)){
            AbxLog.d("parentDecorView is null", true);
            return null;
        }
        ViewGroup viewGroup = decorView.findViewById(android.R.id.content);
        return viewGroup;
    }
    public void close(Activity activity, boolean isCarried){
        close(activity, isCarried, null);
    }

    public void close(Activity activity, boolean isCarried, CloseFinishedListener listener){
        AbxLog.d("closeInAppMessageView", true);
        if(CommonUtils.isNull(activity)){
            AbxLog.d("activity is null", true);
            InAppMessageManager.getInstance().setIsInProgress(false);
            if(CommonUtils.notNull(listener)){
                listener.onFinished();
            }
            return;
        }
        if(CommonUtils.isNull(contentView)){
            AbxLog.d("view is null", true);
            InAppMessageManager.getInstance().setIsInProgress(false);
            if(CommonUtils.notNull(listener)){
                listener.onFinished();
            }
            return;
        }
        activity.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                boolean isSuccess = removeViewFromParent(activity, contentView);
                if(isSuccess){
                    if(CommonUtils.notNull(previouslyFocusedView)){
                        AbxLog.d("Returning focus to previouslyFocusedView. View: "+previouslyFocusedView, true);
                        previouslyFocusedView.requestFocus();
                        if(InAppMessageManager.getInstance().wasUsingKeypad()){
                            if(!isCarried){
                                showKeypad(activity);
                                InAppMessageManager.getInstance().setUsingKeypad(false);
                            }
                        }
                    }
                    finishView();
                    if(CommonUtils.notNull(listener)){
                        listener.onFinished();
                    }
                }
            }
        });
    }

    private boolean removeViewFromParent(Activity activity, View view){
        boolean result = true;
        if(CommonUtils.isNull(activity)){
            AbxLog.d("activity is null", true);
            result = false;
            return result;
        }
        if(CommonUtils.isNull(view)){
            AbxLog.d("view is null", true);
            result = false;
            return result;
        }
        ViewGroup parentView = getParentViewGroup(activity);

        if(CommonUtils.isNull(parentView)){
            AbxLog.d("parentView is null", true);
            result = false;
            return result;
        }
        try {
            parentView.removeView(view);
            AbxLog.d("Removed view: "+view+"\nfrom parent: "+parentView+"\n"+activity, true);
        }catch (IllegalStateException e){
            AbxLog.e(e, true);
            result = false;
            return result;
        }
        return result;
    }

    public InAppMessageViewListener getInAppMessageViewListener(){
        return this.inAppMessageViewListener;
    }

    public void clearInAppMessageViewAnimation(){
        this.contentView.clearAnimation();
    }

    public void loadComplete(){
        Activity currentActivity = InAppMessageManager.getInstance().getCurrentActivity();
        Activity displayedActivity = InAppMessageManager.getInstance().getDisplayedActivity();
        if(currentActivity != displayedActivity){
            AbxLog.w("InAppMessageViewWrapper activity is changed.", true);
            InAppMessageManager.getInstance().cacheInAppMessage(IAMEnums.CloseType.ON_PAUSE);
            close(displayedActivity, true, new CloseFinishedListener() {
                @Override
                public void onFinished() {
                    InAppMessageManager.getInstance().show(InAppMessageManager.getInstance().getCarriedInAppMessage(), null, true);
                }
            });
            return;
        }
        try {
            if (contentView == null) {
                AbxLog.w( "contentView is null", true);
                InAppMessageManager.getInstance().setIsInProgress(false);
                InAppMessageManager.getInstance().setShowing(false);
                return;
            }
            hideKeypad(displayedActivity);
            contentView.post(new Runnable() {
                @Override
                public void run() {
                    AbxLog.d("InAppMessageViewWrapper loadComplete. "+System.currentTimeMillis(), true);
                    if(CommonUtils.isNull(contentView)){
                        AbxLog.e( "InAppMessageViewWrapper loadComplete failed. ", true);
                        InAppMessageManager.getInstance().setIsInProgress(false);
                        InAppMessageManager.getInstance().setShowing(false);
                        return;
                    }
                    contentView.setVisibility(View.VISIBLE);
                    contentView.bringToFront();
                    contentView.invalidate();
                    contentView.requestLayout();
                    finalizeViewBeforeDisplay(displayedActivity, contentView);
                    InAppMessageManager.getInstance().setIsInProgress(false);
                    InAppMessageManager.getInstance().setShowing(true);
                }
            });
        }catch(Exception e){
            AbxLog.e( "InAppMessageViewWrapper loadComplete failed. ", e, true);
            InAppMessageManager.getInstance().setIsInProgress(false);
            InAppMessageManager.getInstance().setShowing(false);
            return;
        }
    }

    private void finishView(){
        AbxLog.d("InAppMessageViewWrapper finishView", true);
        if (contentView != null) {
            contentView.getSettings().setJavaScriptEnabled(false);
            contentView.loadUrl(InAppMessageWebView.InAppMessageWebViewClient.FINISHED_WEBVIEW_URL);
            contentView.removeAllViews();
            contentView.destroy();
            contentView = null;
        }
        InAppMessageManager.getInstance().setShowing(false);
        InAppMessageManager.getInstance().setIsInProgress(false);
        InAppMessageManager.getInstance().flushCarriedInAppMessageViewFactory();
        InAppMessageManager.getInstance().flushDisplayedActivity();
    }

    private void applyWindowInsets(View view, WindowInsetsCompat insets, DfnInAppMessage inAppMessage){
        ViewGroup.LayoutParams layoutParams = view.getLayoutParams();
        if (layoutParams == null || !(layoutParams instanceof ViewGroup.MarginLayoutParams)) {
            AbxLog.d("Not applying window insets.", true);
            return;
        }
        ViewGroup.MarginLayoutParams marginLayoutParams;
        try {
            marginLayoutParams = (ViewGroup.MarginLayoutParams)layoutParams;
        }catch (Exception e){
            AbxLog.w(e, true);
            return;
        }
        Context context = view.getContext();
        int leftInset = getMaxSafeLeftInset(insets);
        int topInset = getMaxSafeTopInset(insets);
        int rightInset = getMaxSafeRightInset(insets);
        int bottomInset = getMaxSafeBottomInset(insets);
        if(InAppMessageFactoryContainer.STICKY_BANNER.equals(inAppMessage.getType())) {
            leftInset += (int) DisplayUtils.dpToPx(context, 8.0f);
            topInset +=  (int) DisplayUtils.dpToPx(context, 8.0f);
            rightInset +=  (int) DisplayUtils.dpToPx(context, 8.0f);
        }
        AbxLog.d("insets: "+leftInset+"/"+topInset+"/"+rightInset+"/"+bottomInset, true);
        marginLayoutParams.setMargins(leftInset, topInset, rightInset, bottomInset);
    }

    private void finalizeViewBeforeDisplay(Activity activity, View view) {
        if(CommonUtils.isNull(inAppMessage.getType())){
            setFocusableInTouchModeAndRequestFocus(view);
            return;
        }
        switch (inAppMessage.getType()){
            case InAppMessageFactoryContainer.MODAL:{
                setFocusableInTouchModeAndRequestFocus(view);
                break;
            }
            case InAppMessageFactoryContainer.FULL_SCREEN:
                setFocusableInTouchModeAndRequestFocus(view);
                break;
            case InAppMessageFactoryContainer.STICKY_BANNER:
                setFocusableInTouchModeAndRequestFocus(view);
                break;
        }
    }

    private int getMaxSafeLeftInset(WindowInsetsCompat windowInsets) {
        if (windowInsets.getDisplayCutout() != null) {
            final DisplayCutoutCompat displayCutout = windowInsets.getDisplayCutout();
            return Math.max(displayCutout.getSafeInsetLeft(), windowInsets.getSystemWindowInsetLeft());
        } else {
            return windowInsets.getSystemWindowInsetLeft();
        }
    }

    private int getMaxSafeRightInset(WindowInsetsCompat windowInsets) {
        if (windowInsets.getDisplayCutout() != null) {
            final DisplayCutoutCompat displayCutout = windowInsets.getDisplayCutout();
            return Math.max(displayCutout.getSafeInsetRight(), windowInsets.getSystemWindowInsetRight());
        } else {
            return windowInsets.getSystemWindowInsetRight();
        }
    }

    private int getMaxSafeTopInset(WindowInsetsCompat windowInsets) {
        if (windowInsets.getDisplayCutout() != null) {
            final DisplayCutoutCompat displayCutout = windowInsets.getDisplayCutout();
            return Math.max(displayCutout.getSafeInsetTop(), windowInsets.getSystemWindowInsetTop());
        } else {
            return windowInsets.getSystemWindowInsetTop();
        }
    }

    private int getMaxSafeBottomInset(WindowInsetsCompat windowInsets) {
        if (windowInsets.getDisplayCutout() != null) {
            final DisplayCutoutCompat displayCutout = windowInsets.getDisplayCutout();
            return Math.max(displayCutout.getSafeInsetBottom(), windowInsets.getSystemWindowInsetBottom());
        } else {
            return windowInsets.getSystemWindowInsetBottom();
        }
    }

    private boolean isDeviceNotInTouchMode(View view) {
        return !view.isInTouchMode();
    }

    private void setFocusableInTouchModeAndRequestFocus(View view) {
        try {
            view.setFocusableInTouchMode(true);
            view.requestFocus();
        } catch (Exception e) {
            AbxLog.e( "Caught exception while setting view to focusable in touch mode and requesting focus.", e, true);
        }
    }

    private void hideKeypad(Activity activity){
        if(this.previouslyFocusedView == null){
            return;
        }
        if(CommonUtils.isNull(activity)){
            return;
        }
        if(inAppMessage.getType().equals(InAppMessageFactoryContainer.STICKY_BANNER)){
            return;
        }
        InputMethodManager inputMethodManager = (InputMethodManager)activity.getSystemService(Context.INPUT_METHOD_SERVICE);
        if(CommonUtils.isNull(inputMethodManager)){
            return;
        }
        inputMethodManager.hideSoftInputFromWindow(this.previouslyFocusedView.getWindowToken(), InputMethodManager.HIDE_NOT_ALWAYS, new ResultReceiver(null){
            @Override
            protected void onReceiveResult(int resultCode, Bundle resultData) {
                super.onReceiveResult(resultCode, resultData);
                if (resultCode == InputMethodManager.RESULT_UNCHANGED_HIDDEN || resultCode == InputMethodManager.RESULT_HIDDEN){
                    InAppMessageManager.getInstance().setUsingKeypad(true);
                    AbxLog.w("InAppMessageViewWrapper Since the user was using the keypad, I redraw the screen.", true);
                    InAppMessageManager.getInstance().cacheInAppMessage(IAMEnums.CloseType.ON_PAUSE);
                    close(activity, true, new CloseFinishedListener() {
                        @Override
                        public void onFinished() {
                            InAppMessageManager.getInstance().show(InAppMessageManager.getInstance().getCarriedInAppMessage(), null, true);
                        }
                    });
                    return;
                }
            }
        });
    }
    private void showKeypad(Activity activity){
        if(CommonUtils.isNull(activity)){
            return;
        }
        InputMethodManager inputMethodManager = (InputMethodManager)activity.getSystemService(Context.INPUT_METHOD_SERVICE);
        if(CommonUtils.isNull(inputMethodManager)){
            return;
        }
        inputMethodManager.showSoftInput(previouslyFocusedView, InputMethodManager.SHOW_IMPLICIT);
    }
}
