package io.adbrix.sdk.ui.inappmessage.webview;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.os.Build;
import android.util.AttributeSet;
import android.util.Log;
import android.view.GestureDetector;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.View;
import android.view.animation.Animation;
import android.webkit.ConsoleMessage;
import android.webkit.RenderProcessGoneDetail;
import android.webkit.ValueCallback;
import android.webkit.WebChromeClient;
import android.webkit.WebResourceError;
import android.webkit.WebResourceRequest;
import android.webkit.WebResourceResponse;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;

import java.lang.reflect.Field;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.domain.model.DfnInAppMessage;
import io.adbrix.sdk.domain.model.IAMEnums;
import io.adbrix.sdk.ui.inappmessage.InAppMessageManager;
import io.adbrix.sdk.utils.AnimationUtils;
import io.adbrix.sdk.utils.CommonUtils;

public class InAppMessageWebView extends WebView {
    public static final String INIT_HTML = "<head>\n" +
            "  <meta charset=\"UTF-8\">\n" +
            "  <meta http-equiv=\"X-UA-Compatible\" content=\"IE=edge\">\n" +
            "  <meta name=\"viewport\" content=\"width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no\">\n" +
            "  <script type=\"text/javascript\" src=\"%s\"></script>\n" +
            "</head><body><div id=\'dfn-in-app-message-root\'></body>";

    private static Field sConfigCallback;
    private InAppMessageWebView webView;
    public static final long ANIMATION_DURATION_MS = 500;

    private boolean onLoadJavascript = false;
    static {
        try {
            sConfigCallback = Class.forName("android.webkit.BrowserFrame").getDeclaredField("sConfigCallback");
            sConfigCallback.setAccessible(true);
        } catch (Exception e) {
        }
    }

    public interface TouchListener  {
        default void onSwipeLeft() {
        }
        default void onSwipeRight() {
        }
    }

    public InAppMessageWebView(Context context) {
        super(context.getApplicationContext());
        webView = this;
        setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View view) {

            }
        });
    }

    public InAppMessageWebView(Context context, AttributeSet attrs) {
        super(context.getApplicationContext(), attrs);
        webView = this;
        setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View view) {

            }
        });
    }

    public InAppMessageWebView(Context context, AttributeSet attrs, int defStyle) {
        super(context.getApplicationContext(), attrs, defStyle);
        webView = this;
        setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View view) {

            }
        });
    }

    @Override
    public boolean onKeyDown(int keyCode, KeyEvent event) {
        if (keyCode == KeyEvent.KEYCODE_BACK){
            //click back button
            if(InAppMessageManager.getInstance().isInProgress()){
                return super.onKeyDown(keyCode, event);
            }
            InAppMessageManager.getInstance().close(IAMEnums.CloseType.KEY_DOWN_KEYCODE_BACK);
            return true;
        }
        return super.onKeyDown(keyCode, event);
    }

    @Override
    public boolean dispatchKeyEvent(KeyEvent event) {
        if(isInTouchMode()){
            return super.dispatchKeyEvent(event);
        }
        if (event.getKeyCode() == KeyEvent.KEYCODE_BACK){
            //touch back button
            if(InAppMessageManager.getInstance().isInProgress()){
                return super.dispatchKeyEvent(event);
            }
            InAppMessageManager.getInstance().close(IAMEnums.CloseType.KEY_DOWN_KEYCODE_BACK);
            return true;
        }
        return super.dispatchKeyEvent(event);
    }

    public void setWebViewSetting(DfnInAppMessage inAppMessage, InAppMessageWebViewInterface inAppMessageWebViewInterface){
        this.clearHistory();
        this.clearFormData();
        this.clearView();
        this.setBackgroundColor(Color.TRANSPARENT);
        this.setHorizontalScrollBarEnabled(false);
        this.setVerticalScrollBarEnabled(false);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
            this.setLayerType(View.LAYER_TYPE_HARDWARE, null);
        }
        else {
            this.setLayerType(View.LAYER_TYPE_SOFTWARE, null);
        }
        this.setWebChromeClient(new WebChromeClient(){
            @Override
            public boolean onConsoleMessage(ConsoleMessage consoleMessage) {
                AbxLog.d("webView#onConsoleMessage Line:"+consoleMessage.lineNumber()
                        +". SourceId:"+consoleMessage.sourceId()
                        +". Log Level:"+consoleMessage.messageLevel()
                        +". Message:"+consoleMessage.message(), true);
                return true;
            }

            @Override
            public Bitmap getDefaultVideoPoster() {
                return Bitmap.createBitmap(1, 1, Bitmap.Config.ARGB_8888);
            }
        });
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            this.setRendererPriorityPolicy(WebView.RENDERER_PRIORITY_BOUND, true);
        }
        this.addJavascriptInterface(inAppMessageWebViewInterface, InAppMessageWebViewInterface.WEB_VIEW_BRIDGE);
        this.setWebViewClient(new InAppMessageWebViewClient(inAppMessage.getHtml(), inAppMessage.getType()));
//        this.clearCache(true);
        WebSettings webSettings = this.getSettings();
        webSettings.setSupportZoom(false);
        webSettings.setBuiltInZoomControls(false);
        webSettings.setDisplayZoomControls(false);
        webSettings.setUseWideViewPort(true);
        webSettings.setLoadWithOverviewMode(true);
        webSettings.setJavaScriptEnabled(true);
        webSettings.setJavaScriptCanOpenWindowsAutomatically(false);
//        webSettings.setCacheMode(WebSettings.LOAD_NO_CACHE);
        webSettings.setCacheMode(WebSettings.LOAD_DEFAULT);
        webSettings.setPluginState(WebSettings.PluginState.ON);
        webSettings.setLoadWithOverviewMode(true);
        webSettings.setSupportMultipleWindows(true);
        webSettings.setLoadsImagesAutomatically(true);
        webSettings.setDomStorageEnabled(true);
        webSettings.setGeolocationEnabled(false);
        webSettings.setDefaultTextEncodingName("utf-8");
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            webSettings.setMixedContentMode(WebSettings.MIXED_CONTENT_ALWAYS_ALLOW);
        }
        if(InAppMessageManager.STICKY_BANNER.equals(inAppMessage.getType())){
            this.setOnTouchListener( new OnSwipeWebviewTouchListener(getContext(), new TouchListener() {
                @Override
                public void onSwipeLeft() {
                    if(!InAppMessageManager.getInstance().isShowing()){
                        AbxLog.w("inAppMessage is Animating..", true);
                        return;
                    }
                    InAppMessageManager.getInstance().close(IAMEnums.CloseType.CLOSE_BY_SWIPE_LEFT);
                }

                @Override
                public void onSwipeRight() {
                    if(!InAppMessageManager.getInstance().isShowing()){
                        AbxLog.w("inAppMessage is Animating..", true);
                        return;
                    }
                    InAppMessageManager.getInstance().close(IAMEnums.CloseType.CLOSE_BY_SWIPE_RIGHT);
                }
            }));
        }
    }

    public void setStickyAnimation(boolean isCarried, String position){
        try {
            if (!isCarried) {
                this.clearAnimation();
                Animation openingAnimation;
                if (position != null && position.equals("bottom")) {
                    openingAnimation = AnimationUtils.createVerticalAnimation(1f, 0f, ANIMATION_DURATION_MS, false);
                }
                else{
                    openingAnimation = AnimationUtils.createVerticalAnimation(-1.0f, 0f, ANIMATION_DURATION_MS, false);
                }
                openingAnimation.setAnimationListener(createAnimationListener());
                this.setAnimation(openingAnimation);
            }
        }catch (Exception e){}
    }

    public class OnSwipeWebviewTouchListener implements View.OnTouchListener {
        private final GestureDetector gestureDetector;
        public OnSwipeWebviewTouchListener(Context ctx, TouchListener touchListener) {
            gestureDetector = new GestureDetector(ctx, new GestureListener(touchListener));
        }
        @Override
        public boolean onTouch(View v, MotionEvent event) {
            return gestureDetector.onTouchEvent(event);
        }
        private final class GestureListener extends GestureDetector.SimpleOnGestureListener {
            private TouchListener touchListener;
            GestureListener(TouchListener touchListener) {
                super();
                this.touchListener = touchListener;
            }
            @Override
            public boolean onDown(MotionEvent e) {
                return false;  // THIS does the trick
            }
            @Override
            public boolean onFling(MotionEvent e1, MotionEvent e2, float velocityX, float velocityY) {
                boolean result = false;
                try {
                    float diffY = e2.getY() - e1.getY();
                    float diffX = e2.getX() - e1.getX();
                    if (Math.abs(diffX) > Math.abs(diffY)) {
                        // You can customize these settings, so 30 is an example
                        if (Math.abs(diffX) > 30 && Math.abs(velocityX) > 30) {
                            if (diffX > 0) {
                                touchListener.onSwipeRight();
                            } else {
                                touchListener.onSwipeLeft();
                            }
                            result = true;
                        }
                    } else {
                        result = false;
                    }
                } catch (Exception exception) {
                    exception.printStackTrace();
                }
                return result;
            }
        }
    }

    private Animation.AnimationListener createAnimationListener(){
        return new Animation.AnimationListener() {
            @Override
            public void onAnimationStart(Animation animation) {
                InAppMessageManager.getInstance().setIsInProgress(true);
            }

            @Override
            public void onAnimationEnd(Animation animation) {
                webView.clearAnimation();
            }

            @Override
            public void onAnimationRepeat(Animation animation) {

            }
        };
    }


    public class InAppMessageWebViewClient extends WebViewClient {
        public static final String FINISHED_WEBVIEW_URL = "about:blank";
        private String html = "";
        private String type = "";

        public InAppMessageWebViewClient(String html, String type){
            this.html = html;
            this.type = type;
        }

        @Override
        public void onPageStarted(WebView view, String url, Bitmap favicon) {
            super.onPageStarted(view, url, favicon);
            AbxLog.d("onPageStarted url: "+url, true);
        }

        @Override
        public void onPageFinished(WebView view, String url) {
            super.onPageFinished(view, url);
            AbxLog.d("onPageFinished url: "+url, true);
            if(FINISHED_WEBVIEW_URL.equals(url)){
                if(!onLoadJavascript) {
                    if(type.equals(InAppMessageManager.STICKY_BANNER)){
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                            webView.evaluateJavascript("javascript:DfnPopUp.appendHTMLForNativeSticky(`" + html + "`)", new ValueCallback<String>() {
                                @Override
                                public void onReceiveValue(String s) {
                                    AbxLog.d("onReceiveValue: "+s, true);
                                }
                            });
                        } else{
                            webView.loadUrl("javascript:DfnPopUp.appendHTMLForNativeSticky(`" + html + "`)");
                        }
                    }
                    else{
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                            webView.evaluateJavascript("javascript:DfnPopUp.appendHTMLForNative(`" + html + "`)", new ValueCallback<String>() {
                                @Override
                                public void onReceiveValue(String s) {
                                    AbxLog.d("onReceiveValue: "+s, true);
                                }
                            });
                        } else{
                            webView.loadUrl("javascript:DfnPopUp.appendHTMLForNative(`" + html + "`)");
                        }
                    }
                }
                onLoadJavascript = true;
            }
        }

        @Override
        public boolean onRenderProcessGone(WebView view, RenderProcessGoneDetail detail) {
            if(CommonUtils.isNull(detail)){
                AbxLog.d("RenderProcessGoneDetail is null", true);
                return super.onRenderProcessGone(view, detail);
            }
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                if(detail.didCrash()){
                    AbxLog.d("webview rendering process crashed", true);
                    return false;
                } else{
                    InAppMessageManager.getInstance().close(IAMEnums.CloseType.CLOSE_BY_SYSTEM_KILL);
                    return true;
                }
            }
            return super.onRenderProcessGone(view, detail);
        }

        @Override
        public void onReceivedError(WebView view, WebResourceRequest request, WebResourceError error) {
            super.onReceivedError(view, request, error);
            int errorCode = -1;
            CharSequence errorDescription = "";
            if(CommonUtils.notNull(error)){
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                    errorCode = error.getErrorCode();
                    errorDescription = error.getDescription();
                }
            }
            AbxLog.w("onReceivedError: "+errorCode+"/"+errorDescription, true);
            InAppMessageManager.getInstance().close(IAMEnums.CloseType.WEB_VIEW_ERROR);
        }

        @Override
        public void onReceivedHttpError(WebView view, WebResourceRequest request, WebResourceResponse errorResponse) {
            super.onReceivedHttpError(view, request, errorResponse);
            int statusCode = -1;
            String reasonPhrase = "";
            if(CommonUtils.notNull(errorResponse)){
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                    statusCode = errorResponse.getStatusCode();
                    reasonPhrase = errorResponse.getReasonPhrase();
                }
            }
            AbxLog.w("onReceivedHttpError: "+statusCode+"/"+reasonPhrase, true);
            InAppMessageManager.getInstance().close(IAMEnums.CloseType.WEB_VIEW_HTTP_ERROR);
        }
    }

    @Override
    public void destroy() {
        try {
            if( sConfigCallback!=null )
                sConfigCallback.set(null, null);
        } catch (Exception e) {
        }
        super.destroy();
    }
}