package io.adbrix.sdk.ui.push.utils;

import android.app.AlarmManager;
import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.media.AudioAttributes;
import android.media.RingtoneManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.text.Html;
import android.text.Spanned;
import android.util.TypedValue;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.domain.ABXConstants;
import io.adbrix.sdk.domain.CompatConstants;
import io.adbrix.sdk.ui.push.models.BigTextPushProperties;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.IoUtils;

public class PushUtils {
    private static final float BASE_WIDTH = 720f;
    private static final float BASE_HEIGHT = 1280f;
    public static int waitingForInitCount = 1;

    private PushUtils() {
    }

    public static JSONObject getJsonStringFromBundle(Bundle bundle) {
        JSONObject obj = new JSONObject();
        try {
            for (String key : bundle.keySet()) {
                Object temp = bundle.get(key);
                if (temp != null && key.equals(ABXConstants.PUSH_REMOTE_KEY_DEEPLINKJSON)) {
                    try {
                        JSONObject deepLinkJSON = new JSONObject((String) temp);
                        obj.put(key, deepLinkJSON);
                        continue;
                    } catch (Exception e) {
                        AbxLog.d("ERROR:: Invaild deeplink Json", true);
                        continue;
                    }
                }
                if (temp != null) {
                    obj.put(key, bundle.get(key));
                }
            }
        } catch (JSONException e) {
            AbxLog.e(e, true);
        }
        return obj;
    }

    @SuppressWarnings("deprecation")
    public static Spanned fromHtml(String html) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            return Html.fromHtml(html, Html.FROM_HTML_MODE_LEGACY);
        } else {
            return Html.fromHtml(html);
        }
    }

    public static int convertPixelToDP(Context context, int px, boolean isX) {

        int width = context.getResources().getDisplayMetrics().widthPixels;
        int height = context.getResources().getDisplayMetrics().heightPixels;

        if (context.getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE) {
            width = context.getResources().getDisplayMetrics().heightPixels;
            height = context.getResources().getDisplayMetrics().widthPixels;
        }

        float difX = width / BASE_WIDTH;
        float difY = height / BASE_HEIGHT;

        if (difX != difY) {
            difY = difX;
        }

        float norPx = px;

        if (isX) {
            norPx = (px * difX);
        } else {
            norPx = (px * difY);
        }

        if (norPx < 1.5f) {
            norPx = 1.5f;
        }

        return (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_PX, norPx, context.getResources().getDisplayMetrics());

    }

    public static int calNormPixel(Context context, int px, boolean isX) {

        int width = context.getResources().getDisplayMetrics().widthPixels;
        int height = context.getResources().getDisplayMetrics().heightPixels;

        if (context.getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE) {
            width = context.getResources().getDisplayMetrics().heightPixels;
            height = context.getResources().getDisplayMetrics().widthPixels;
        }
        float difX = width / BASE_WIDTH;
        float difY = height / BASE_HEIGHT;
        if (difX != difY) {
            difY = difX;
        }
        int norPx = px;
        if (isX) {
            norPx = (int) (px * width / BASE_WIDTH);
        } else {
            norPx = (int) (px * height / BASE_HEIGHT);
        }
        return norPx;
    }

    public static Object getObjectFromByteArray(byte[] byteArray) {

        ByteArrayInputStream bis = new ByteArrayInputStream(byteArray);
        ObjectInputStream in = null;
        BigTextPushProperties prop = null;
        Object object = null;
        try {
            in = new ObjectInputStream(bis);
            object = in.readObject();
        } catch (ClassNotFoundException e) {
            AbxLog.e(e, true);
        } catch (IOException e) {
            AbxLog.e(e, true);
        } finally {
            IoUtils.close(bis);
            IoUtils.close(in);
        }
        return object;
    }

    public static byte[] getByteArrayFromObject(Object bigTextPushProperties) {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        ObjectOutputStream out = null;
        byte[] data = null;
        try {
            out = new ObjectOutputStream(bos);
            out.writeObject(bigTextPushProperties);
            out.flush();
            data = bos.toByteArray();
        } catch (IOException e) {
            AbxLog.e(e, true);
        } finally {
            IoUtils.close(bos);
            IoUtils.close(out);
        }
        return data;
    }

    public static boolean checkHasVaildKey(JSONObject objFromServer, String key) {
        boolean result = false;
        if(objFromServer == null){
            return result;
        }
        if (!objFromServer.has(key)){
            return result;
        }
        try {
            Object value = objFromServer.get(key);
            if(JSONObject.NULL == objFromServer.get(key)){
                return result;
            }
            if(value == null){
                return result;
            }
            if(value instanceof String){
                if(CommonUtils.isNullOrEmpty(objFromServer.getString(key))){
                    return result;
                }
            }
        } catch (JSONException e) {
            AbxLog.e(e, true);
            return result;
        }
        result = true;
        return result;
    }

    @SuppressWarnings("deprecation")
    public static Notification.Builder createNotificationBuilder(Context context) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            return new Notification.Builder(context, CompatConstants.PUSH_NOTIFICATION_CHANNEL_ID);
        } else return new Notification.Builder(context);
    }

    public static int getPendingIntentFlags(Context context){
        if(Build.VERSION.SDK_INT >= Build.VERSION_CODES.M){
            if(getTargetSdkVersion(context) >= 31){
                return PendingIntent.FLAG_UPDATE_CURRENT|PendingIntent.FLAG_IMMUTABLE;
            }
        }
        return PendingIntent.FLAG_UPDATE_CURRENT;
    }

    public static int getTargetSdkVersion(Context context){
        int version = 0;
        PackageManager pm = context.getPackageManager();
        String packageName = context.getPackageName();
        try {
            ApplicationInfo applicationInfo = pm.getApplicationInfo(packageName, 0);
            if (applicationInfo != null) {
                version = applicationInfo.targetSdkVersion;
            }
        }catch (PackageManager.NameNotFoundException e) {
            AbxLog.e(e, true);
        }
        return version;
    }

    public static void setAlarm(Context context, int type, long triggerAtMillis, PendingIntent operation){
        AlarmManager alarmManager = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
        if(getTargetSdkVersion(context) >= 31){
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                alarmManager.setAndAllowWhileIdle(type, triggerAtMillis, operation);
                return;
            }
        }
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            alarmManager.setAndAllowWhileIdle(type, triggerAtMillis, operation);
            return;
        }
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
            alarmManager.setExact(type, triggerAtMillis, operation);
            return;
        }
        alarmManager.set(type, triggerAtMillis, operation);
    }

    public static JSONObject getLanguageJson(String language, JSONObject customAlertJSON, JSONObject objFromServer){
        JSONObject result = new JSONObject();
        if(CommonUtils.isNullOrEmpty(customAlertJSON)){
            return result;
        }
        try {
            if (customAlertJSON.has(language)) {
                result = customAlertJSON.getJSONObject(language);
                return result;
            }
            String subStringLanguage = language.substring(0, 2);
            if(customAlertJSON.has(subStringLanguage)){
                result = customAlertJSON.getJSONObject(subStringLanguage);
                return result;
            }
            if (objFromServer.has(ABXConstants.PUSH_REMOTE_KEY_ALERT)){
                Object object = objFromServer.get(ABXConstants.PUSH_REMOTE_KEY_ALERT);
                if(object != null && object instanceof JSONObject){
                    result = (JSONObject) object;
                    return result;
                }
            }
        } catch (JSONException e){
            AbxLog.e("ERROR::Can't create JSON Object from received data", false);
        }
        return result;
    }

    public static void createNotificationChannel(Context context, String channelName, String channelDescription, int importance, boolean vibrateEnable) {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.O) {
            AbxLog.d("createNotificationChannel: not supported ", false);
            return;
        }
        /*
        사용자로부터 channel, description 설정값 받아옴
         */
        NotificationManager notificationManager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
        String notificationChannelId = CompatConstants.PUSH_NOTIFICATION_CHANNEL_ID;
        NotificationChannel channel = notificationManager.getNotificationChannel(notificationChannelId);
        if(channel != null){
            AbxLog.d("notification channel is exist", false);
            return;
        }

        channel = new NotificationChannel(
            notificationChannelId,
            channelName,
            importance
        );
        if(!CommonUtils.isNullOrEmpty(channelDescription)){
            channel.setDescription(channelDescription);
        }

        Uri soundUri = RingtoneManager.getDefaultUri(RingtoneManager.TYPE_NOTIFICATION);
        AudioAttributes audioAttributes = new AudioAttributes.Builder()
                .setUsage(AudioAttributes.USAGE_NOTIFICATION)
                .build();
        channel.setSound(soundUri, audioAttributes);
        channel.enableVibration(vibrateEnable);
        channel.setImportance(importance);
        notificationManager.createNotificationChannel(channel);

        AbxLog.d("Creating channel is finished!", true);
    }

    public static String getApplicationName(Context context){
        String result = "";
        if(CommonUtils.isNull(context)){
            AbxLog.e("context is null", false);
            return result;
        }
        PackageManager packageManager = context.getPackageManager();
        ApplicationInfo applicationInfo = null;
        try {
            applicationInfo = packageManager.getApplicationInfo(context.getApplicationInfo().packageName, 0);
        }
        catch (final PackageManager.NameNotFoundException e) {
            AbxLog.e(e, false);
            return result;
        }
        if(applicationInfo == null){
            AbxLog.e("applicationInfo is null", false);
            result = context.getPackageName();
            return result;
        }
        result = (String) packageManager.getApplicationLabel(applicationInfo);
        return result;
    }
}
