package io.digified.digified_library.capture.v2.display;

import android.app.Activity;
import android.os.Build;
import android.util.Log;
import android.widget.FrameLayout;

import androidx.annotation.Nullable;

import io.digified.digified_library.R;
import io.digified.digified_library.capture.CaptureScreenMetaData;
import io.digified.digified_library.capture.v2.camera.CameraErrorCallback;
import io.digified.digified_library.capture.v2.camera.CameraSource;
import io.digified.digified_library.capture.v2.camera.CameraSourcePreview;
import io.digified.digified_library.capture.v2.detectors.DetectorInterface;
import io.digified.digified_library.capture.v2.detectors.VisionProcessorBase;
import io.digified.digified_library.capture.v2.detectors.manual_detector.ManualDetectorProcessor;
import io.digified.digified_library.capture.v2.detectors.manual_detector.core.ManualDetectorConstants;
import io.digified.digified_library.digified.DigifiedConstants;
import io.digified.digified_library.errors.DigifiedError;
import io.digified.digified_library.errors.ErrorConstants;

public class CameraMLViewModel implements CameraErrorCallback {

    private static final String TAG = "CameraMLViewModel";

    private static final int SELFIE_IMAGE_COUNT = 3;
    private static final int DOCUMENT_IMAGE_COUNT = 1;

    private static final int INSTRUCTIONS_DISPLAY_TIME = 2000;

    private CameraSource cameraSource;
    private final Activity activity;
    private final DetectorInterface detectorInterface;
    private final FrameLayout previewFrameLayout;

    private Integer desiredWidth;
    private Integer desiredHeight;
    private Integer captureType;

    private static final float CAMERA_FACTOR_DOCUMENT = 2.5f;
    private static final float CAMERA_FACTOR_SELFIE = 2f;

    private final FrameProcessorSelector frameProcessorSelector;

    private CaptureScreenMetaData captureScreenMetaData;
    private ManualDetectorProcessor manualDetectorProcessor;

    private CameraSourcePreview cameraSourcePreview;

    public CameraMLViewModel(Activity activity,
                             FrameLayout previewFrameLayout,
                             DetectorInterface detectorInterface) {
        this.activity = activity;
        this.detectorInterface = detectorInterface;
        this.previewFrameLayout = previewFrameLayout;
        this.frameProcessorSelector = new FrameProcessorSelector();
        this.captureScreenMetaData = null;
    }

    public void setCaptureScreenMetaData(@Nullable CaptureScreenMetaData captureScreenMetaData) {
        this.captureScreenMetaData = captureScreenMetaData;
    }


    //    /**
//     * Starts or restarts the camera source, if it exists. If the camera source doesn't exist yet
//     * (e.g., because onResume was called before the camera source was created), this will be called
//     * again when the camera source is created.
//     */
    private void startCameraSource() {
        if (cameraSource == null) {
            Log.e(TAG, "CameraSource is null");
            detectorInterface.onError(new DigifiedError(ErrorConstants.CaptureError.CAMERA_ERROR));
        }

        this.cameraSourcePreview = new CameraSourcePreview(activity, cameraSource);

        this.previewFrameLayout.addView(cameraSourcePreview);

        try {
            cameraSourcePreview.start(cameraSource);
        } catch (Exception e) {
            cameraSource.release();
            cameraSource = null;

            Log.e(TAG, "Unable to start camera source.", e);
            detectorInterface.onError(new DigifiedError(ErrorConstants.CaptureError.CAMERA_ERROR));
        }

    }

    public DisplayModel prepareCameraSource() {
        if (activity == null || desiredWidth == null || desiredHeight == null) return null;

        VisionProcessorBase<?> visionProcessorBase;

        DisplayModel displayModel;

        cameraSource = new CameraSource(activity, this);

        boolean shouldUseML = Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP;

        switch (captureType) {
            case DigifiedConstants.CaptureType.ID_FRONT:
                cameraSource.setPreviewParams(desiredWidth, desiredHeight, CAMERA_FACTOR_DOCUMENT);
                cameraSource.setFacing(CameraSource.CAMERA_FACING_BACK);

                visionProcessorBase = frameProcessorSelector.getIdFrontFrameProcessor(activity, desiredWidth, desiredHeight, DOCUMENT_IMAGE_COUNT, shouldUseML, detectorInterface);

                if (visionProcessorBase instanceof ManualDetectorProcessor) {
                    manualDetectorProcessor = (ManualDetectorProcessor) visionProcessorBase;
                    shouldUseML = false;
                }

                cameraSource.setFrameProcessor(visionProcessorBase);

                displayModel = new DisplayModel(R.string.digified_front_automatic_capture_title,
                        R.string.digified_front_automatic_capture_subtitle,
                        DisplayModel.DisplayType.DOCUMENT_SMALL,
                        false,
                        shouldUseML);

                if (captureScreenMetaData != null)
                    displayModel.overrideWithMetaData(captureScreenMetaData);

                return displayModel;

            case DigifiedConstants.CaptureType.ID_BACK:
                cameraSource.setPreviewParams(desiredWidth, desiredHeight, CAMERA_FACTOR_DOCUMENT);
                cameraSource.setFacing(CameraSource.CAMERA_FACING_BACK);

                visionProcessorBase = frameProcessorSelector.getIdBackFrameProcessor(activity, desiredWidth, desiredHeight, DOCUMENT_IMAGE_COUNT, shouldUseML, detectorInterface);

                if (visionProcessorBase instanceof ManualDetectorProcessor) {
                    manualDetectorProcessor = (ManualDetectorProcessor) visionProcessorBase;
                    shouldUseML = false;
                }

                cameraSource.setFrameProcessor(visionProcessorBase);

                displayModel = new DisplayModel(R.string.digified_back_automatic_capture_title,
                        R.string.digified_back_automatic_capture_subtitle,
                        DisplayModel.DisplayType.DOCUMENT_SMALL,
                        false,
                        shouldUseML);

                if (captureScreenMetaData != null)
                    displayModel.overrideWithMetaData(captureScreenMetaData);

                return displayModel;

            case DigifiedConstants.CaptureType.Generic_ID_BACK:
                cameraSource.setPreviewParams(desiredWidth, desiredHeight, CAMERA_FACTOR_DOCUMENT);
                cameraSource.setFacing(CameraSource.CAMERA_FACING_BACK);

                visionProcessorBase = frameProcessorSelector.getIdGenericBackFrameProcessor(activity, desiredWidth, desiredHeight, DOCUMENT_IMAGE_COUNT, shouldUseML, detectorInterface);

                if (visionProcessorBase instanceof ManualDetectorProcessor) {
                    manualDetectorProcessor = (ManualDetectorProcessor) visionProcessorBase;
                    shouldUseML = false;
                }

                cameraSource.setFrameProcessor(visionProcessorBase);

                displayModel = new DisplayModel(R.string.digified_generic_back_capture_title,
                        R.string.digified_generic_back_capture_subtitle,
                        DisplayModel.DisplayType.DOCUMENT_SMALL,
                        false,
                        false);

                if (captureScreenMetaData != null)
                    displayModel.overrideWithMetaData(captureScreenMetaData);

                return displayModel;

            case DigifiedConstants.CaptureType.VEHICLE_LICENSE_FRONT:
                cameraSource.setPreviewParams(desiredWidth, desiredHeight, CAMERA_FACTOR_DOCUMENT);
                cameraSource.setFacing(CameraSource.CAMERA_FACING_BACK);

                visionProcessorBase = frameProcessorSelector.getVehicleLicenseFrontFrameProcessor(activity, desiredWidth, desiredHeight, DOCUMENT_IMAGE_COUNT, shouldUseML, detectorInterface);

                if (visionProcessorBase instanceof ManualDetectorProcessor) {
                    manualDetectorProcessor = (ManualDetectorProcessor) visionProcessorBase;
                    shouldUseML = false;
                }

                cameraSource.setFrameProcessor(visionProcessorBase);

                displayModel = new DisplayModel(R.string.digified_vehicle_license_front_capture_title,
                        R.string.digified_vehicle_license_front_capture_subtitle,
                        DisplayModel.DisplayType.DOCUMENT_SMALL,
                        false,
                        false);

                if (captureScreenMetaData != null)
                    displayModel.overrideWithMetaData(captureScreenMetaData);

                return displayModel;

            case DigifiedConstants.CaptureType.VEHICLE_LICENSE_BACK:
                cameraSource.setPreviewParams(desiredWidth, desiredHeight, CAMERA_FACTOR_DOCUMENT);
                cameraSource.setFacing(CameraSource.CAMERA_FACING_BACK);

                visionProcessorBase = frameProcessorSelector.getVehicleLicenseBackFrameProcessor(activity, desiredWidth, desiredHeight, DOCUMENT_IMAGE_COUNT, shouldUseML, detectorInterface);

                if (visionProcessorBase instanceof ManualDetectorProcessor) {
                    manualDetectorProcessor = (ManualDetectorProcessor) visionProcessorBase;
                    shouldUseML = false;
                }

                cameraSource.setFrameProcessor(visionProcessorBase);

                displayModel = new DisplayModel(R.string.digified_vehicle_license_back_capture_title,
                        R.string.digified_vehicle_license_back_capture_subtitle,
                        DisplayModel.DisplayType.DOCUMENT_SMALL,
                        false,
                        false);

                if (captureScreenMetaData != null)
                    displayModel.overrideWithMetaData(captureScreenMetaData);

                return displayModel;

            case DigifiedConstants.CaptureType.PASSPORT:
                cameraSource.setPreviewParams(desiredWidth, desiredHeight, CAMERA_FACTOR_DOCUMENT);
                cameraSource.setFacing(CameraSource.CAMERA_FACING_BACK);

                visionProcessorBase = frameProcessorSelector.getPassportFrameProcessor(activity, desiredWidth, desiredHeight, DOCUMENT_IMAGE_COUNT, shouldUseML, detectorInterface);

                if (visionProcessorBase instanceof ManualDetectorProcessor) {
                    manualDetectorProcessor = (ManualDetectorProcessor) visionProcessorBase;
                    shouldUseML = false;
                }

                cameraSource.setFrameProcessor(visionProcessorBase);

                displayModel = new DisplayModel(R.string.digified_passport_capture_title,
                        R.string.digified_passport_capture_subtitle,
                        DisplayModel.DisplayType.DOCUMENT_SMALL,
                        false,
                        false);
                ;

                if (captureScreenMetaData != null)
                    displayModel.overrideWithMetaData(captureScreenMetaData);

                return displayModel;

            case DigifiedConstants.CaptureType.SELFIE:
                cameraSource.setPreviewParams(desiredWidth, desiredHeight, CAMERA_FACTOR_SELFIE);
                cameraSource.setFacing(CameraSource.CAMERA_FACING_FRONT);

                visionProcessorBase = frameProcessorSelector.getSelfieFrameProcessor(activity, desiredWidth, desiredHeight, SELFIE_IMAGE_COUNT, shouldUseML, detectorInterface);

                if (visionProcessorBase instanceof ManualDetectorProcessor) {
                    manualDetectorProcessor = (ManualDetectorProcessor) visionProcessorBase;
                    shouldUseML = false;
                }

                cameraSource.setFrameProcessor(visionProcessorBase);

                displayModel = new DisplayModel(R.string.digified_selfie_automatic_capture_title,
                        R.string.digified_selfie_automatic_capture_subtitle,
                        DisplayModel.DisplayType.SELFIE,
                        false,
                        shouldUseML);

                if (captureScreenMetaData != null)
                    displayModel.overrideWithMetaData(captureScreenMetaData);

                return displayModel;

            default: // case DigifiedConstants.CaptureType.Generic_ID_FRONT:
                cameraSource.setPreviewParams(desiredWidth, desiredHeight, CAMERA_FACTOR_DOCUMENT);
                cameraSource.setFacing(CameraSource.CAMERA_FACING_BACK);

                visionProcessorBase = frameProcessorSelector.getIdGenericFrontFrameProcessor(activity, desiredWidth, desiredHeight, DOCUMENT_IMAGE_COUNT, shouldUseML, detectorInterface);

                if (visionProcessorBase instanceof ManualDetectorProcessor) {
                    manualDetectorProcessor = (ManualDetectorProcessor) visionProcessorBase;
                }

                cameraSource.setFrameProcessor(visionProcessorBase);

                displayModel = new DisplayModel(R.string.digified_generic_front_capture_title,
                        R.string.digified_generic_front_capture_subtitle,
                        DisplayModel.DisplayType.DOCUMENT_SMALL,
                        false,
                        false);

                if (captureScreenMetaData != null)
                    displayModel.overrideWithMetaData(captureScreenMetaData);

                return displayModel;
        }
    }


    public void setParams(int captureType, int desiredWidth, int desiredHeight) {
        this.captureType = captureType;
        this.desiredWidth = desiredWidth;
        this.desiredHeight = desiredHeight;
    }


    public void onResume() {
        startCameraSource();
    }


    public void onPause() {
        cameraSourcePreview.stop();
    }

    public void onStop() {
        if (cameraSource != null) {
            cameraSource.release();
        }
    }

    @Override
    public void onCameraError() {
        detectorInterface.onError(new DigifiedError(ErrorConstants.CaptureError.CAMERA_ERROR));
    }

    public void manualCaptureRequested() {
        if (manualDetectorProcessor != null) manualDetectorProcessor.takePhoto();
    }

    public void switchToManualCapturingIfAuto() {
        if (cameraSource != null) {
            switch (captureType) {
                case DigifiedConstants.CaptureType.ID_FRONT:
                case DigifiedConstants.CaptureType.ID_BACK:
                    manualDetectorProcessor = new ManualDetectorProcessor(activity, desiredWidth, desiredHeight, DOCUMENT_IMAGE_COUNT, ManualDetectorConstants.Mode.SMALL_AREA, false, detectorInterface);
                    cameraSource.setFrameProcessor(manualDetectorProcessor);
//                    detectorInterface.onInstructions(R.string.digified_switched_to_manual_capturing, INSTRUCTIONS_DISPLAY_TIME);
                    detectorInterface.onStickyInstruction(R.string.digified_switched_to_manual_capturing);
                    return;
                case DigifiedConstants.CaptureType.SELFIE:
                    manualDetectorProcessor = new ManualDetectorProcessor(activity, desiredWidth, desiredHeight, SELFIE_IMAGE_COUNT, ManualDetectorConstants.Mode.SMALL_AREA, true, detectorInterface);
                    cameraSource.setFrameProcessor(manualDetectorProcessor);
//                    detectorInterface.onInstructions(R.string.digified_switched_to_manual_capturing, INSTRUCTIONS_DISPLAY_TIME);
                    detectorInterface.onStickyInstruction(R.string.digified_switched_to_manual_capturing);
                    return;
            }
        }
    }
}
