package io.digified.digified_library.digified;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.FragmentManager;

import io.digified.digified_library.api.callbacks.FaceMatchCallback;
import io.digified.digified_library.api.callbacks.GenericIdExtractionCallback;
import io.digified.digified_library.api.callbacks.IdExtractionCallback;
import io.digified.digified_library.api.callbacks.InitializeCallback;
import io.digified.digified_library.api.callbacks.PassportExtractionCallback;
import io.digified.digified_library.api.callbacks.TransliterationCallback;
import io.digified.digified_library.api.callbacks.VehicleLicenseExtractionCallback;
import io.digified.digified_library.authorize.DigifiedAuthorizationCallback;
import io.digified.digified_library.capture.CaptureCallback;
import io.digified.digified_library.capture.CaptureResult;
import io.digified.digified_library.capture.CaptureScreenMetaData;


abstract class BiometricDigified {

    private static final String TAG = "Digified";
    private final DigifiedInternal digifiedInternal;

    /**
     * private constructor
     *
     * @param context context
     * @param baseUrl https url of the backend hosting Digified API
     * @param apiKey  API Key of the Digified API
     */
    protected BiometricDigified(Context context, String baseUrl, String apiKey) {
        digifiedInternal = new DigifiedInternal(context, apiKey, baseUrl);
        resetSequence();
    }


    /**
     * updating the API key, usually this will be used to verify another document or use case
     *
     * @param newApiKey New API Key of the Digified API
     */
    public void updateApiKey(String newApiKey) {
        digifiedInternal.updateApiKey(newApiKey);
    }

    /**
     * Setting the max dimensions of images not to be exceeded after cropping the image
     * (the exact size of the cropped image can't be set to avoid stretching the image)
     *
     * @param maxWidth  maximum width of the output image
     * @param maxHeight maximum height of the output image
     */
    public void setMaxDimensions(int maxWidth, int maxHeight) {
        digifiedInternal.setMaxDimensions(maxWidth, maxHeight);
    }

    /**
     * Setting the capture time out, this controls when will {@link CaptureCallback#onTimeout()}
     * is called
     *
     * @param captureTimeoutMillis time out in milli seconds
     */
    public void setCaptureTimeout(int captureTimeoutMillis) {
        digifiedInternal.setCaptureTimeout(captureTimeoutMillis);
    }

    /**
     * Enables fallback to manual capturing after {@link CaptureCallback#onTimeout()} happens
     * if capturing is using ML models, it is disabled by default.
     * (time out can be set from {@link #setCaptureTimeout(int)}
     */
    public void enableManualCapturingFallback() {
        digifiedInternal.enableManualCapturingFallback();
    }

    /**
     * Use this method to get the final status of the verification process, the final status will be
     * on of those statuses {@link DigifiedConstants.Status}
     *
     * @return final status of the verification process
     */
    public int getFinalStatus() {
        return digifiedInternal.getFinalStatus();
    }

    /**
     * Use this method to get the next action that you should let the user perform based on
     * the predefined sequence in the SDK. Actions can be found here {@link DigifiedConstants.Action}
     *
     * @return the next action to be performed (the us should let the user performs)
     */
    public int getNextAction() {
        return digifiedInternal.getCurrentAction();
    }

    /**
     * rests the SDK to the initial stats
     */
    public void resetSequence() {
        digifiedInternal.resetSequence();
    }

    /**
     * Performs the init request of the API
     *
     * @param warningAction setting the warning action for the api (optional - set to null if you will not use it)
     * @param userName      name of the user (optional - set to if you will not use it)
     * @param phoneNumber   user's phone number (optional - set to null if you will not use it)
     * @param email         user's email (optional - set to null if you will not use it)
     * @param callback      callback interface to let you know the status of the request
     */
    public void initialize(@Nullable String warningAction, @Nullable String userName, @Nullable String phoneNumber, @Nullable String email, @NonNull InitializeCallback callback) {
        digifiedInternal.initialize(warningAction, userName, phoneNumber, email, callback);
    }

    /**
     * Performs the init request of the API
     * if use this method, then the following fields be set as follows:
     *
     *  phoneNumber, userName, email are going to be set to null
     * <p>
     * warningAction is going to be set to report {@link DigifiedConstants.WarningAction#REPORT}
     *
     * @param callback callback interface to let you know the status of the request
     */
    public void initialize(InitializeCallback callback) {
        digifiedInternal.initialize(null, null, null, null, callback);
    }

    /**
     * This method is used to capture images (ID front image, ID back image and Selfie)
     *
     * @param fragmentManager FragmentManager that will handle the display of the fragments of the SDK
     * @param containerId     Resource Id of the view that will be replaced with the capturing fragment
     * @param captureType     Specifies what you are capturing (ID Front, ID Back or Selfie)
     *                        find the types here {@link DigifiedConstants.CaptureType}
     * @param captureCallback interface that will let you deal with the capture request
     */
    public void capture(FragmentManager fragmentManager, int containerId, int captureType, CaptureCallback captureCallback) {
        digifiedInternal.capture(fragmentManager, containerId, captureType, null, captureCallback);
    }

    /**
     * This method is used to capture images (ID front image, ID back image and Selfie)
     *
     * @param fragmentManager FragmentManager that will handle the display of the fragments of the SDK
     * @param containerId     Resource Id of the view that will be replaced with the capturing fragment
     * @param captureType     Specifies what you are capturing (ID Front, ID Back or Selfie)
     *                        find the types here {@link DigifiedConstants.CaptureType}
     * @param desiredWidth    required image width (not guaranteed in order not to stretch the image,
     *                        however SDK will not exceed that width)
     * @param desiredHeight   required image height (not guaranteed in order not to stretch the image,
     *                        however SDK will not exceed that height)
     * @param captureCallback interface that will let you deal with the capture request
     */
    public void capture(FragmentManager fragmentManager, int containerId, int captureType, int desiredWidth, int desiredHeight, CaptureCallback captureCallback) {
        digifiedInternal.capture(fragmentManager, containerId, captureType, desiredWidth, desiredHeight, null, captureCallback);
    }

    /**
     * This method is used to capture images (ID front image, ID back image and Selfie)
     *
     * @param fragmentManager       FragmentManager that will handle the display of the fragments of the SDK
     * @param containerId           Resource Id of the view that will be replaced with the capturing fragment
     * @param captureType           Specifies what you are capturing (ID Front, ID Back or Selfie)
     *                              find the types here {@link DigifiedConstants.CaptureType}
     * @param captureScreenMetaData {@link CaptureScreenMetaData} that will be displayed while capturing
     * @param captureCallback       interface that will let you deal with the capture request
     */
    public void capture(FragmentManager fragmentManager, int containerId, int captureType, CaptureScreenMetaData captureScreenMetaData, CaptureCallback captureCallback) {
        digifiedInternal.capture(fragmentManager, containerId, captureType, captureScreenMetaData, captureCallback);
    }

    /**
     * This method is used to capture images (ID front image, ID back image and Selfie)
     *
     * @param fragmentManager       FragmentManager that will handle the display of the fragments of the SDK
     * @param containerId           Resource Id of the view that will be replaced with the capturing fragment
     * @param captureType           Specifies what you are capturing (ID Front, ID Back or Selfie)
     *                              find the types here {@link DigifiedConstants.CaptureType}
     * @param desiredWidth          required image width (not guaranteed in order not to stretch the image,
     *                              however SDK will not exceed that width)
     * @param desiredHeight         required image height (not guaranteed in order not to stretch the image,
     *                              however SDK will not exceed that height)
     * @param captureScreenMetaData {@link CaptureScreenMetaData} that will be displayed while capturing
     * @param captureCallback       interface that will let you deal with the capture request
     */
    public void capture(FragmentManager fragmentManager, int containerId, int captureType, int desiredWidth, int desiredHeight, CaptureScreenMetaData captureScreenMetaData, CaptureCallback captureCallback) {
        digifiedInternal.capture(fragmentManager, containerId, captureType, desiredWidth, desiredHeight, captureScreenMetaData, captureCallback);
    }

    /**
     * cancelling the capture request (usually will be used if capture timeout passed and you want
     * to allow the user to cancel the capturing request)
     */
    public void cancelCapture() {
        digifiedInternal.cancel();
    }

    /**
     * Sending the captured image of ID front and back
     * (encapsulated in CaptureResult object) to the API
     * CaptureResult will be obtained after calling {@link #capture(FragmentManager, int, int, CaptureCallback)}
     * or {@link #capture(FragmentManager, int, int, int, int, CaptureCallback)}
     *
     * @param captureResult        CaptureResult object containing the ID Front or ID Back image
     * @param idExtractionCallback callback interface containing the result of the API call
     */
    public void send(CaptureResult captureResult, IdExtractionCallback idExtractionCallback) {
        digifiedInternal.send(captureResult, idExtractionCallback);
    }

    /**
     * Sending the captured image of Vehicle License front and back
     * (encapsulated in CaptureResult object) to the API
     * CaptureResult will be obtained after calling {@link #capture(FragmentManager, int, int, CaptureCallback)}
     * or {@link #capture(FragmentManager, int, int, int, int, CaptureCallback)}
     *
     * @param captureResult                    CaptureResult object containing the ID Front or ID Back image
     * @param vehicleLicenseExtractionCallback interface containing the result of the API call
     */
    public void send(CaptureResult captureResult, VehicleLicenseExtractionCallback vehicleLicenseExtractionCallback) {
        digifiedInternal.send(captureResult, vehicleLicenseExtractionCallback);
    }

    /**
     * Sending the captured selfie image (encapsulated in CaptureResult object) to the API
     * CaptureResult will be obtained after calling {@link #capture(FragmentManager, int, int, CaptureCallback)}
     * or {@link #capture(FragmentManager, int, int, int, int, CaptureCallback)}
     *
     * @param captureResult     CaptureResult object containing the Selfie image
     * @param faceMatchCallback callback interface containing the result of the API call
     */
    public void send(CaptureResult captureResult, FaceMatchCallback faceMatchCallback) {
        digifiedInternal.send(captureResult, faceMatchCallback);
    }

    /**
     * Sending the captured passport image (encapsulated in CaptureResult object) to the API
     * CaptureResult will be obtained after calling {@link #capture(FragmentManager, int, int, CaptureCallback)}
     * or {@link #capture(FragmentManager, int, int, int, int, CaptureCallback)}
     *
     * @param captureResult              CaptureResult object containing the Selfie image
     * @param passportExtractionCallback callback interface containing the result of the API call
     */
    public void send(CaptureResult captureResult, PassportExtractionCallback passportExtractionCallback) {
        digifiedInternal.send(captureResult, passportExtractionCallback);
    }

    /**
     * Sending the captured image of a generic ID front and back
     * (encapsulated in CaptureResult object) to the API
     * CaptureResult will be obtained after calling {@link #capture(FragmentManager, int, int, CaptureCallback)}
     * or {@link #capture(FragmentManager, int, int, int, int, CaptureCallback)}
     *
     * @param captureResult               CaptureResult object containing the ID Front or ID Back image
     * @param genericIdExtractionCallback callback interface containing the result of the API call
     */
    public void send(CaptureResult captureResult, GenericIdExtractionCallback genericIdExtractionCallback) {
        digifiedInternal.send(captureResult, genericIdExtractionCallback);
    }

    /**
     * Getting the transliteration of the arabic name in the document scanned
     *
     * @param transliterationCallback callback interface containing the result of the API call
     */
    public void transliterate(TransliterationCallback transliterationCallback) {
        digifiedInternal.transliterate(transliterationCallback);
    }

//    /**
//     * Fetching the {@link IdFetchedResult} which is the ID Verification result associated
//     * with a certain session token
//     *
//     * @param token                           session token for an old ID verification session
//     * @param idCompletedVerificationCallback callback interface containing the result of the API call
//     */
//    public void fetchResult(String token, IdCompletedVerificationCallback idCompletedVerificationCallback) {
//        digifiedInternal.fetchResult(token, idCompletedVerificationCallback);
//    }
//
//    /**
//     * Fetching the {@link GenericIdFetchedResult} which is the Generic ID Verification result
//     * associated with a certain session token
//     *
//     * @param token                                  session token for an old ID verification session
//     * @param genericIdCompletedVerificationCallback callback interface containing the result of the API call
//     */
//    public void fetchResult(String token, GenericIdCompletedVerificationCallback genericIdCompletedVerificationCallback) {
//        digifiedInternal.fetchResult(token, genericIdCompletedVerificationCallback);
//    }
//
//    /**
//     * Fetching the {@link PassportFetchedResult} which is the Passport Verification result
//     * associated with a certain session token
//     *
//     * @param token                                 session token for an old ID verification session
//     * @param passportCompletedVerificationCallback callback interface containing the result of the API call
//     */
//    public void fetchResult(String token, PassportCompletedVerificationCallback passportCompletedVerificationCallback) {
//        digifiedInternal.fetchResult(token, passportCompletedVerificationCallback);
//    }
//

    /**
     * This methods helps you to decide if the user is authorized to do some critical actions on your app by
     * checking if the current user made a successful verification before, by:
     * 1- Getting the verification result by the provided token
     * 2- Taking a new selfie of the user and match the selfie image with
     * the verified selfie image from the old verified result
     *
     * @param token                 biometric token from older request
     * @param fragmentManager       FragmentManager that will handle the display of the fragments of the SDK
     * @param containerId           Resource Id of the view that will be replaced with the capturing fragment
     * @param authorizationCallback callback interface containing the result of the authorization process
     */
    public void shouldAuthorize(String token, FragmentManager fragmentManager, int containerId, DigifiedAuthorizationCallback authorizationCallback) {
        digifiedInternal.shouldAuthorize(token, fragmentManager, containerId, null, authorizationCallback);
    }

    /**
     * This methods helps you to decide if the user is authorized to do some critical actions on your app by
     * checking if the current user made a successful verification before, by:
     * 1- Getting the verification result by the provided token
     * 2- Taking a new selfie of the user and match the selfie image with
     * the verified selfie image from the old verified result
     *
     * @param token                 biometric token from older request
     * @param fragmentManager       FragmentManager that will handle the display of the fragments of the SDK
     * @param containerId           Resource Id of the view that will be replaced with the capturing fragment
     * @param captureScreenMetaData {@link CaptureScreenMetaData} that will be displayed while capturing
     * @param authorizationCallback callback interface containing the result of the authorization process
     */
    public void shouldAuthorize(String token, FragmentManager fragmentManager, int containerId, CaptureScreenMetaData captureScreenMetaData, DigifiedAuthorizationCallback authorizationCallback) {
        digifiedInternal.shouldAuthorize(token, fragmentManager, containerId, captureScreenMetaData, authorizationCallback);
    }

}
