package io.digified.digified_library.api;

import android.content.Context;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.concurrent.TimeUnit;

import io.digified.digified_library.api.ApiConstants.ContentType;
import io.digified.digified_library.api.ApiConstants.DocumentType;
import io.digified.digified_library.api.ApiConstants.FileFormat;
import io.digified.digified_library.api.ApiConstants.ParamsKeys;
import io.digified.digified_library.api.ApiConstants.Url;
import io.digified.digified_library.api.callbacks.DigifiedCallback;
import io.digified.digified_library.api.callbacks.FaceMatchCallback;
import io.digified.digified_library.api.callbacks.GenericIdExtractionCallback;
import io.digified.digified_library.api.callbacks.IdExtractionCallback;
import io.digified.digified_library.api.callbacks.PassportExtractionCallback;
import io.digified.digified_library.api.callbacks.VehicleLicenseExtractionCallback;
import io.digified.digified_library.api.network.DigifiedAPI;
import io.digified.digified_library.api.network.ProgressRequestBody;
import io.digified.digified_library.api.network.ProgressUpdater;
import io.digified.digified_library.api.result.BaseDigifiedResult;
import io.digified.digified_library.api.result.DigifiedBitmapResult;
import io.digified.digified_library.api.result.FaceMatchResult;
import io.digified.digified_library.api.result.GenericIdExtractionResult;
import io.digified.digified_library.api.result.GenericIdFetchedResult;
import io.digified.digified_library.api.result.IdExtractionResult;
import io.digified.digified_library.api.result.IdFetchedResult;
import io.digified.digified_library.api.result.InitializationResult;
import io.digified.digified_library.api.result.PassportExtractionResult;
import io.digified.digified_library.api.result.PassportFetchedResult;
import io.digified.digified_library.api.result.TransliterationResult;
import io.digified.digified_library.api.result.VehicleLicenseExtractionResult;
import io.digified.digified_library.api.result.internal.generic_id.GenericIdFrontData;
import io.digified.digified_library.api.utils.NetworkConnectionChecker;
import io.digified.digified_library.digified.DigifiedConstants.CaptureType;
import io.digified.digified_library.errors.DigifiedError;
import io.digified.digified_library.errors.ErrorConstants;
import okhttp3.CipherSuite;
import okhttp3.ConnectionSpec;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.RequestBody;
import okhttp3.ResponseBody;
import okhttp3.TlsVersion;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;

public class ApiManager {

    private static final String TAG = "ApiManager";

    private static final int TIME_OUT_SECONDS = 60;

    private static final int EXPECTED_SELFIE_FRAMES = 3;

    private static ApiManager apiManager;
    private final NetworkConnectionChecker networkConnectionChecker;
    private String token;
    private final DigifiedAPI digifiedAPI;
    private String apiKey;
    private final String apiVersion;

    private GenericIdFrontData regulaTempGenericFrontId = null;

    private ApiManager(Context context, String apiKey, String baseUrl) {
        networkConnectionChecker = new NetworkConnectionChecker(context);
        this.apiKey = apiKey;
        this.apiVersion = Url.API_VERSION;
        this.digifiedAPI = createDigifiedAPI(baseUrl);
        this.token = null;
    }

    private DigifiedAPI createDigifiedAPI(String baseUrl) {

        ConnectionSpec spec = new ConnectionSpec.Builder(ConnectionSpec.MODERN_TLS).tlsVersions(TlsVersion.TLS_1_2).cipherSuites(CipherSuite.TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256, CipherSuite.TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256, CipherSuite.TLS_DHE_RSA_WITH_AES_128_GCM_SHA256).build();

        ArrayList<ConnectionSpec> specArrayList = new ArrayList<>();
        specArrayList.add(spec);

        OkHttpClient okHttpClient = new OkHttpClient.Builder().readTimeout(TIME_OUT_SECONDS, TimeUnit.SECONDS).connectTimeout(TIME_OUT_SECONDS, TimeUnit.SECONDS).connectionSpecs(specArrayList).cache(null).build();


        Retrofit retrofit = new Retrofit.Builder().baseUrl(baseUrl).addConverterFactory(GsonConverterFactory.create()).client(okHttpClient).build();

        return retrofit.create(DigifiedAPI.class);
    }

    public static ApiManager getInstance(Context context, String apiKey, String baseUrl) {
        if (apiManager == null) {
            apiManager = new ApiManager(context, apiKey, baseUrl);
        }
        return apiManager;
    }

    public void updateApiKey(String apiKey) {
        this.apiKey = apiKey;
    }

    public void initialize(@Nullable String warningAction, @Nullable String userName, @Nullable String phoneNumber, @Nullable String email, @NonNull DigifiedCallback<InitializationResult> callback) {
        if (apiKey == null) {
            Log.e(TAG, "api key is null");
            callback.onFailure(new DigifiedError(ErrorConstants.SendError.API_KEY_ERROR));
            return;
        } else if (apiKey.isEmpty()) {
            Log.e(TAG, "api key is empty");
            callback.onFailure(new DigifiedError(ErrorConstants.SendError.API_KEY_ERROR));
            return;
        }

        if (isDisconnected(callback)) return;

        HashMap<String, String> hashMap = new HashMap<>();
        hashMap.put(ParamsKeys.API_KEY, apiKey);
        hashMap.put(ParamsKeys.API_VERSION, apiVersion);

        if (warningAction != null) hashMap.put(ParamsKeys.WARNING_ACTION, warningAction);
        if (userName != null) hashMap.put(ParamsKeys.USER_NAME, userName);
        if (phoneNumber != null) hashMap.put(ParamsKeys.PHONE, phoneNumber);
        if (email != null) hashMap.put(ParamsKeys.EMAIL, email);

        Call<InitializationResult> call = digifiedAPI.init(hashMap);

        call.enqueue(new Callback<>() {
            @Override
            public void onResponse(@NonNull Call<InitializationResult> call, @NonNull Response<InitializationResult> response) {
                boolean shouldTerminate = false;
                if (response != null) {
                    InitializationResult initializationResult = response.body();

                    if (initializationResult != null) {
                        token = initializationResult.getToken();

                        callback.onResult(initializationResult);
                        return;
                    }

                    if (response.errorBody() != null) {
                        shouldTerminate = checkErrorBodyForTermination(response.errorBody());
                    }

                }
                onResponseError(shouldTerminate, callback);
            }

            @Override
            public void onFailure(@NonNull Call<InitializationResult> call, @NonNull Throwable t) {
                callback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));
            }
        });
    }

    public void extractFrontId(@NonNull String idFrontBase64, @NonNull IdExtractionCallback callback) {

        if (shouldEndCall(callback, token, idFrontBase64)) return;

        ProgressUpdater progressUpdater = new ProgressUpdater(CaptureType.ID_FRONT, callback);

        HashMap<String, RequestBody> hashMap = new HashMap<>();
        hashMap.put(ParamsKeys.FORMAT, createRequestBody(FileFormat.BASE64));
//        hashMap.put(ParamsKeys.DOCUMENT, createRequestBody(DocumentType.EGYPTIAN_ID));
        hashMap.put(ParamsKeys.FRONT, createRequestBody(idFrontBase64, progressUpdater));
        hashMap.put(ParamsKeys.TOKEN, createRequestBody(token));

        Call<IdExtractionResult> call = digifiedAPI.extractID(hashMap);


        call.enqueue(new Callback<IdExtractionResult>() {
            @Override
            public void onResponse(@NonNull Call<IdExtractionResult> call, @NonNull Response<IdExtractionResult> response) {
                boolean shouldTerminate = false;
                if (response != null) {
                    IdExtractionResult frontIdExtractionResult = response.body();
                    if (frontIdExtractionResult != null) {
                        callback.onResult(frontIdExtractionResult);
                        return;
                    }
                    if (response.errorBody() != null) {
                        shouldTerminate = checkErrorBodyForTermination(response.errorBody());
                    }

                }
                onResponseError(shouldTerminate, callback);
            }

            @Override
            public void onFailure(@NonNull Call<IdExtractionResult> call, @NonNull Throwable t) {
                callback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));

            }
        });
    }

    public void extractFrontGenericId(@NonNull String frontGenericIdBase64, @NonNull GenericIdExtractionCallback callback) {

        if (shouldEndCall(callback, token, frontGenericIdBase64)) return;

        ProgressUpdater progressUpdater = new ProgressUpdater(CaptureType.Generic_ID_FRONT, callback);

        HashMap<String, RequestBody> hashMap = new HashMap<>();
        hashMap.put(ParamsKeys.FORMAT, createRequestBody(FileFormat.BASE64));
//        hashMap.put(ParamsKeys.DOCUMENT, createRequestBody(DocumentType.GENERIC_ID));
        hashMap.put(ParamsKeys.FRONT, createRequestBody(frontGenericIdBase64, progressUpdater));
        hashMap.put(ParamsKeys.TOKEN, createRequestBody(token));

        Call<GenericIdExtractionResult> call = digifiedAPI.extractGenericID(hashMap);


        call.enqueue(new Callback<>() {
            @Override
            public void onResponse(@NonNull Call<GenericIdExtractionResult> call, @NonNull Response<GenericIdExtractionResult> response) {
                boolean shouldTerminate = false;
                if (response != null) {
                    GenericIdExtractionResult genericFrontIdExtractionResult = response.body();
                    if (genericFrontIdExtractionResult != null) {
                        regulaTempGenericFrontId = genericFrontIdExtractionResult.getGenericIdFrontData();
                        callback.onResult(genericFrontIdExtractionResult);
                        return;
                    }
                    if (response.errorBody() != null) {
                        shouldTerminate = checkErrorBodyForTermination(response.errorBody());
                    }

                }
                onResponseError(shouldTerminate, callback);
            }

            @Override
            public void onFailure(@NonNull Call<GenericIdExtractionResult> call, @NonNull Throwable t) {
                callback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));
            }
        });
    }

    public void extractFrontVehicleLicense(@NonNull String vehicleLicenseFrontBase64, @NonNull VehicleLicenseExtractionCallback callback) {

        if (shouldEndCall(callback, token, vehicleLicenseFrontBase64)) return;

        ProgressUpdater progressUpdater = new ProgressUpdater(CaptureType.VEHICLE_LICENSE_FRONT, callback);

        HashMap<String, RequestBody> hashMap = new HashMap<>();
        hashMap.put(ParamsKeys.FORMAT, createRequestBody(FileFormat.BASE64));
        hashMap.put(ParamsKeys.DOCUMENT, createRequestBody(DocumentType.VEHICLE_LICENSE));
        hashMap.put(ParamsKeys.FRONT, createRequestBody(vehicleLicenseFrontBase64, progressUpdater));
        hashMap.put(ParamsKeys.TOKEN, createRequestBody(token));

        Call<VehicleLicenseExtractionResult> call = digifiedAPI.extractVehicleLicense(hashMap);

        call.enqueue(new Callback<>() {
            @Override
            public void onResponse(@NonNull Call<VehicleLicenseExtractionResult> call, @NonNull Response<VehicleLicenseExtractionResult> response) {
                boolean shouldTerminate = false;
                if (response != null) {
                    VehicleLicenseExtractionResult vehicleLicenseExtractionResult = response.body();
                    if (vehicleLicenseExtractionResult != null) {
                        callback.onResult(vehicleLicenseExtractionResult);
                        return;
                    }
                    if (response.errorBody() != null) {
                        shouldTerminate = checkErrorBodyForTermination(response.errorBody());
                    }

                }
                onResponseError(shouldTerminate, callback);
            }

            @Override
            public void onFailure(@NonNull Call<VehicleLicenseExtractionResult> call, @NonNull Throwable t) {
                callback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));

            }
        });
    }

    public void extractBackId(@NonNull String idBackBase64, @NonNull IdExtractionCallback callback) {

        if (shouldEndCall(callback, token, idBackBase64)) return;

//        callback.onUploadProgress(CaptureType.ID_BACK);
        ProgressUpdater progressUpdater = new ProgressUpdater(CaptureType.ID_BACK, callback);

        HashMap<String, RequestBody> hashMap = new HashMap<>();
        hashMap.put(ParamsKeys.FORMAT, createRequestBody(FileFormat.BASE64));
//        hashMap.put(ParamsKeys.DOCUMENT, createRequestBody(DocumentType.EGYPTIAN_ID));
        hashMap.put(ParamsKeys.BACK, createRequestBody(idBackBase64, progressUpdater));
        hashMap.put(ParamsKeys.TOKEN, createRequestBody(token));

        Call<IdExtractionResult> call = digifiedAPI.extractID(hashMap);

        call.enqueue(new Callback<>() {
            @Override
            public void onResponse(@NonNull Call<IdExtractionResult> call, @NonNull Response<IdExtractionResult> response) {
                boolean shouldTerminate = false;
                if (response != null) {
                    IdExtractionResult backIdExtractionResult = response.body();
                    if (backIdExtractionResult != null) {
                        callback.onResult(backIdExtractionResult);

                        return;
                    }
                    if (response.errorBody() != null) {
                        shouldTerminate = checkErrorBodyForTermination(response.errorBody());
                    }

                }
                onResponseError(shouldTerminate, callback);
            }

            @Override
            public void onFailure(@NonNull Call<IdExtractionResult> call, @NonNull Throwable t) {
                callback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));

            }
        });
    }

    public void extractBackGenericId(@NonNull String idBackBase64, @NonNull GenericIdExtractionCallback callback) {

        if (shouldEndCall(callback, token, idBackBase64)) return;

//        callback.onUploadProgress(CaptureType.ID_BACK);

        ProgressUpdater progressUpdater = new ProgressUpdater(CaptureType.Generic_ID_BACK, callback);


        HashMap<String, RequestBody> hashMap = new HashMap<>();
        hashMap.put(ParamsKeys.FORMAT, createRequestBody(FileFormat.BASE64));
//        hashMap.put(ParamsKeys.DOCUMENT, createRequestBody(DocumentType.GENERIC_ID));
        hashMap.put(ParamsKeys.BACK, createRequestBody(idBackBase64, progressUpdater));
        hashMap.put(ParamsKeys.TOKEN, createRequestBody(token));

        Call<GenericIdExtractionResult> call = digifiedAPI.extractGenericID(hashMap);

        call.enqueue(new Callback<>() {
            @Override
            public void onResponse(@NonNull Call<GenericIdExtractionResult> call, @NonNull Response<GenericIdExtractionResult> response) {
                boolean shouldTerminate = false;
                if (response != null) {
                    GenericIdExtractionResult genericBackIdExtractionResult = response.body();
                    if (genericBackIdExtractionResult != null) {
                        genericBackIdExtractionResult.setGenericIdFrontData(regulaTempGenericFrontId);
                        callback.onResult(genericBackIdExtractionResult);

                        return;
                    }
                    if (response.errorBody() != null) {
                        shouldTerminate = checkErrorBodyForTermination(response.errorBody());
                    }

                }
                onResponseError(shouldTerminate, callback);
            }

            @Override
            public void onFailure(@NonNull Call<GenericIdExtractionResult> call, @NonNull Throwable t) {
                callback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));

            }
        });
    }

    public void extractBackVehicleLicense(@NonNull String vehicleLicenseBackBase64, @NonNull VehicleLicenseExtractionCallback callback) {

        if (shouldEndCall(callback, token, vehicleLicenseBackBase64)) return;

        ProgressUpdater progressUpdater = new ProgressUpdater(CaptureType.VEHICLE_LICENSE_BACK, callback);

        HashMap<String, RequestBody> hashMap = new HashMap<>();
        hashMap.put(ParamsKeys.FORMAT, createRequestBody(FileFormat.BASE64));
        hashMap.put(ParamsKeys.DOCUMENT, createRequestBody(DocumentType.VEHICLE_LICENSE));
        hashMap.put(ParamsKeys.BACK, createRequestBody(vehicleLicenseBackBase64, progressUpdater));
        hashMap.put(ParamsKeys.TOKEN, createRequestBody(token));

        Call<VehicleLicenseExtractionResult> call = digifiedAPI.extractVehicleLicense(hashMap);

        call.enqueue(new Callback<>() {
            @Override
            public void onResponse(@NonNull Call<VehicleLicenseExtractionResult> call, @NonNull Response<VehicleLicenseExtractionResult> response) {
                boolean shouldTerminate = false;
                if (response != null) {
                    VehicleLicenseExtractionResult vehicleLicenseExtractionResult = response.body();
                    if (vehicleLicenseExtractionResult != null) {
                        callback.onResult(vehicleLicenseExtractionResult);
                        return;
                    }
                    if (response.errorBody() != null) {
                        shouldTerminate = checkErrorBodyForTermination(response.errorBody());
                    }

                }
                onResponseError(shouldTerminate, callback);
            }

            @Override
            public void onFailure(@NonNull Call<VehicleLicenseExtractionResult> call, @NonNull Throwable t) {
                callback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));

            }
        });
    }

    public void extractPassport(@NonNull String passportBase64, @NonNull PassportExtractionCallback callback) {
        if (shouldEndCall(callback, token, passportBase64)) return;

        ProgressUpdater progressUpdater = new ProgressUpdater(CaptureType.PASSPORT, callback);

        HashMap<String, RequestBody> hashMap = new HashMap<>();
        hashMap.put(ParamsKeys.FORMAT, createRequestBody(FileFormat.BASE64));
//        hashMap.put(ParamsKeys.DOCUMENT, createRequestBody(DocumentType.PASSPORT));
        hashMap.put(ParamsKeys.FRONT, createRequestBody(passportBase64, progressUpdater));
        hashMap.put(ParamsKeys.TOKEN, createRequestBody(token));

        Call<PassportExtractionResult> call = digifiedAPI.extractPassport(hashMap);

        call.enqueue(new Callback<>() {
            @Override
            public void onResponse(@NonNull Call<PassportExtractionResult> call, @NonNull Response<PassportExtractionResult> response) {
                boolean shouldTerminate = false;
                if (response != null) {
                    PassportExtractionResult passportExtractionResult = response.body();
                    if (passportExtractionResult != null) {
                        callback.onResult(passportExtractionResult);
                        return;
                    }
                    if (response.errorBody() != null) {
                        shouldTerminate = checkErrorBodyForTermination(response.errorBody());
                    }

                }
                onResponseError(shouldTerminate, callback);
            }

            @Override
            public void onFailure(@NonNull Call<PassportExtractionResult> call, @NonNull Throwable t) {
                callback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));

            }
        });
    }

    public void faceMatch(@NonNull ArrayList<String> selfieFramesBase64, @NonNull FaceMatchCallback callback) {

        if (selfieFramesBase64.size() < EXPECTED_SELFIE_FRAMES)
            callback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));

        if (shouldEndCall(callback, token, selfieFramesBase64.get(0), selfieFramesBase64.get(1), selfieFramesBase64.get(2)))
            return;

        ProgressUpdater progressUpdater = new ProgressUpdater(CaptureType.SELFIE, callback);

        HashMap<String, RequestBody> hashMap = new HashMap<>();
        hashMap.put(ParamsKeys.FORMAT, createRequestBody(FileFormat.BASE64));
        hashMap.put(ParamsKeys.TOKEN, createRequestBody(token));
        hashMap.put(ParamsKeys.FRAME_1, createRequestBody(selfieFramesBase64.get(0), progressUpdater));
        hashMap.put(ParamsKeys.FRAME_2, createRequestBody(selfieFramesBase64.get(1), progressUpdater));
        hashMap.put(ParamsKeys.FRAME_3, createRequestBody(selfieFramesBase64.get(2), progressUpdater));

        Call<FaceMatchResult> call = digifiedAPI.faceMatch(hashMap);

        call.enqueue(new Callback<>() {
            @Override
            public void onResponse(@NonNull Call<FaceMatchResult> call, @NonNull Response<FaceMatchResult> response) {
                boolean shouldTerminate = false;
                if (response != null) {
                    FaceMatchResult faceMatchResult = response.body();

                    if (faceMatchResult != null) {
                        callback.onResult(faceMatchResult);
                        return;
                    }

                    if (response.errorBody() != null) {
                        shouldTerminate = checkErrorBodyForTermination(response.errorBody());
                    }

                }
                onResponseError(shouldTerminate, callback);
            }

            @Override
            public void onFailure(@NonNull Call<FaceMatchResult> call, @NonNull Throwable t) {
                callback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));
            }
        });

    }

    public void faceMatch(@NonNull ArrayList<String> selfieFramesBase64, @NonNull String oldToken, @NonNull FaceMatchCallback callback) {

        if (selfieFramesBase64.size() < EXPECTED_SELFIE_FRAMES)
            callback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));

        if (shouldEndCall(callback, token, selfieFramesBase64.get(0), selfieFramesBase64.get(1), selfieFramesBase64.get(2)))
            return;

        ProgressUpdater progressUpdater = new ProgressUpdater(CaptureType.SELFIE, callback);

        HashMap<String, RequestBody> hashMap = new HashMap<>();
        hashMap.put(ParamsKeys.FORMAT, createRequestBody(FileFormat.BASE64));
        hashMap.put(ParamsKeys.TOKEN, createRequestBody(token));
        hashMap.put(ParamsKeys.PREVIOUS, createRequestBody(oldToken));
        //we're taking only one image not the whole three selfie images
        hashMap.put(ParamsKeys.SELFIE, createRequestBody(selfieFramesBase64.get(0), progressUpdater));

        Call<FaceMatchResult> call = digifiedAPI.faceMatch(hashMap);

        call.enqueue(new Callback<>() {
            @Override
            public void onResponse(@NonNull Call<FaceMatchResult> call, @NonNull Response<FaceMatchResult> response) {
                boolean shouldTerminate = false;
                if (response != null) {
                    FaceMatchResult faceMatchResult = response.body();

                    if (faceMatchResult != null) {
                        callback.onResult(faceMatchResult);
                        return;
                    }

                    if (response.errorBody() != null) {
                        shouldTerminate = checkErrorBodyForTermination(response.errorBody());
                    }

                }
                onResponseError(shouldTerminate, callback);
            }

            @Override
            public void onFailure(@NonNull Call<FaceMatchResult> call, @NonNull Throwable t) {
                callback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));
            }
        });

    }

    public void matchFacesImages(@NonNull String retrievedImageBase64, @NonNull String capturedImageBase64, FaceMatchCallback callback) {

        if (shouldEndCall(callback, token, retrievedImageBase64, capturedImageBase64)) return;

        ProgressUpdater progressUpdater = new ProgressUpdater(CaptureType.SELFIE, callback);

        HashMap<String, RequestBody> hashMap = new HashMap<>();
        hashMap.put(ParamsKeys.FORMAT, createRequestBody(FileFormat.BASE64));
        hashMap.put(ParamsKeys.TOKEN, createRequestBody(token));
        hashMap.put(ParamsKeys.ID_FACE, createRequestBody(retrievedImageBase64, progressUpdater));
        hashMap.put(ParamsKeys.SELFIE, createRequestBody(capturedImageBase64, progressUpdater));

        Call<FaceMatchResult> call = digifiedAPI.faceMatch(hashMap);
        call.enqueue(new Callback<>() {
            @Override
            public void onResponse(@NonNull Call<FaceMatchResult> call, @NonNull Response<FaceMatchResult> response) {
                boolean shouldTerminate = false;
                if (response != null) {
                    FaceMatchResult faceMatchResult = response.body();

                    if (faceMatchResult != null) {
                        callback.onResult(faceMatchResult);
                        return;
                    }

                    if (response.errorBody() != null) {
                        shouldTerminate = checkErrorBodyForTermination(response.errorBody());
                    }

                }
                onResponseError(shouldTerminate, callback);
            }

            @Override
            public void onFailure(@NonNull Call<FaceMatchResult> call, @NonNull Throwable t) {
                callback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));
            }
        });


    }

    public void transliterate(@NonNull DigifiedCallback<TransliterationResult> callback) {

        HashMap<String, String> hashMap = new HashMap<>();
        hashMap.put(ParamsKeys.TOKEN, token);

        Call<TransliterationResult> call = digifiedAPI.transliterate(hashMap);

        call.enqueue(new Callback<>() {
            @Override
            public void onResponse(@NonNull Call<TransliterationResult> call, @NonNull Response<TransliterationResult> response) {
                boolean shouldTerminate = false;

                if (response != null) {
                    TransliterationResult transliterationResult = response.body();

                    if (transliterationResult != null) {
                        callback.onResult(transliterationResult);
                        return;
                    }

                    if (response.errorBody() != null) {
                        shouldTerminate = checkErrorBodyForTermination(response.errorBody());
                    }
                }
                onResponseError(shouldTerminate, callback);
            }

            @Override
            public void onFailure(@NonNull Call<TransliterationResult> call, @NonNull Throwable t) {
                callback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));
            }
        });
    }

    public void fetchIdResult(@NonNull String token, @NonNull DigifiedCallback<IdFetchedResult> callback) {

        if (isTokenNull(token, callback)) return;

        Call<IdFetchedResult> call = digifiedAPI.fetchIdResults(token);
        call.enqueue(new Callback<>() {
            @Override
            public void onResponse(@NonNull Call<IdFetchedResult> call, @NonNull Response<IdFetchedResult> response) {

                if (response != null) {
                    IdFetchedResult idFetchedResult = response.body();

                    if (idFetchedResult != null) {
                        callback.onResult(idFetchedResult);
                        return;
                    }
                }

                onResponseError(false, callback);
            }

            @Override
            public void onFailure(@NonNull Call<IdFetchedResult> call, @NonNull Throwable t) {
                callback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));
            }
        });

    }

    public void fetchGenericIdResult(@NonNull String token, @NonNull DigifiedCallback<GenericIdFetchedResult> callback) {

        if (isTokenNull(token, callback)) return;

        Call<GenericIdFetchedResult> call = digifiedAPI.fetchGenericIdResults(token);
        call.enqueue(new Callback<>() {
            @Override
            public void onResponse(@NonNull Call<GenericIdFetchedResult> call, @NonNull Response<GenericIdFetchedResult> response) {

                if (response != null) {
                    GenericIdFetchedResult genericIdFetchedResult = response.body();

                    if (genericIdFetchedResult != null) {
                        callback.onResult(genericIdFetchedResult);
                        return;
                    }
                }

                onResponseError(false, callback);
            }

            @Override
            public void onFailure(@NonNull Call<GenericIdFetchedResult> call, @NonNull Throwable t) {
                callback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));
            }
        });

    }

    public void fetchPassportResult(@NonNull String token, @NonNull DigifiedCallback<PassportFetchedResult> callback) {

        if (isTokenNull(token, callback)) return;

        Call<PassportFetchedResult> call = digifiedAPI.fetchPassportResults(token);
        call.enqueue(new Callback<>() {
            @Override
            public void onResponse(@NonNull Call<PassportFetchedResult> call, @NonNull Response<PassportFetchedResult> response) {

                if (response != null) {
                    PassportFetchedResult passportFetchedResult = response.body();

                    if (passportFetchedResult != null) {

                        callback.onResult(passportFetchedResult);
                        return;
                    }
                }

                onResponseError(false, callback);
            }

            @Override
            public void onFailure(@NonNull Call<PassportFetchedResult> call, @NonNull Throwable t) {
                callback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));
            }
        });
    }

    public void downloadImageBitmap(@NonNull String url, @NonNull DigifiedCallback<DigifiedBitmapResult> bitmapDownloadCallback) {


        if (shouldEndCall(bitmapDownloadCallback, token, url)) return;

        Call<ResponseBody> call = digifiedAPI.downloadImageBitmap(url);
        call.enqueue(new Callback<>() {
            @Override
            public void onResponse(@NonNull Call<ResponseBody> call, @NonNull Response<ResponseBody> response) {

                if (response != null) {
                    ResponseBody responseBody = response.body();

                    if (responseBody != null) {
                        DigifiedBitmapResult digifiedBitmapResult = new DigifiedBitmapResult(responseBody);
                        bitmapDownloadCallback.onResult(digifiedBitmapResult);
                        return;
                    }
                }

                bitmapDownloadCallback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));
            }

            @Override
            public void onFailure(@NonNull Call<ResponseBody> call, @NonNull Throwable t) {
                bitmapDownloadCallback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));
            }
        });

    }

    ////////// checking for errors ///////////
    private boolean checkErrorBodyForTermination(ResponseBody errorResponseBody) {
        final String shouldTerminate = "should_terminate";
        final String error = "true";
        final String splitterCharacter = ",";

        try {
            String errorMessage = errorResponseBody.string();
            String[] splits = errorMessage.split(splitterCharacter);

            for (String s : splits) {
                if (s.contains(shouldTerminate) && s.contains(error)) {
                    return true;
                }
            }
            return false;
        } catch (IOException e) {
            return false;
        }
    }

    private <T extends BaseDigifiedResult> boolean shouldEndCall(DigifiedCallback<T> digifiedCallback, String token, String... imageBase64Strings) {
        if (isDisconnected(digifiedCallback)) return true;

        if (isTokenNull(token, digifiedCallback)) return true;

        for (String base64 : imageBase64Strings)
            if (isImageNull(base64, digifiedCallback)) return true;

        return false;
    }

    private <T extends BaseDigifiedResult> boolean isDisconnected(DigifiedCallback<T> digifiedCallback) {
        if (networkConnectionChecker.isDisconnected()) {
            Log.e(TAG, "Not connected to the internet!");
            digifiedCallback.onFailure(new DigifiedError(ErrorConstants.SendError.NO_NETWORK_CONNECTION));
            return true;
        } else {
            return false;
        }
    }

    private <T extends BaseDigifiedResult> boolean isTokenNull(String token, DigifiedCallback<T> digifiedCallback) {
        if (token == null) {
            Log.e(TAG, "Token is null! make sure you made the initialization");
            digifiedCallback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));
            return true;
        } else {
            return false;
        }
    }

    private <T extends BaseDigifiedResult> boolean isImageNull(String imageBase64, DigifiedCallback<T> digifiedCallback) {
        if (imageBase64 == null || imageBase64.isEmpty()) {
            Log.e(TAG, "Image file is null! make sure you captured an image first");
            digifiedCallback.onFailure(new DigifiedError(ErrorConstants.SendError.IMAGE_FILE_ERROR));
            return true;
        } else {
            return false;
        }
    }

    private <T extends BaseDigifiedResult> void onResponseError(boolean shouldTerminate, DigifiedCallback<T> digifiedCallback) {
        Log.e(TAG, "Response is null!");
        digifiedCallback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR, shouldTerminate));
    }

    private RequestBody createRequestBody(String text) {
        return RequestBody.create(MediaType.parse(ContentType.TEXT_PLAIN), text);
//        MediaType.parse(ContentType.TEXT_PLAIN)
    }


    private ProgressRequestBody createRequestBody(String imageBase64, ProgressUpdater progressUpdater) {
        return new ProgressRequestBody(imageBase64, ContentType.TEXT_PLAIN, progressUpdater);
    }


}
