package io.digified.digified_library.capture;

import android.content.Context;
import android.graphics.Bitmap;
import android.os.CountDownTimer;
import android.util.Log;

import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;

import java.util.ArrayList;

import io.digified.digified_library.capture.utils.CameraPermissionChecker;
import io.digified.digified_library.capture.utils.ImageConverter;
import io.digified.digified_library.capture.v2.display.CameraMLFragment;
import io.digified.digified_library.digified.DigifiedConstants.CaptureType;
import io.digified.digified_library.errors.DigifiedError;
import io.digified.digified_library.errors.ErrorConstants;

public class CaptureManager implements CaptureManagerFragmentInterface {

    private static final String TAG = "CaptureManager";

    private static final int MIN_ID_WIDTH = 800;
    private static final int MIN_ID_HEIGHT = 600;

    private static final int MIN_SELFIE_WIDTH = 700;
    private static final int MIN_SELFIE_HEIGHT = 900;

    private static final int DEFAULT_MAX_WIDTH = 1000;
    private static final int DEFAULT_MAX_HEIGHT = 1000;

    private static final int DEFAULT_CAPTURE_TIMEOUT_MILLIS = 15000;

    private final CameraMLFragment cameraMLFragment;


    private CaptureCallback captureCallback;
    private FragmentManager fragmentManager;
    private Fragment fragment;
    private final Context context;

    private int captureType;
    private int maxImageWidth;
    private int maxImageHeight;
    private int captureTimeoutMillis;
    private CountDownTimer timer;

    private boolean isCancelled;

    private boolean isManualCapturingFallbackEnabled;

    public CaptureManager(Context context) {
        this.context = context;
        cameraMLFragment = new CameraMLFragment();
        maxImageWidth = DEFAULT_MAX_WIDTH;
        maxImageHeight = DEFAULT_MAX_HEIGHT;
        captureTimeoutMillis = DEFAULT_CAPTURE_TIMEOUT_MILLIS;
        fragment = null;
        captureType = -1;
        isCancelled = false;
        isManualCapturingFallbackEnabled = false;
        initializeTimer();
    }

    public void setCaptureTimeout(int captureTimeoutMillis) {
        this.captureTimeoutMillis = captureTimeoutMillis;
        initializeTimer();
    }

    public void setMaxDimensions(int maxImageWidth, int maxImageHeight) {
        if (maxImageWidth < MIN_ID_WIDTH || maxImageWidth < MIN_SELFIE_WIDTH) {
            maxImageWidth = DEFAULT_MAX_WIDTH;
            Log.e(TAG, "max image width is less than minimums, new maxImageWidth is " + maxImageWidth);
            Log.e(TAG, "Minimums are: Minimum id width = " + MIN_ID_WIDTH + ", Minimum selfie width = " + MIN_SELFIE_WIDTH);

        }

        if (maxImageHeight < MIN_ID_HEIGHT || maxImageHeight < MIN_SELFIE_HEIGHT) {
            maxImageHeight = DEFAULT_MAX_HEIGHT;
            Log.e(TAG, "max image height is less than minimums, new maxImageHeight is " + maxImageHeight);
            Log.e(TAG, "Minimums are: Minimum id height = " + MIN_ID_HEIGHT + ", Minimum selfie height = " + MIN_ID_HEIGHT);
        }

        this.maxImageWidth = maxImageWidth;
        this.maxImageHeight = maxImageHeight;
    }


    //desired are treated as maximums, they are met if possible or less than them
    public void capture(FragmentManager fragmentManager,
                        int containerId,
                        int captureType,
                        int desiredWidth,
                        int desiredHeight,
                        @Nullable CaptureScreenMetaData captureScreenMetaData,
                        CaptureCallback captureCallback) {
        Log.d(TAG, "capture: captureType = " + captureType);
        this.captureCallback = captureCallback;
        this.fragmentManager = fragmentManager;
        this.captureType = captureType;

        startCapturingTimer();

        if (CameraPermissionChecker.isCameraPermissionGranted(context)) {

            if (desiredWidth > maxImageWidth) {
                desiredWidth = maxImageWidth;
                Log.e(TAG, "desired width is greater than max image width which is " + maxImageWidth + ", new desiredWidth is " + desiredWidth);
            }

            if (desiredHeight > maxImageHeight) {
                desiredHeight = maxImageHeight;
                Log.e(TAG, "desired height is greater than max image height which is " + maxImageHeight + ", new desiredHeight is " + desiredHeight);
            }

            if (captureType == CaptureType.SELFIE) {
                if (desiredWidth < MIN_SELFIE_WIDTH) {
                    desiredWidth = MIN_SELFIE_WIDTH;
                    Log.e(TAG, "desiredWidth is less than minimum selfie width which is " + MIN_SELFIE_WIDTH + ". new desiredWidth is " + desiredWidth);
                }
                if (desiredHeight < MIN_SELFIE_HEIGHT) {
                    desiredHeight = MIN_SELFIE_HEIGHT;
                    Log.e(TAG, "desiredHeight is less than minimum selfie height which is " + MIN_SELFIE_HEIGHT + ", new desiredHeight is " + desiredHeight);
                }
            } else {
                if (desiredWidth < MIN_ID_WIDTH) {
                    desiredWidth = MIN_ID_WIDTH;
                    Log.e(TAG, "desiredWidth is less than minimum id width which is " + MIN_ID_WIDTH + ", new desiredWidth is " + desiredWidth);
                }
                if (desiredHeight < MIN_ID_HEIGHT) {
                    desiredHeight = MIN_ID_HEIGHT;
                    Log.e(TAG, "desiredHeight is less than minimum id height which is " + MIN_ID_HEIGHT + ", new desiredHeight is " + desiredHeight);
                }
            }

            prepareCapturing(captureType, desiredWidth, desiredHeight, captureScreenMetaData);

            fragmentManager.beginTransaction().replace(containerId, fragment).commit();
        } else {
            captureCallback.onFailed(new DigifiedError(ErrorConstants.CaptureError.CAMERA_PERMISSION_ERROR), captureType);
        }
    }

    public void capture(FragmentManager fragmentManager,
                        int containerId,
                        int captureType,
                        @Nullable CaptureScreenMetaData captureScreenMetaData,
                        CaptureCallback captureCallback) {
        this.captureCallback = captureCallback;
        this.fragmentManager = fragmentManager;
        this.captureType = captureType;

        startCapturingTimer();

        if (CameraPermissionChecker.isCameraPermissionGranted(context)) {

            int width;
            int height;
            if (captureType == CaptureType.SELFIE) {
                width = MIN_SELFIE_WIDTH;
                height = MIN_SELFIE_HEIGHT;
            } else {
                width = MIN_ID_WIDTH;
                height = MIN_ID_HEIGHT;
            }

            prepareCapturing(captureType, width, height, captureScreenMetaData);

            fragmentManager.beginTransaction().replace(containerId, fragment).commit();
        } else {
            captureCallback.onFailed(new DigifiedError(ErrorConstants.CaptureError.CAMERA_PERMISSION_ERROR), captureType);
        }
    }

    private void prepareCapturing(int captureType,
                                  int width,
                                  int height,
                                  @Nullable CaptureScreenMetaData captureScreenMetaData) {
        fragment = cameraMLFragment;

        cameraMLFragment.prepare(captureType, width, height, captureScreenMetaData, this);
    }

    public void cancel() {
        cancelCapturing(captureType, false);
    }

    @Override
    public void onCaptured(CaptureResult captureResult) {
        stopCapturingTimer();
        if (fragmentManager != null && !fragmentManager.isDestroyed() && fragment != null) {
            fragmentManager.beginTransaction().remove(fragment).commit();
        }

        if (captureCallback != null) captureCallback.onCaptured(captureResult);
    }

    @Override
    public void onFailed(DigifiedError digifiedError, int captureType) {
        stopCapturingTimer();
        if (fragmentManager != null && !fragmentManager.isDestroyed() && fragment != null) {
            fragmentManager.beginTransaction().remove(fragment).commit();
        }

        if (captureCallback != null) captureCallback.onFailed(digifiedError, captureType);
    }

    @Override
    public void onCancelled(int captureType) {
        isCancelled = false;
        if (captureCallback != null) captureCallback.onCancelled(captureType);
    }

    @Override
    public void onTimeout() {
        if (isManualCapturingFallbackEnabled && cameraMLFragment != null)
            cameraMLFragment.switchToManualCapturingIfAuto();
        if (captureCallback != null) captureCallback.onTimeout();
    }

    @Override
    public void cancelCapturing(int captureType, boolean isOnPause) {
        if (!isCancelled) {
            isCancelled = true;
            stopCapturingTimer();
            if (fragmentManager != null && !fragmentManager.isDestroyed() && fragment != null) {
                Log.e(TAG, "removing fragment");
                fragmentManager.beginTransaction().remove(fragment).commit();
            }
        }
        if (isOnPause) {
            onCancelled(captureType);
        }
    }

    public ArrayList<String> buildBase64List(CaptureResult captureResult) {
        ArrayList<String> result = new ArrayList<>();
        for (Bitmap bitmap : captureResult.getCapturedFramesArrayList()) {
            result.add(bitmapToBase64ConsideringDimensions(bitmap));
        }
        return result;
    }

    public String bitmapToBase64ConsideringDimensions(Bitmap bitmap) {
        return ImageConverter.imageToBase64(bitmap, maxImageWidth, maxImageHeight);
    }

    private void initializeTimer() {
        timer = new CountDownTimer(captureTimeoutMillis, captureTimeoutMillis) {
            @Override
            public void onTick(long millisUntilFinished) {

            }

            @Override
            public void onFinish() {
                CaptureManager.this.onTimeout();
            }
        };
    }

    private void stopCapturingTimer() {
        timer.cancel();
    }

    private void startCapturingTimer() {
        timer.cancel();
        timer.start();
    }


    public void enableManualCapturingFallback() {
        isManualCapturingFallbackEnabled = true;
    }
}
