package io.digified.digified_library.capture.v2.camera;

import android.Manifest;
import android.content.Context;
import android.content.res.Configuration;
import android.util.Log;
import android.view.SurfaceHolder;
import android.view.SurfaceView;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresPermission;

import com.google.android.gms.common.images.Size;

import java.io.IOException;

public class CameraSourcePreview extends SurfaceView implements SurfaceHolder.Callback {

    private static final String TAG = "CameraSourcePreview";
    private CameraSource cameraSource;
    private final SurfaceHolder surfaceHolder;
    private boolean startRequested;
    private boolean surfaceAvailable;
    private Context context;
    private boolean onLayoutHappened;

    public CameraSourcePreview(Context context, CameraSource cameraSource) {
        super(context);
        this.context = context;

        this.cameraSource = cameraSource;
        this.onLayoutHappened = false;

        // Install a SurfaceHolder.Callback so we get notified when the
        // underlying surface is created and destroyed.
        surfaceHolder = getHolder();
        surfaceHolder.addCallback(this);
        // deprecated setting, but required on Android versions prior to 3.0
        surfaceHolder.setType(SurfaceHolder.SURFACE_TYPE_PUSH_BUFFERS);

        cameraSource.setSurfaceHolder(surfaceHolder);


    }


    public void start(CameraSource cameraSource) throws SecurityException {
        this.cameraSource = cameraSource;
        onLayoutHappened = false;

        if (this.cameraSource != null) {
            startRequested = true;
            startIfReady();
        }
    }

    @RequiresPermission(Manifest.permission.CAMERA)
    private void startIfReady() {
        if (startRequested && surfaceAvailable) {
            try {
                if (cameraSource != null) {
                    cameraSource.setSurfaceHolder(surfaceHolder);
                    cameraSource.start();
                }
            } catch (IOException e) {
                Log.e(TAG, "surfaceCreated: error setting camera preview: ", e);
            }
//            requestLayout();
            startRequested = false;
        }
    }

    @RequiresPermission(Manifest.permission.CAMERA)
    @Override
    public void surfaceCreated(@NonNull SurfaceHolder surfaceHolder) {
        surfaceAvailable = true;
//         The Surface has been created, now tell the camera where to draw the preview.
        try {
            if (cameraSource != null) {
                cameraSource.setSurfaceHolder(surfaceHolder);
                cameraSource.start();
            }
        } catch (IOException e) {
            Log.e(TAG, "surfaceCreated: error setting camera preview: ", e);
        }
    }

    @RequiresPermission(Manifest.permission.CAMERA)
    @Override
    public void surfaceChanged(@NonNull SurfaceHolder surfaceHolder, int format, int w, int h) {
        if (!onLayoutHappened) {
            onLayoutHappened = true;
            try {
                if (cameraSource != null) {

                    int width = 0;
                    int height = 0;
                    Size size = cameraSource.getPreviewSize();
                    if (size != null) {
                        width = size.getWidth();
                        height = size.getHeight();
                    }

                    if (isPortraitMode()) {
                        int tmp = width;
                        width = height;
                        height = tmp;
                    }

                    float previewAspectRatio = (float) width / height;
                    int layoutWidth = getRight() - getLeft();
                    int layoutHeight = getBottom() - getTop();
                    float layoutAspectRatio = (float) layoutWidth / layoutHeight;

                    if (previewAspectRatio > layoutAspectRatio) {
                        // The preview input is wider than the layout area. Fit the layout height and crop
                        // the preview input horizontally while keep the center.
                        int horizontalOffset = (int) (previewAspectRatio * layoutHeight - layoutWidth) / 2;
                        layout(-horizontalOffset, 0, layoutWidth + horizontalOffset, layoutHeight);
                    } else {
                        // The preview input is taller than the layout area. Fit the layout width and crop the preview
                        // input vertically while keep the center.
                        int verticalOffset = (int) (layoutWidth / previewAspectRatio - layoutHeight) / 2;
                        layout(0, -verticalOffset, layoutWidth, layoutHeight + verticalOffset);
                    }

                    cameraSource.stop();

                    cameraSource.setSurfaceHolder(surfaceHolder);
                    cameraSource.start();
                }
            } catch (Exception e) {
                Log.e(TAG, "surfaceCreated: error setting camera preview: ", e);
            }
        }
    }


    @Override
    public void surfaceDestroyed(@NonNull SurfaceHolder surfaceHolder) {
        surfaceAvailable = false;
        onLayoutHappened = false;
    }

    private boolean isPortraitMode() {
        int orientation = context.getResources().getConfiguration().orientation;
        if (orientation == Configuration.ORIENTATION_LANDSCAPE) {
            return false;
        } else if (orientation == Configuration.ORIENTATION_PORTRAIT) {
            return true;
        }
        return false;
    }

    public void stop() {
        if (cameraSource != null) {
            cameraSource.stop();
        }
        onLayoutHappened = false;
    }

    public void release() {
        if (cameraSource != null) {
            cameraSource.release();
            cameraSource = null;
        }
        onLayoutHappened = false;
        surfaceHolder.getSurface().release();
    }
}
