package io.digified.digified_library.capture.v2.detectors;

import android.graphics.Bitmap;
import android.graphics.Rect;
import android.os.CountDownTimer;

import java.util.ArrayList;

import io.digified.digified_library.R;
import io.digified.digified_library.capture.utils.ImageHandler;
import io.digified.digified_library.capture.v2.detectors.modules.BaseSelected;

public abstract class BaseDetectionSelector<T> {

    private static final String TAG = "BaseDetectionSelector";

    private static final int PROCESSING_DELAY_MS = 2000;
    private static final int INSTRUCTIONS_DELAY_MS = 2000;
    private static final int PREPARE_CAPTURING_DELAY_MS = 500;
    private static final int PREPARE_CAPTURING_INSTRUCTIONS_DELAY_MS = 2000;

    private final int numberOfImagesToDetect;

    private static final float CAMERA_FAR_THRESHOLD = 0.8f;

    private boolean isCaptureDelayDone;
    private boolean isCapturingDelayStarted;
    private boolean isBeginningDelayStarted;

    private boolean isPreparedForCapturing;

    private final DetectorInterface detectorInterface;
    private final int desiredWidth;
    private final int desiredHeight;

    protected final ArrayList<BaseSelected> capturedDetections;

    private boolean isLoadingTriggered;


    public BaseDetectionSelector(int desiredWidth,
                                 int desiredHeight,
                                 int numberOfImagesToCapture,
                                 DetectorInterface detectorInterface) {
        this.detectorInterface = detectorInterface;
        this.desiredWidth = desiredWidth;
        this.desiredHeight = desiredHeight;
        this.numberOfImagesToDetect = numberOfImagesToCapture;
        capturedDetections = new ArrayList<>();

        isCaptureDelayDone = false;
        isCapturingDelayStarted = false;
        isBeginningDelayStarted = false;

        isPreparedForCapturing = false;
        isLoadingTriggered = false;
    }

    public void check(T moduleResults, Bitmap cameraBitmap) {
        if (!isBeginningDelayStarted) {
            isBeginningDelayStarted = true;
            startCaptureDelay(PROCESSING_DELAY_MS);
        }
    }

    protected boolean isCaptureDelayDone() {
        return isCaptureDelayDone;
    }


    protected void displayInstructionsAndStopLoading(int instructionsId) {
        detectorInterface.onLoading(false);
        isLoadingTriggered = false;
        isPreparedForCapturing = false; // because this means that we displayed an instruction to the user
        if (startCaptureDelay(PROCESSING_DELAY_MS)) {
            detectorInterface.onInstructions(instructionsId, INSTRUCTIONS_DELAY_MS);
        }
    }

    /**
     * storing BaseDetected items in {@link #capturedDetections} ArrayList
     *
     * @param baseSelected: detections by the selector
     * @return if capturing done or not (reached the number of pics that we wnated to capture)
     */
    protected boolean capture(BaseSelected baseSelected) {
        capturedDetections.add(baseSelected);
        if (capturedDetections.size() < numberOfImagesToDetect) {
            isPreparedForCapturing = false;
            if (!isLoadingTriggered) {
                detectorInterface.onLoading(true);
                isLoadingTriggered = true;
            }
            detectorInterface.onInstructions(R.string.digified_taking_picture_hold_still, PREPARE_CAPTURING_INSTRUCTIONS_DELAY_MS);
            return false;
        } else {
            if (!isLoadingTriggered) {
                detectorInterface.onLoading(true);
                isLoadingTriggered = true;
                detectorInterface.onInstructions(R.string.digified_taking_picture_hold_still, PREPARE_CAPTURING_INSTRUCTIONS_DELAY_MS);
                return false;
            } else {
                return true;
            }
            //        Bitmap croppedBitmap = cropToDetection(originalBitmap, cropRect);
//            detectorInterface.onImageDetected(capturedImages.get(0));
        }
    }


    protected abstract BaseSelected selectBestDetection();

    /**
     * @return whether it has prepared or not
     */
    protected boolean prepareForCapturing() {
        if (!isPreparedForCapturing) {
            isPreparedForCapturing = true;
            startCaptureDelay(PREPARE_CAPTURING_DELAY_MS);
            return false;
        }
        return true;
    }

    /**
     * @return whether or not there is a capture delay running
     */
    private boolean startCaptureDelay(long duration) {
        if (!isCapturingDelayStarted) {
            isCapturingDelayStarted = true;
            isCaptureDelayDone = false;
            new CountDownTimer(duration, duration) {
                @Override
                public void onTick(long millisUntilFinished) {

                }

                @Override
                public void onFinish() {
                    isCaptureDelayDone = true;
                    isCapturingDelayStarted = false;
                }
            }.start();
            return true;
        }
        return false;
    }

    //checking if the user is close or far from the camera
    protected boolean isDistanceError(Rect cropRect) {
        //check the zoom
        int cropWidth = cropRect.width();
        int cropHeight = cropRect.height();

        //check if user should move away from the device
        if (cropWidth > desiredWidth && cropHeight > desiredHeight) {
            displayInstructionsAndStopLoading(R.string.digified_live_instruction_move_the_phone_away);
            return true;
        }

        float widthRatio = (float) cropWidth / desiredWidth;
        float heightRatio = (float) cropHeight / desiredHeight;

        //check if user should move close to the device
        if (widthRatio < CAMERA_FAR_THRESHOLD && heightRatio < CAMERA_FAR_THRESHOLD) {
            displayInstructionsAndStopLoading(R.string.digified_live_instruction_move_the_phone_closer);
            return true;
        }
        return false;
    }

    protected boolean isOutOfBoundaries(Bitmap bitmap, Rect cropRect) {

//        Log.e(TAG, "isOutOfBoundaries: cropRect.left = " + cropRect.left);
//        Log.e(TAG, "isOutOfBoundaries: cropRect.top = " + cropRect.top);
//        Log.e(TAG, "isOutOfBoundaries: cropRect.right = " + cropRect.right + "image width = " + bitmap.getWidth());
//        Log.e(TAG, "isOutOfBoundaries: cropRect.bottom = " + cropRect.bottom + "image width = " + bitmap.getHeight());

        Rect squareRect = ImageHandler.getSquareCroppingRect(bitmap.getWidth(), bitmap.getHeight(), false);

        if (cropRect.left <= 0 ||
                cropRect.top <= squareRect.top ||
                cropRect.right > bitmap.getWidth() ||
                cropRect.bottom > squareRect.bottom) {
            displayInstructionsAndStopLoading(R.string.digified_live_instruction_adjust_capturing_inside_frame);
//            Log.e(TAG, "isOutOfBoundaries: true!");
            return true;
        } else {
//            Log.e(TAG, "isOutOfBoundaries: false!");
            return false;
        }
    }

//    protected void cropToDetectionAndSend(BaseSelected baseSelected) {
    protected void cropToDetectionAndSend() {
//        Bitmap bitmap = baseSelected.getDetectionBitmap();
//        Rect cropRect = baseSelected.getCropRect();

//        Bitmap croppedBitmap = ImageHandler.cropImage(bitmap, cropRect);
//        Log.e(TAG, "==================================\n");

        ArrayList<Bitmap> bitmapArrayList = new ArrayList<>();

        for (BaseSelected detected : capturedDetections) {
            Bitmap temp = ImageHandler.cropImage(detected.getDetectionBitmap(), detected.getCropRect());
            bitmapArrayList.add(temp);
        }
        detectorInterface.onImageDetected(bitmapArrayList);
    }

    protected Rect getScaledRect(Rect rect,
                                 float horizontalCropFactor,
                                 float verticalCropFactor) {

        int x = rect.centerX();
        int y = rect.centerY();
        int width = rect.width();
        int height = rect.height();

//        Log.e(TAG, "getScaledRect: ================ ");
//
//        Log.e(TAG, "getScaledRect: x = " + x);
//        Log.e(TAG, "getScaledRect: y = " + y);
//        Log.e(TAG, "getScaledRect: width = " + width);
//        Log.e(TAG, "getScaledRect: height = " + height);

        int scaledLeft = (int) (x - (width * (1 + horizontalCropFactor) / 2));
        int scaledTop = (int) (y - (height * (1 + verticalCropFactor) / 2));

        int scaledRight = (int) (x + (width * (1 + horizontalCropFactor) / 2));
        int scaledBottom = (int) (y + (height * (1 + verticalCropFactor) / 2));

        //        Log.e(TAG, "getScaledRect: scaledRect x = " + scaledRect.centerX());
//        Log.e(TAG, "getScaledRect: scaledRect y = " + scaledRect.centerY());
//        Log.e(TAG, "getScaledRect: scaledRect width = " + scaledRect.width());
//        Log.e(TAG, "getScaledRect: scaledRect height = " + scaledRect.height());
//
//        Log.e(TAG, "getScaledRect: ================ ");
        return new Rect(scaledLeft, scaledTop, scaledRight, scaledBottom);

    }

    //crop to the exact size required
//    private Bitmap cropToDetectionAndSend(Bitmap cameraBitmap,
//                                          Rect cropRect) {


//        Log.e(TAG, "cropToDetection: cameraBitmap.width() = " + cameraBitmap.getWidth());
//        Log.e(TAG, "cropToDetection: cameraBitmap.height() = " + cameraBitmap.getHeight());
//
//
//        Log.e(TAG, "cropToDetection: cropRect.top = " + cropRect.top);
//        Log.e(TAG, "cropToDetection: cropRect.left = " + cropRect.left);
//        Log.e(TAG, "cropToDetection: cropRect.bottom = " + cropRect.bottom);
//        Log.e(TAG, "cropToDetection: cropRect.right = " + cropRect.right);

//        int x = cropRect.centerX();
//        int y = cropRect.centerY();
//
//        int left = (int) (x - (desiredWidth / 2.0f));
//        int top = (int) (y - (desiredHeight / 2.0f));
//        int right = (int) (x + (desiredWidth / 2.0f));
//        int bottom = (int) (y + (desiredHeight / 2.0f));
//
//        Rect rect = new Rect(left, top, right, bottom);

//        Log.e(TAG, "cropToDetection: cropRect.top = " + rect.top);
//        Log.e(TAG, "cropToDetection: cropRect.left = " + rect.left);
//        Log.e(TAG, "cropToDetection: cropRect.bottom = " + rect.bottom);
//        Log.e(TAG, "cropToDetection: cropRect.right = " + rect.right);

//        Log.e(TAG, "cropToDetection: actual cropRect.width() = " + rect.width());
//        Log.e(TAG, "cropToDetection: actual cropRect.height() = " + rect.height());

//        return ImageHandler.cropImage(cameraBitmap, cropRect);
//    }

}
