/*
 * Copyright 2020 Google LLC. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.digified.digified_library.capture.v2.detectors;

import static java.lang.Math.max;
import static java.lang.Math.min;

import android.app.ActivityManager;
import android.content.Context;
import android.graphics.Bitmap;
import android.os.SystemClock;

import androidx.annotation.GuardedBy;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.android.gms.tasks.Task;
import com.google.android.gms.tasks.TaskExecutors;

import java.nio.ByteBuffer;
import java.util.Timer;
import java.util.TimerTask;

import io.digified.digified_library.capture.v2.utils.BitmapUtils;
import io.digified.digified_library.capture.v2.utils.FrameMetadata;
//import io.digified.digified_library.capture.v2.utils.TemperatureMonitor;

/**
 * Abstract base class for vision frame processors. Subclasses need to implement {@link
// * #onSuccess(Object, Bitmap, GraphicOverlay)} to define what they want to with the detection results and
 * {link #detectInImage(InputImage)} to specify the detector object.
 *
 * @param <T> The type of the detected feature.
 */
public abstract class VisionProcessorBase<T> implements VisionImageProcessor {

    protected static final String MANUAL_TESTING_LOG = "LogTagForTest";
    private static final String TAG = "VisionProcessorBase";

    protected final Context context;

    private final ActivityManager activityManager;
    private final Timer fpsTimer = new Timer();
    private final ScopedExecutor executor;
//    private final TemperatureMonitor temperatureMonitor;

    // Whether this processor is already shut down
    private boolean isShutdown;

    // Used to calculate latency, running in the same thread, no sync needed.
    private int numRuns = 0;
    private long totalFrameMs = 0;
    private long maxFrameMs = 0;
    private long minFrameMs = Long.MAX_VALUE;
    private long totalDetectorMs = 0;
    private long maxDetectorMs = 0;
    private long minDetectorMs = Long.MAX_VALUE;

    // Frame count that have been processed so far in an one second interval to calculate FPS.
    private int frameProcessedInOneSecondInterval = 0;
    private int framesPerSecond = 0;

    // To keep the latest images and its metadata.
    @GuardedBy("this")
    private ByteBuffer latestImage;

    @GuardedBy("this")
    private FrameMetadata latestImageMetaData;
    // To keep the images and metadata in process.
    @GuardedBy("this")
    private ByteBuffer processingImage;

    @GuardedBy("this")
    private FrameMetadata processingMetaData;

    protected DetectorInterface detectorInterface;

    protected int numberOfImages;

    protected VisionProcessorBase(Context context,
                                  int numberOfImages,
                                  DetectorInterface detectorInterface) {
        this.context = context;
        this.numberOfImages = numberOfImages;
        this.detectorInterface = detectorInterface;
        activityManager = (ActivityManager) context.getSystemService(Context.ACTIVITY_SERVICE);
        executor = new ScopedExecutor(TaskExecutors.MAIN_THREAD);
        fpsTimer.scheduleAtFixedRate(
                new TimerTask() {
                    @Override
                    public void run() {
                        framesPerSecond = frameProcessedInOneSecondInterval;
                        frameProcessedInOneSecondInterval = 0;
                    }
                },
                /* delay= */ 0,
                /* period= */ 1000);
//        temperatureMonitor = new TemperatureMonitor(context);
    }

    // -----------------Code for processing live preview frame from Camera1 API-----------------------
    @Override
    public synchronized void processByteBuffer(
            ByteBuffer data, final FrameMetadata frameMetadata) {

//        Log.e(TAG, "processByteBuffer:: byteBuffer: position = " + data.position());
//        Log.e(TAG, "processByteBuffer:: byteBuffer: limit = " + data.limit());


        latestImage = data;
        latestImageMetaData = frameMetadata;
        if (processingImage == null && processingMetaData == null) {
            processLatestImage();
        }
    }

    private synchronized void processLatestImage() {
        processingImage = latestImage;
        processingMetaData = latestImageMetaData;
        latestImage = null;
        latestImageMetaData = null;
        if (processingImage != null && processingMetaData != null && !isShutdown) {
            processImage(processingImage, processingMetaData);
        }
    }

    private void processImage(
            ByteBuffer data, final FrameMetadata frameMetadata) {
        long frameStartMs = SystemClock.elapsedRealtime();

        Bitmap bitmap = BitmapUtils.getBitmap(data, frameMetadata);

        requestDetectInImage(bitmap,
                /* shouldShowFps= */ true,
                frameStartMs)
                .addOnSuccessListener(executor, results -> processLatestImage());
    }



    // -----------------Common processing logic-------------------------------------------------------
    private Task<T> requestDetectInImage(
            @Nullable final Bitmap originalCameraImage,
            boolean shouldShowFps,
            long frameStartMs) {
        return setUpListener(
                detectInImage(originalCameraImage), originalCameraImage, shouldShowFps, frameStartMs);
    }


    private Task<T> setUpListener(
            Task<T> task,
            @Nullable final Bitmap originalCameraImage,
            boolean shouldShowFps,
            long frameStartMs) {
        final long detectorStartMs = SystemClock.elapsedRealtime();
        return task.addOnSuccessListener(
                executor,
                results -> {
                    long endMs = SystemClock.elapsedRealtime();
                    long currentFrameLatencyMs = endMs - frameStartMs;
                    long currentDetectorLatencyMs = endMs - detectorStartMs;
                    if (numRuns >= 500) {
                        resetLatencyStats();
                    }
                    numRuns++;
                    frameProcessedInOneSecondInterval++;
                    totalFrameMs += currentFrameLatencyMs;
                    maxFrameMs = max(currentFrameLatencyMs, maxFrameMs);
                    minFrameMs = min(currentFrameLatencyMs, minFrameMs);
                    totalDetectorMs += currentDetectorLatencyMs;
                    maxDetectorMs = max(currentDetectorLatencyMs, maxDetectorMs);
                    minDetectorMs = min(currentDetectorLatencyMs, minDetectorMs);

                    // Only log inference info once per second. When frameProcessedInOneSecondInterval is
                    // equal to 1, it means this is the first frame processed during the current second.
                    if (frameProcessedInOneSecondInterval == 1) {
//                    Log.e("PERFORMANCE", "framesPerSecond = " + framesPerSecond);

//                        Log.e("PERFORMANCE", "--------------");
//                        Log.e("PERFORMANCE", "Num of Runs: " + numRuns);
//                        Log.e(
//                                "PERFORMANCE",
//                                "Frame latency: max="
//                                        + maxFrameMs
//                                        + ", min="
//                                        + minFrameMs
//                                        + ", avg="
//                                        + totalFrameMs / numRuns);
//                        Log.e(
//                                "PERFORMANCE",
//                                "Detector latency: max="
//                                        + maxDetectorMs
//                                        + ", min="
//                                        + minDetectorMs
//                                        + ", avg="
//                                        + totalDetectorMs / numRuns);
//                        Log.e("PERFORMANCE", "--------------");

//                        MemoryInfo mi = new MemoryInfo();
//                        activityManager.getMemoryInfo(mi);
//                        long availableMegs = mi.availMem / 0x100000L;
////                        Log.d(TAG, "Memory available in system: " + availableMegs + " MB");
//                        temperatureMonitor.logTemperature();
                    }

//                    graphicOverlay.clear();
//                    if (originalCameraImage != null) {
                        //here is where the frame is displayed!
//                        graphicOverlay.add(new CameraImageGraphic(graphicOverlay, originalCameraImage));
//                    }
                    VisionProcessorBase.this.onSuccess(results, originalCameraImage);

//                    Log.e("PERFORMANCE", "currentFrameLatencyMs = " + currentFrameLatencyMs);
//                    Log.e("PERFORMANCE", "currentDetectorLatencyMs = " + currentDetectorLatencyMs);
//                    Log.e("PERFORMANCE", "framesPerSecond = " + framesPerSecond);

//                    if (!PreferenceUtils.shouldHideDetectionInfo(graphicOverlay.getContext())) {
//                    graphicOverlay.add(
//                            new InferenceInfoGraphic(
//                                    graphicOverlay,
//                                    currentFrameLatencyMs,
//                                    currentDetectorLatencyMs,
//                                    shouldShowFps ? framesPerSecond : null));
////                    }
//                    graphicOverlay.postInvalidate();
                })
                .addOnFailureListener(
                        executor,
                        e -> {
//                            graphicOverlay.clear();
//                            graphicOverlay.postInvalidate();
                            String error = "Failed to process. Error: " + e.getLocalizedMessage();
//                            Toast.makeText(
//                                    graphicOverlay.getContext(),
//                                    error + "\nCause: " + e.getCause(),
//                                    Toast.LENGTH_SHORT)
//                                    .show();
////                            Log.d(TAG, error);
                            e.printStackTrace();
                            VisionProcessorBase.this.onFailure(e);
                        });
    }

    @Override
    public void stop() {
        executor.shutdown();
        isShutdown = true;
        resetLatencyStats();
        fpsTimer.cancel();
//        temperatureMonitor.stop();
    }

    private void resetLatencyStats() {
        numRuns = 0;
        totalFrameMs = 0;
        maxFrameMs = 0;
        minFrameMs = Long.MAX_VALUE;
        totalDetectorMs = 0;
        maxDetectorMs = 0;
        minDetectorMs = Long.MAX_VALUE;
    }

    protected abstract Task<T> detectInImage(Bitmap image);


    protected abstract void onSuccess(@NonNull T results, @NonNull Bitmap originalCameraImage);

    protected abstract void onFailure(@NonNull Exception e);

//    protected boolean isMlImageEnabled(Context context) {
//        return false;
//    }

}
