package io.digified.digified_library.capture.v2.display;

import android.graphics.Bitmap;
import android.graphics.drawable.Animatable;
import android.os.Bundle;
import android.os.CountDownTimer;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.ProgressBar;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;

import java.util.ArrayList;

import io.digified.digified_library.R;
import io.digified.digified_library.capture.CaptureManagerFragmentInterface;
import io.digified.digified_library.capture.CaptureResult;
import io.digified.digified_library.capture.CaptureScreenMetaData;
import io.digified.digified_library.capture.v2.detectors.DetectorInterface;
import io.digified.digified_library.digified.DigifiedConstants.CaptureType;
import io.digified.digified_library.errors.DigifiedError;

public class CameraMLFragment extends Fragment implements DetectorInterface {

    private static final String TAG = "CameraMLFragment";

    private TextView titleTextView;
    private TextView subtitleTextView;
    private ImageButton cancelCapturingImageButton;
    private TextView liveInstructionsTextView;
    private ImageView markerImageView;

    private ImageButton captureImageButton;
    private ProgressBar progressBar;

    private CameraMLViewModel cameraMLViewModel;
    private FrameLayout previewFrameLayout;

    private CountDownTimer instructionsTimer;
    private boolean isInstructionsTimerRunning;

    private CaptureManagerFragmentInterface captureManagerFragmentInterface;
    private Integer captureType;
    private Integer desiredWidth;
    private Integer desiredHeight;
    private boolean isCaptured;

    private CaptureScreenMetaData captureScreenMetaData;

    public CameraMLFragment() {
    }

    public void prepare(Integer captureType, Integer desiredWidth, Integer desiredHeight, @Nullable CaptureScreenMetaData captureScreenMetaData, CaptureManagerFragmentInterface captureManagerFragmentInterface) {
        this.captureManagerFragmentInterface = captureManagerFragmentInterface;
        this.desiredWidth = desiredWidth;
        this.desiredHeight = desiredHeight;
        this.captureType = captureType;
        this.captureScreenMetaData = captureScreenMetaData;
        this.isCaptured = false;
    }

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {

        View rootView = inflater.inflate(R.layout.ml_camera_fragment, container, false);

        initViews(rootView);

        initVariables();

        setListeners();

        return rootView;
    }

    private void adjustDisplay(DisplayModel displayModel) {
        if (displayModel == null) return;
        titleTextView.setText(displayModel.getTitleId());
        subtitleTextView.setText(displayModel.getSubtitleId());

        switch (displayModel.getDisplayType()) {
            case DisplayModel.DisplayType.DOCUMENT_SMALL:
                markerImageView.setImageResource(R.drawable.id_mark_with_animation);
                break;
            case DisplayModel.DisplayType.SELFIE:
                markerImageView.setImageResource(R.drawable.face_mark_with_animation_square);
                break;
        }

        if (displayModel.isCancelButtonEnabled()) {
            cancelCapturingImageButton.setVisibility(View.VISIBLE);
        } else {
            cancelCapturingImageButton.setVisibility(View.GONE);
        }

        if (displayModel.isShouldUseML()) {
            captureImageButton.setVisibility(View.GONE);
        } else {
            captureImageButton.setVisibility(View.VISIBLE);
        }
    }

    private void setListeners() {
        captureImageButton.setOnClickListener(v -> {
            if (cameraMLViewModel != null) {
                cameraMLViewModel.manualCaptureRequested();
                captureImageButton.setVisibility(View.GONE);
                progressBar.setVisibility(View.VISIBLE);
            }
        });

        cancelCapturingImageButton.setOnClickListener(v -> {
            if (captureManagerFragmentInterface != null && cameraMLViewModel != null) {
                captureManagerFragmentInterface.cancelCapturing(captureType, false);
                cameraMLViewModel.onPause();
                stopInstructionsTimer();
            }
        });
    }

    private void initViews(View rootView) {
        progressBar = rootView.findViewById(R.id.ml_progress_bar);
        captureImageButton = rootView.findViewById(R.id.ml_capture_button);
        previewFrameLayout = rootView.findViewById(R.id.ml_preview_frame_layout);
        titleTextView = rootView.findViewById(R.id.ml_camera_instructions_title_text_view);
        subtitleTextView = rootView.findViewById(R.id.ml_camera_instructions_subtitle_text_view);
        cancelCapturingImageButton = rootView.findViewById(R.id.ml_cancel_capture_image_button);
        liveInstructionsTextView = rootView.findViewById(R.id.ml_camera_live_instructions_text_view);
        markerImageView = rootView.findViewById(R.id.ml_marker_image_view);
    }


    private void initVariables() {
        isInstructionsTimerRunning = false;
        cameraMLViewModel = new CameraMLViewModel(requireActivity(), previewFrameLayout, this);
    }


    @Override
    public void onResume() {
        super.onResume();
        cameraMLViewModel.setParams(captureType, desiredWidth, desiredHeight);
        cameraMLViewModel.setCaptureScreenMetaData(captureScreenMetaData);

        DisplayModel displayModel = cameraMLViewModel.prepareCameraSource();
        adjustDisplay(displayModel);

        cameraMLViewModel.onResume();
    }

    @Override
    public void onPause() {
        super.onPause();
//        Log.e(TAG, "onPause: ");
        if (!isCaptured && captureManagerFragmentInterface != null) {
            captureManagerFragmentInterface.cancelCapturing(captureType, true);
        }
        if (cameraMLViewModel != null) {
            cameraMLViewModel.onPause();
        }
        stopInstructionsTimer();
    }


    @Override
    public void onStop() {
        super.onStop();
        Log.e(TAG, "onStop: ");
        if (cameraMLViewModel != null) {
            cameraMLViewModel.onStop();
        }
        stopInstructionsTimer();
    }


    @Override
//    public void onImageDetected(Bitmap bestDetectedFrame, ArrayList<Bitmap> allDetectedFrames) {
    public void onImageDetected(ArrayList<Bitmap> allDetectedFrames) {
        isCaptured = true;
        stopInstructionsTimer();
        captureManagerFragmentInterface.onCaptured(new CaptureResult(captureType, allDetectedFrames));
    }

    @Override
    public void onLoading(boolean isLoading) {
        Animatable animatable = (Animatable) markerImageView.getDrawable();
        if (isLoading) {
            animatable.start();
        } else {
            animatable.stop();
            stopInstructionsTimer();
            if (captureType == CaptureType.SELFIE) {
                markerImageView.setImageResource(R.drawable.face_mark_with_animation_square);
            } else {
                markerImageView.setImageResource(R.drawable.id_mark_with_animation);
            }
        }
    }

    @Override
    public void onInstructions(int instructionId, long durationMillis) {
        if (!isInstructionsTimerRunning) {
            isInstructionsTimerRunning = true;
            liveInstructionsTextView.setText(instructionId);

            instructionsTimer = new CountDownTimer(durationMillis, durationMillis) {
                @Override
                public void onTick(long millisUntilFinished) {

                }

                @Override
                public void onFinish() {
                    liveInstructionsTextView.setText("");
                    isInstructionsTimerRunning = false;
                }
            };
            instructionsTimer.start();
        }
    }

    @Override
    public void onStickyInstruction(int instructionId) {
        liveInstructionsTextView.setText(instructionId);
    }

    @Override
    public void onError(DigifiedError digifiedError) {
        captureManagerFragmentInterface.onFailed(digifiedError, captureType);
        stopInstructionsTimer();
    }

    private void stopInstructionsTimer() {
        if (instructionsTimer != null) instructionsTimer.cancel();
        isInstructionsTimerRunning = false;
    }

    public void switchToManualCapturingIfAuto() {
        if (cameraMLViewModel != null) {
            cameraMLViewModel.switchToManualCapturingIfAuto();
            captureImageButton.setVisibility(View.VISIBLE);
        }
    }
}
