package io.digified.digified_library.digified;

import android.content.Context;
import android.graphics.Bitmap;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.FragmentManager;

import io.digified.digified_library.api.ApiConstants;
import io.digified.digified_library.api.ApiManager;
import io.digified.digified_library.api.callbacks.DigifiedCallback;
import io.digified.digified_library.api.callbacks.FaceMatchCallback;
import io.digified.digified_library.api.callbacks.GenericIdCompletedVerificationCallback;
import io.digified.digified_library.api.callbacks.GenericIdExtractionCallback;
import io.digified.digified_library.api.callbacks.IdCompletedVerificationCallback;
import io.digified.digified_library.api.callbacks.IdExtractionCallback;
import io.digified.digified_library.api.callbacks.PassportCompletedVerificationCallback;
import io.digified.digified_library.api.callbacks.PassportExtractionCallback;
import io.digified.digified_library.api.callbacks.TransliterationCallback;
import io.digified.digified_library.api.callbacks.VehicleLicenseExtractionCallback;
import io.digified.digified_library.api.result.DigifiedResult;
import io.digified.digified_library.api.result.FaceMatchResult;
import io.digified.digified_library.api.result.GenericIdExtractionResult;
import io.digified.digified_library.api.result.IdExtractionResult;
import io.digified.digified_library.api.result.InitializationResult;
import io.digified.digified_library.api.result.PassportExtractionResult;
import io.digified.digified_library.api.result.TransliterationResult;
import io.digified.digified_library.api.result.VehicleLicenseExtractionResult;
import io.digified.digified_library.authorize.DigifiedAuthorizationCallback;
import io.digified.digified_library.authorize.DigifiedAuthorizationManager;
import io.digified.digified_library.capture.CaptureCallback;
import io.digified.digified_library.capture.CaptureManager;
import io.digified.digified_library.capture.CaptureResult;
import io.digified.digified_library.capture.CaptureScreenMetaData;
import io.digified.digified_library.errors.DigifiedError;
import io.digified.digified_library.errors.ErrorConstants;

class DigifiedInternal extends MainDigified {
    private static final String TAG = "Digified";

    private final ApiManager apiManager;
    private final CaptureManager captureManager;
    private final SequenceManager sequenceManager;

    private final DigifiedAuthorizationManager digifiedAuthorizationManager;

    private int currentAction;
    private int finalStatus;
    private Boolean apiIsSuccess;

    DigifiedInternal(Context context, String apiKey, String baseUrl) {
        apiManager = ApiManager.getInstance(context, apiKey, baseUrl);
        captureManager = new CaptureManager(context);
        sequenceManager = new SequenceManager();
        digifiedAuthorizationManager = new DigifiedAuthorizationManager(apiManager, captureManager);
    }

    void updateApiKey(String apiKey) {
        apiManager.updateApiKey(apiKey);
        sequenceManager.reset();
        resetSequence();
    }

    int getCurrentAction() {
        return currentAction;
    }

    int getFinalStatus() {
        return finalStatus;
    }

    void resetSequence() {
        setCurrentAction(DigifiedConstants.Action.INITIALIZE);
        setFinalStatus(DigifiedConstants.Status.UNKNOWN);
        reset();
        setApiIsSuccess(null);
    }

    void enableManualCapturingFallback() {
        captureManager.enableManualCapturingFallback();
    }

    void setMaxDimensions(int maxWidth, int maxHeight) {
        captureManager.setMaxDimensions(maxWidth, maxHeight);
    }

    void setCaptureTimeout(int captureTimeoutMillis) {
        captureManager.setCaptureTimeout(captureTimeoutMillis);
    }

    void initialize(@Nullable String warningAction, @Nullable String userName, @Nullable String phoneNumber, @Nullable String email, @NonNull DigifiedCallback<InitializationResult> callback) {
        if (sequenceManager.isEligibleForAction(DigifiedConstants.Action.INITIALIZE)) {
            apiManager.initialize(warningAction, userName, phoneNumber, email, new DigifiedCallback<>() {
                @Override
                public void onResult(InitializationResult result) {
                    updateStatus(result);
                    setApiIsSuccess(result.getError() == null);
                    setCurrentAction(selectNextAction(getApiIsSuccess()));
                    biometricToken = result.getToken();
                    callback.onResult(result);
                }

                @Override
                public void onFailure(DigifiedError digifiedError) {
                    setApiIsSuccess(false);
                    setCurrentAction(selectNextAction(false));
                    callback.onFailure(digifiedError);
                }
            });
        } else {
            Log.e(TAG, "Follow the sequence of the Digified SDK");
            callback.onFailure(new DigifiedError(ErrorConstants.CaptureError.SEQUENCE_ERROR));
        }
    }

    void capture(FragmentManager fragmentManager, int containerId, int captureType, @Nullable CaptureScreenMetaData captureScreenMetaData, CaptureCallback captureCallback) {
        if (isEligibleForAction(captureTypeToAction(captureType))) {
            captureManager.capture(fragmentManager, containerId, captureType, captureScreenMetaData, new CaptureCallback() {
                @Override
                public void onCaptured(CaptureResult captureResult) {
                    setApiIsSuccess(true);
                    setCurrentAction(selectNextAction(true));
                    captureCallback.onCaptured(captureResult);
                }

                @Override
                public void onFailed(DigifiedError digifiedError, int captureType) {
                    setApiIsSuccess(false);
                    setCurrentAction(selectNextAction(false));
                    captureCallback.onFailed(digifiedError, captureType);
                }

                @Override
                public void onCancelled(int captureType) {
                    setApiIsSuccess(false);
                    setCurrentAction(selectNextAction(false));
                    captureCallback.onCancelled(captureType);
                }

                @Override
                public void onTimeout() {
                    captureCallback.onTimeout();
                }
            });
        } else {
            Log.e(TAG, "Follow the sequence of the Digified SDK");
            captureCallback.onFailed(new DigifiedError(ErrorConstants.CaptureError.SEQUENCE_ERROR), captureType);
        }
    }

    void capture(FragmentManager fragmentManager, int containerId, int captureType, int desiredWidth, int desiredHeight, @Nullable CaptureScreenMetaData captureScreenMetaData, CaptureCallback captureCallback) {
        if (isEligibleForAction(captureTypeToAction(captureType))) {
            captureManager.capture(fragmentManager, containerId, captureType, desiredWidth, desiredHeight, captureScreenMetaData, new CaptureCallback() {
                @Override
                public void onCaptured(CaptureResult captureResult) {
                    setApiIsSuccess(true);
                    setCurrentAction(selectNextAction(true));
                    captureCallback.onCaptured(captureResult);
                }

                @Override
                public void onFailed(DigifiedError digifiedError, int captureType) {
                    setApiIsSuccess(false);
                    setCurrentAction(selectNextAction(false));
                    captureCallback.onFailed(digifiedError, captureType);
                }

                @Override
                public void onCancelled(int captureType) {
                    setApiIsSuccess(false);
                    setCurrentAction(selectNextAction(false));
                    captureCallback.onCancelled(captureType);
                }

                @Override
                public void onTimeout() {
                    captureCallback.onTimeout();
                }
            });
        } else {
            Log.e(TAG, "Follow the sequence of the Digified SDK");
            captureCallback.onFailed(new DigifiedError(ErrorConstants.CaptureError.SEQUENCE_ERROR), captureType);
        }
    }

    void cancel() {
        captureManager.cancel();
    }

    void send(CaptureResult captureResult, IdExtractionCallback idExtractionCallback) {
        if (captureResult != null) {
            switch (captureResult.getCaptureType()) {
                case DigifiedConstants.CaptureType.ID_FRONT:
                    sendFrontId(captureResult, idExtractionCallback);
                    return;
                case DigifiedConstants.CaptureType.ID_BACK:
                    sendBackId(captureResult, idExtractionCallback);
                    return;
                default:
                    Log.e(TAG, "used invalid capture type, make sure you are using the capture " + "type that is DigifiedConstants.CaptureType.ID_FRONT or DigifiedConstants.CaptureType.ID_BACK");
                    idExtractionCallback.onFailure(new DigifiedError(ErrorConstants.SendError.INVALID_CAPTURE_TYPE));
            }
        } else {
            Log.e(TAG, "capture result is null");
            idExtractionCallback.onFailure(new DigifiedError(ErrorConstants.SendError.CAPTURE_RESULT_NULL));
        }
    }

    void send(CaptureResult captureResult, GenericIdExtractionCallback genericIdExtractionCallback) {
        if (captureResult != null) {
            switch (captureResult.getCaptureType()) {
                case DigifiedConstants.CaptureType.Generic_ID_FRONT:
                    sendFrontGenericId(captureResult, genericIdExtractionCallback);
                    return;
                case DigifiedConstants.CaptureType.Generic_ID_BACK:
                    sendBackGenericId(captureResult, genericIdExtractionCallback);
                    return;
                default:
                    Log.e(TAG, "used invalid capture type, make sure you are using the capture type that is" + " DigifiedConstants.CaptureType.Generic_ID_FRONT or DigifiedConstants.CaptureType.Generic_ID_BACK");
                    genericIdExtractionCallback.onFailure(new DigifiedError(ErrorConstants.SendError.INVALID_CAPTURE_TYPE));
            }
        } else {
            Log.e(TAG, "capture result is null");
            genericIdExtractionCallback.onFailure(new DigifiedError(ErrorConstants.SendError.CAPTURE_RESULT_NULL));
        }
    }

    void send(CaptureResult captureResult, VehicleLicenseExtractionCallback vehicleLicenseExtractionCallback) {
        if (captureResult != null) {
            switch (captureResult.getCaptureType()) {
                case DigifiedConstants.CaptureType.VEHICLE_LICENSE_FRONT:
                    sendVehicleLicenseFront(captureResult, vehicleLicenseExtractionCallback);
                    return;
                case DigifiedConstants.CaptureType.VEHICLE_LICENSE_BACK:
                    sendVehicleLicenseBack(captureResult, vehicleLicenseExtractionCallback);
                    return;
                default:
                    Log.e(TAG, "used invalid capture type, make sure you are using the capture " + "type that is DigifiedConstants.CaptureType.VEHICLE_LICENSE_FRONT or DigifiedConstants.CaptureType.VEHICLE_LICENSE_BACK");
                    vehicleLicenseExtractionCallback.onFailure(new DigifiedError(ErrorConstants.SendError.INVALID_CAPTURE_TYPE));
            }
        } else {
            Log.e(TAG, "capture result is null");
            vehicleLicenseExtractionCallback.onFailure(new DigifiedError(ErrorConstants.SendError.CAPTURE_RESULT_NULL));
        }
    }

    void send(CaptureResult captureResult, FaceMatchCallback faceMatchCallback) {
        if (captureResult != null) {
            if (captureResult.getCaptureType() == DigifiedConstants.CaptureType.SELFIE) {
                sendSelfie(captureResult, faceMatchCallback);
            } else {
                Log.e(TAG, "used invalid capture type, make sure you are using the capture type that is DigifiedConstants.CaptureType.SELFIE");
                faceMatchCallback.onFailure(new DigifiedError(ErrorConstants.SendError.INVALID_CAPTURE_TYPE));
            }
        } else {
            Log.e(TAG, "capture result is null");
            faceMatchCallback.onFailure(new DigifiedError(ErrorConstants.SendError.CAPTURE_RESULT_NULL));
        }
    }

    void send(CaptureResult captureResult, PassportExtractionCallback passportExtractionCallback) {
        if (captureResult != null) {
            if (captureResult.getCaptureType() == DigifiedConstants.CaptureType.PASSPORT) {
                sendPassport(captureResult, passportExtractionCallback);
            } else {
                Log.e(TAG, "used invalid capture type, make sure you are using the capture type that is DigifiedConstants.CaptureType.PASSPORT");
                passportExtractionCallback.onFailure(new DigifiedError(ErrorConstants.SendError.INVALID_CAPTURE_TYPE));
            }
        } else {
            Log.e(TAG, "capture result is null");
            passportExtractionCallback.onFailure(new DigifiedError(ErrorConstants.SendError.CAPTURE_RESULT_NULL));
        }
    }

    void fetchResult(String token, IdCompletedVerificationCallback idCompletedVerificationCallback) {
        if (token != null) {
            apiManager.fetchIdResult(token, idCompletedVerificationCallback);
        } else {
            idCompletedVerificationCallback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));
        }
    }

    void fetchResult(String token, PassportCompletedVerificationCallback passportCompletedVerificationCallback) {
        if (token != null) {
            apiManager.fetchPassportResult(token, passportCompletedVerificationCallback);
        } else {
            passportCompletedVerificationCallback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));
        }
    }

    void fetchResult(String token, GenericIdCompletedVerificationCallback genericIdCompletedVerificationCallback) {
        if (token != null) {
            apiManager.fetchGenericIdResult(token, genericIdCompletedVerificationCallback);
        } else {
            genericIdCompletedVerificationCallback.onFailure(new DigifiedError(ErrorConstants.SendError.RESPONSE_ERROR));
        }
    }

    void transliterate(TransliterationCallback transliterationCallback) {
        if (sequenceManager.isEligibleForAction(DigifiedConstants.Action.TRANSLITERATE)) {

            apiManager.transliterate(new TransliterationCallback() {
                @Override
                public void onResult(TransliterationResult transliterationResult) {
                    sequenceManager.updateStatus(transliterationResult);
                    apiIsSuccess = transliterationResult.getError() == null;
                    currentAction = sequenceManager.selectNextAction(apiIsSuccess);
                    updateFinalResult(transliterationResult.getOverallStatus());
                    transliterationCallback.onResult(transliterationResult);
                }

                @Override
                public void onFailure(DigifiedError digifiedError) {
                    sequenceManager.updateStatus(digifiedError);
                    apiIsSuccess = false;
                    currentAction = sequenceManager.selectNextAction(false);
                    transliterationCallback.onFailure(digifiedError);
                }
            });

        } else {
            Log.e(TAG, "Follow the sequence of the Digified SDK");
            transliterationCallback.onFailure(new DigifiedError(ErrorConstants.SendError.SEQUENCE_ERROR));
        }
    }

    private void setCurrentAction(int currentAction) {
        this.currentAction = currentAction;
    }

    private void setFinalStatus(int finalStatus) {
        this.finalStatus = finalStatus;
    }

    private Boolean getApiIsSuccess() {
        return apiIsSuccess;
    }

    private void setApiIsSuccess(Boolean apiIsSuccess) {
        this.apiIsSuccess = apiIsSuccess;
    }


    private void reset() {
        sequenceManager.reset();
    }

    private void updateStatus(DigifiedResult result) {
        sequenceManager.updateStatus(result);
    }

    private int selectNextAction(Boolean apiIsSuccess) {
        return sequenceManager.selectNextAction(apiIsSuccess);
    }

    private boolean isEligibleForAction(int action) {
        return sequenceManager.isEligibleForAction(action);
    }

    private void sendFrontId(CaptureResult captureResult, @NonNull IdExtractionCallback callback) {

        if (sequenceManager.isEligibleForAction(DigifiedConstants.Action.SEND_DOCUMENT_FIRST)) {

            apiManager.extractFrontId(buildBase64Image(captureResult), new IdExtractionCallback() {
                @Override
                public void onResult(IdExtractionResult idExtractionResult) {
                    sequenceManager.updateStatus(idExtractionResult);
                    apiIsSuccess = idExtractionResult.getError() == null;
                    currentAction = sequenceManager.selectNextAction(apiIsSuccess);
                    updateFinalResult(idExtractionResult.getOverallStatus());
                    callback.onResult(idExtractionResult);
                }

                @Override
                public void onUploadProgress(int captureType, int progress) {
                    callback.onUploadProgress(captureType, progress);
                }

                @Override
                public void onUploadingDone(int captureType) {
                    callback.onUploadingDone(captureType);
                }

                @Override
                public void onFailure(DigifiedError digifiedError) {
                    sequenceManager.updateStatus(digifiedError);
                    apiIsSuccess = false;
                    currentAction = sequenceManager.selectNextAction(false);
                    callback.onFailure(digifiedError);
                }
            });

        } else {
            Log.e(TAG, "Follow the sequence of the Digified SDK");
            callback.onFailure(new DigifiedError(ErrorConstants.SendError.SEQUENCE_ERROR));
        }

    }

    private void sendBackId(@NonNull CaptureResult captureResult, @NonNull IdExtractionCallback callback) {

        if (sequenceManager.isEligibleForAction(DigifiedConstants.Action.SEND_DOCUMENT_SECOND)) {

            apiManager.extractBackId(buildBase64Image(captureResult), new IdExtractionCallback() {
                @Override
                public void onResult(IdExtractionResult idExtractionResult) {
                    sequenceManager.updateStatus(idExtractionResult);
                    apiIsSuccess = idExtractionResult.getError() == null;
                    currentAction = sequenceManager.selectNextAction(apiIsSuccess);
                    updateFinalResult(idExtractionResult.getOverallStatus());
                    callback.onResult(idExtractionResult);
                }

                @Override
                public void onUploadProgress(int captureType, int progress) {
                    callback.onUploadProgress(captureType, progress);
                }

                @Override
                public void onUploadingDone(int captureType) {
                    callback.onUploadingDone(captureType);
                }

                @Override
                public void onFailure(DigifiedError digifiedError) {
                    sequenceManager.updateStatus(digifiedError);
                    apiIsSuccess = false;
                    currentAction = sequenceManager.selectNextAction(false);
                    callback.onFailure(digifiedError);
                }
            });
        } else {
            Log.e(TAG, "Follow the sequence of the Digified SDK");
            callback.onFailure(new DigifiedError(ErrorConstants.SendError.SEQUENCE_ERROR));
        }
    }

    private void sendVehicleLicenseFront(CaptureResult captureResult, @NonNull VehicleLicenseExtractionCallback callback) {

        if (sequenceManager.isEligibleForAction(DigifiedConstants.Action.SEND_DOCUMENT_FIRST)) {

            apiManager.extractFrontVehicleLicense(buildBase64Image(captureResult), new VehicleLicenseExtractionCallback() {
                @Override
                public void onResult(VehicleLicenseExtractionResult vehicleLicenseExtractionResult) {
                    sequenceManager.updateStatus(vehicleLicenseExtractionResult);
                    apiIsSuccess = vehicleLicenseExtractionResult.getError() == null;
                    currentAction = sequenceManager.selectNextAction(apiIsSuccess);
                    updateFinalResult(vehicleLicenseExtractionResult.getOverallStatus());
                    callback.onResult(vehicleLicenseExtractionResult);
                }

                @Override
                public void onUploadProgress(int captureType, int progress) {
                    callback.onUploadProgress(captureType, progress);
                }

                @Override
                public void onUploadingDone(int captureType) {
                    callback.onUploadingDone(captureType);
                }

                @Override
                public void onFailure(DigifiedError digifiedError) {
                    sequenceManager.updateStatus(digifiedError);
                    apiIsSuccess = false;
                    currentAction = sequenceManager.selectNextAction(false);
                    callback.onFailure(digifiedError);
                }
            });

        } else {
            Log.e(TAG, "Follow the sequence of the Digified SDK");
            callback.onFailure(new DigifiedError(ErrorConstants.SendError.SEQUENCE_ERROR));
        }

    }

    private void sendVehicleLicenseBack(CaptureResult captureResult, @NonNull VehicleLicenseExtractionCallback callback) {

        if (sequenceManager.isEligibleForAction(DigifiedConstants.Action.SEND_DOCUMENT_SECOND)) {

            apiManager.extractBackVehicleLicense(buildBase64Image(captureResult), new VehicleLicenseExtractionCallback() {
                @Override
                public void onResult(VehicleLicenseExtractionResult vehicleLicenseExtractionResult) {
                    sequenceManager.updateStatus(vehicleLicenseExtractionResult);
                    apiIsSuccess = vehicleLicenseExtractionResult.getError() == null;
                    currentAction = sequenceManager.selectNextAction(apiIsSuccess);
                    updateFinalResult(vehicleLicenseExtractionResult.getOverallStatus());
                    callback.onResult(vehicleLicenseExtractionResult);
                }

                @Override
                public void onUploadProgress(int captureType, int progress) {
                    callback.onUploadProgress(captureType, progress);
                }

                @Override
                public void onUploadingDone(int captureType) {
                    callback.onUploadingDone(captureType);
                }

                @Override
                public void onFailure(DigifiedError digifiedError) {
                    sequenceManager.updateStatus(digifiedError);
                    apiIsSuccess = false;
                    currentAction = sequenceManager.selectNextAction(false);
                    callback.onFailure(digifiedError);
                }
            });

        } else {
            Log.e(TAG, "Follow the sequence of the Digified SDK");
            callback.onFailure(new DigifiedError(ErrorConstants.SendError.SEQUENCE_ERROR));
        }

    }

    private void sendSelfie(@NonNull CaptureResult captureResult, @NonNull FaceMatchCallback callback) {

        if (sequenceManager.isEligibleForAction(DigifiedConstants.Action.SEND_SELFIE)) {

            apiManager.faceMatch(captureManager.buildBase64List(captureResult), new FaceMatchCallback() {
                @Override
                public void onResult(FaceMatchResult faceMatchResult) {
                    sequenceManager.updateStatus(faceMatchResult);
                    apiIsSuccess = faceMatchResult.getError() == null;
                    currentAction = sequenceManager.selectNextAction(apiIsSuccess);
                    updateFinalResult(faceMatchResult.getOverallStatus());
                    callback.onResult(faceMatchResult);
                }

                @Override
                public void onUploadProgress(int captureType, int progress) {
                    callback.onUploadProgress(captureType, progress);
                }

                @Override
                public void onUploadingDone(int captureType) {
                    callback.onUploadingDone(captureType);
                }


                @Override
                public void onFailure(DigifiedError digifiedError) {
                    sequenceManager.updateStatus(digifiedError);
                    apiIsSuccess = false;
                    currentAction = sequenceManager.selectNextAction(false);
                    callback.onFailure(digifiedError);
                }
            });

        } else {
            Log.e(TAG, "Follow the sequence of the Digified SDK");
            callback.onFailure(new DigifiedError(ErrorConstants.SendError.SEQUENCE_ERROR));
        }

    }

    private void sendPassport(CaptureResult captureResult, @NonNull PassportExtractionCallback callback) {

        if (sequenceManager.isEligibleForAction(DigifiedConstants.Action.SEND_DOCUMENT_FIRST)) {

            apiManager.extractPassport(buildBase64Image(captureResult), new PassportExtractionCallback() {
                @Override
                public void onResult(PassportExtractionResult passportExtractionResult) {
                    sequenceManager.updateStatus(passportExtractionResult);
                    apiIsSuccess = passportExtractionResult.getError() == null;
                    currentAction = sequenceManager.selectNextAction(apiIsSuccess);
                    updateFinalResult(passportExtractionResult.getOverallStatus());
                    callback.onResult(passportExtractionResult);
                }

                @Override
                public void onUploadProgress(int captureType, int progress) {
                    callback.onUploadProgress(captureType, progress);
                }

                @Override
                public void onUploadingDone(int captureType) {
                    callback.onUploadingDone(captureType);
                }

                @Override
                public void onFailure(DigifiedError digifiedError) {
                    sequenceManager.updateStatus(digifiedError);
                    apiIsSuccess = false;
                    currentAction = sequenceManager.selectNextAction(false);
                    callback.onFailure(digifiedError);
                }
            });

        } else {
            Log.e(TAG, "Follow the sequence of the Digified SDK");
            callback.onFailure(new DigifiedError(ErrorConstants.SendError.SEQUENCE_ERROR));
        }

    }

    private void sendFrontGenericId(CaptureResult captureResult, @NonNull GenericIdExtractionCallback callback) {

        if (sequenceManager.isEligibleForAction(DigifiedConstants.Action.SEND_DOCUMENT_FIRST)) {

            apiManager.extractFrontGenericId(buildBase64Image(captureResult), new GenericIdExtractionCallback() {
                @Override
                public void onResult(GenericIdExtractionResult genericIdExtractionResult) {
                    sequenceManager.updateStatus(genericIdExtractionResult);
                    apiIsSuccess = genericIdExtractionResult.getError() == null;
                    currentAction = sequenceManager.selectNextAction(apiIsSuccess);
                    updateFinalResult(genericIdExtractionResult.getOverallStatus());
                    callback.onResult(genericIdExtractionResult);
                }

                @Override
                public void onUploadProgress(int captureType, int progress) {
                    callback.onUploadProgress(captureType, progress);
                }

                @Override
                public void onUploadingDone(int captureType) {
                    callback.onUploadingDone(captureType);
                }

                @Override
                public void onFailure(DigifiedError digifiedError) {
                    sequenceManager.updateStatus(digifiedError);
                    apiIsSuccess = false;
                    currentAction = sequenceManager.selectNextAction(false);
                    callback.onFailure(digifiedError);
                }
            });

        } else {
            Log.e(TAG, "Follow the sequence of the Digified SDK");
            callback.onFailure(new DigifiedError(ErrorConstants.SendError.SEQUENCE_ERROR));
        }

    }

    private void sendBackGenericId(CaptureResult captureResult, @NonNull GenericIdExtractionCallback callback) {

        if (sequenceManager.isEligibleForAction(DigifiedConstants.Action.SEND_DOCUMENT_SECOND)) {

            apiManager.extractBackGenericId(buildBase64Image(captureResult), new GenericIdExtractionCallback() {
                @Override
                public void onResult(GenericIdExtractionResult genericIdExtractionResult) {
                    sequenceManager.updateStatus(genericIdExtractionResult);
                    apiIsSuccess = genericIdExtractionResult.getError() == null;
                    currentAction = sequenceManager.selectNextAction(apiIsSuccess);
                    updateFinalResult(genericIdExtractionResult.getOverallStatus());
                    callback.onResult(genericIdExtractionResult);
                }

                @Override
                public void onUploadProgress(int captureType, int progress) {
                    callback.onUploadProgress(captureType, progress);
                }

                @Override
                public void onUploadingDone(int captureType) {
                    callback.onUploadingDone(captureType);
                }

                @Override
                public void onFailure(DigifiedError digifiedError) {
                    sequenceManager.updateStatus(digifiedError);
                    apiIsSuccess = false;
                    currentAction = sequenceManager.selectNextAction(false);
                    callback.onFailure(digifiedError);
                }
            });

        } else {
            Log.e(TAG, "Follow the sequence of the Digified SDK");
            callback.onFailure(new DigifiedError(ErrorConstants.SendError.SEQUENCE_ERROR));
        }

    }

    void shouldAuthorize(String token, FragmentManager fragmentManager, int captureContainerId, CaptureScreenMetaData captureScreenMetaData, DigifiedAuthorizationCallback authorizationCallback) {
        if (sequenceManager.isEligibleForAction(DigifiedConstants.Action.AUTHORIZE)) {
            digifiedAuthorizationManager.runAuthorizationCheck(token, fragmentManager, captureContainerId, captureScreenMetaData, authorizationCallback);
        } else {
            Log.e(TAG, "Follow the sequence of the Digified SDK, you probably haven't called initialize");
            authorizationCallback.onFailure(new DigifiedError(ErrorConstants.SendError.SEQUENCE_ERROR));
        }
    }

    private int captureTypeToAction(int captureType) {
        if (captureType == DigifiedConstants.CaptureType.ID_FRONT || captureType == DigifiedConstants.CaptureType.VEHICLE_LICENSE_FRONT || captureType == DigifiedConstants.CaptureType.Generic_ID_FRONT || captureType == DigifiedConstants.CaptureType.PASSPORT)
            return DigifiedConstants.Action.CAPTURE_DOCUMENT_FIRST;
        else if (captureType == DigifiedConstants.CaptureType.ID_BACK || captureType == DigifiedConstants.CaptureType.VEHICLE_LICENSE_BACK || captureType == DigifiedConstants.CaptureType.Generic_ID_BACK)
            return DigifiedConstants.Action.CAPTURE_DOCUMENT_SECOND;
        else if (captureType == DigifiedConstants.CaptureType.SELFIE)
            return DigifiedConstants.Action.CAPTURE_SELFIE;
        else return DigifiedConstants.Action.UNKNOWN;
    }

    private void updateFinalResult(Integer overallStatus) {
        if (overallStatus != null) {
            switch (overallStatus) {
                case ApiConstants.OverAllStatus.FAILED:
                    finalStatus = DigifiedConstants.Status.REJECTED;
                    return;
                case ApiConstants.OverAllStatus.VERIFIED:
                    finalStatus = DigifiedConstants.Status.VERIFIED;
                    return;
                case ApiConstants.OverAllStatus.IN_PROGRESS:
                    finalStatus = DigifiedConstants.Status.IN_PROGRESS;
                    return;
                case ApiConstants.OverAllStatus.PENDING:
                    finalStatus = DigifiedConstants.Status.PENDING;
                    return;
            }
        }
        finalStatus = DigifiedConstants.Status.UNKNOWN;
    }

    private String buildBase64Image(CaptureResult captureResult) {
        Bitmap imageBitmap = captureResult.getCapturedBitmap();
        return captureManager.bitmapToBase64ConsideringDimensions(imageBitmap);
    }

}
