//
//  StandardSuite.java
//  DiME - Data Identity Message Envelope
//  A powerful universal data format that is built for secure, and integrity protected communication between trusted
//  entities in a network.
//
//  Released under the MIT licence, see LICENSE for more information.
//  Copyright (c) 2022 Shift Everywhere AB. All rights reserved.
//
package io.dimeformat.crypto;

import com.goterl.lazysodium.SodiumJava;
import io.dimeformat.Base58;
import io.dimeformat.enums.KeyCapability;
import io.dimeformat.Utility;
import io.dimeformat.exceptions.CryptographyException;

import java.util.List;

/**
 * Implements the DiME Standard Cryptographic suite (DSC).
 */
class StandardSuite implements ICryptoSuite {

    static final String STANDARD_SUITE = "DSC";
    static final String BASE58_SUITE = "STN"; // This is legacy base58 encoding

    public String getName() {
        return _suiteName;
    }

    public StandardSuite(String name) {
        this._sodium = new SodiumJava();
        this._suiteName = name;
    }

    public byte[] generateKeyName(byte[][] key) {
        // This only supports key identifier for public keys, may be different for other crypto suites
        byte[] identifier = null;
        byte[] bytes = key[ICryptoSuite.PUBLIC_KEY_INDEX];
        if (bytes != null && bytes.length > 0) {
            try {
                byte[] hash = hash(bytes);
                identifier = Utility.subArray(hash, 0, 8); // First 8 bytes are used as an identifier
            } catch (CryptographyException e) { /* ignored */ }
        }
        return identifier;
    }

    public byte[] generateSignature(byte[] data, byte[] key) throws CryptographyException {
        byte[] signature = new byte[StandardSuite.NBR_SIGNATURE_BYTES];
        if (this._sodium.crypto_sign_detached(signature, null, data, data.length, key) != 0) {
            throw new CryptographyException("Cryptographic operation failed.");
        }
        return signature;
    }

    public boolean verifySignature(byte[] data, byte[] signature, byte[] key) {
        return (this._sodium.crypto_sign_verify_detached(signature, data, data.length, key) == 0);
    }

    public byte[][] generateKey(List<KeyCapability> capabilities) throws CryptographyException {
        if (capabilities == null || capabilities.size() != 1) { throw new IllegalArgumentException("Unable to generate, invalid key capabilities requested."); }
        KeyCapability firstUse = capabilities.get(0);
        if (firstUse == KeyCapability.ENCRYPT) {
            byte[] secretKey = new byte[StandardSuite.NBR_S_KEY_BYTES];
            this._sodium.crypto_secretbox_keygen(secretKey);
            return new byte[][] { secretKey };
        } else {
            byte[] publicKey = new byte[StandardSuite.NBR_A_KEY_BYTES];
            byte[] secretKey;
            switch (capabilities.get(0)) {
                case SIGN:
                    secretKey = new byte[StandardSuite.NBR_A_KEY_BYTES * 2];
                    this._sodium.crypto_sign_keypair(publicKey, secretKey);
                    break;
                case EXCHANGE:
                    secretKey = new byte[StandardSuite.NBR_A_KEY_BYTES];
                    this._sodium.crypto_kx_keypair(publicKey, secretKey);
                    break;
                default:
                    throw new CryptographyException("Unable to generate keypair for key type " + capabilities + ".");
            }
            return new byte[][] { secretKey, publicKey };
        }
    }

    public byte[] generateSharedSecret(byte[][] clientKey, byte[][] serverKey, List<KeyCapability> capabilities) throws CryptographyException {
        if (!capabilities.contains(KeyCapability.ENCRYPT)) { throw new IllegalArgumentException("Unable to generate, key capability for shared secret must be ENCRYPT."); }
        if (capabilities.size() > 1) { throw new IllegalArgumentException("Unable to generate, key capability for shared secret may only be ENCRYPT."); }
        byte[] shared = new byte[StandardSuite.NBR_X_KEY_BYTES];
        if (clientKey[0] != null && clientKey.length == 2) { // has both private and public key
            byte[] secret = Utility.combine(clientKey[0], clientKey[1]);
            if (this._sodium.crypto_kx_client_session_keys(shared, null, clientKey[1], secret, serverKey[1]) != 0) {
                throw new CryptographyException("Unable to generate, cryptographic operation failed.");
            }
        } else if (serverKey[0] != null && serverKey.length == 2) { // has both private and public key
            if (this._sodium.crypto_kx_server_session_keys(null, shared, serverKey[1], serverKey[0], clientKey[1]) != 0) {
                throw new CryptographyException("Unable to generate, cryptographic operation failed.");
            }
        } else {
            throw new CryptographyException("Unable to generate, invalid keys provided.");
        }
        return shared;
    }

    public byte[] encrypt(byte[] data, byte[] key) throws CryptographyException {
        byte[] nonce = Utility.randomBytes(StandardSuite.NBR_NONCE_BYTES);
        if (nonce.length > 0) {
            byte[] cipherText = new byte[StandardSuite.NBR_MAC_BYTES + data.length];
            if (this._sodium.crypto_secretbox_easy(cipherText, data, data.length, nonce, key) != 0) {
                throw new CryptographyException("Cryptographic operation failed.");
            }
            return Utility.combine(nonce, cipherText);
        }
        throw new CryptographyException("Unable to generate sufficient nonce.");

    }

    public byte[] decrypt(byte[] data, byte[] key) throws CryptographyException {
        byte[] nonce = Utility.subArray(data, 0, StandardSuite.NBR_NONCE_BYTES);
        byte[] bytes = Utility.subArray(data, StandardSuite.NBR_NONCE_BYTES);
        byte[] plain = new byte[bytes.length - StandardSuite.NBR_MAC_BYTES];
        int result = this._sodium.crypto_secretbox_open_easy(plain, bytes, bytes.length, nonce, key);
        if (result != 0) {
            throw new CryptographyException("Cryptographic operation failed (" + result + ").");
        }
        return plain;
    }

    public String generateHash(byte[] data) throws CryptographyException {
        return Utility.toHex(hash(data));
    }

    public String encodeKey(byte[] key) {
        if (_suiteName.equals(StandardSuite.BASE58_SUITE)) {
            return Base58.encode(key);
        }
        return Utility.toBase64(key);
    }

    public byte[] decodeKey(String encodedKey) {
        if (_suiteName.equals(StandardSuite.BASE58_SUITE)) {
            return Base58.decode(encodedKey);
        }
        return Utility.fromBase64(encodedKey);
    }

    /// PRIVATE ///

    private static final int NBR_SIGNATURE_BYTES = 64;
    private static final int NBR_A_KEY_BYTES = 32;
    private static final int NBR_S_KEY_BYTES = 32;
    private static final int NBR_X_KEY_BYTES = 32;
    private static final int NBR_NONCE_BYTES = 24;
    private static final int NBR_MAC_BYTES = 16;
    private static final int NBR_HASH_BYTES = 32;

    private final SodiumJava _sodium;
    private final String _suiteName;

    private byte[] hash(byte[] data) throws CryptographyException {
        byte[] hash = new byte[StandardSuite.NBR_HASH_BYTES];
        if (this._sodium.crypto_generichash(hash, hash.length, data, data.length, null, 0) != 0) {
            throw new CryptographyException("Cryptographic operation failed.");
        }
        return hash;
    }

}
