package io.dyte.core.chat

import io.dyte.core.chat.channel.DefaultChatChannelController
import io.dyte.core.chat.roomnode.ChatRoomNodeController
import io.dyte.core.chat.socketservice.ChatSocketServiceController
import io.dyte.core.events.EventEmitter
import io.dyte.core.feat.DyteChat
import io.dyte.core.listeners.DyteChatChannelEventsListener
import io.dyte.core.models.DyteMeetingType
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.network.PresignedUrlApiService
import io.dyte.core.network.info.ChatChannelPermissions
import io.dyte.core.network.info.ChatMessagePermissions
import io.dyte.core.network.info.PrivateChatPermissions
import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.IDytePlatformUtilsProvider
import io.dyte.core.socket.ISocketMessageResponseParser
import io.dyte.core.socket.RoomNodeSocketService
import io.dyte.core.socket.socketservice.SockratesSocketService
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.newSingleThreadContext

/**
 * Factory to create [DyteChat] instance based on **chat_socket_server** feature flag. The feature
 * flag tells whether Chat in the current meeting use RoomNode or SocketService.
 */
internal class DyteChatFactory(
  private val selfPeerId: String,
  private val selfUserId: String,
  private val selfDisplayName: String,
  private val selfParticipant: DyteSelfParticipant,
  private val privateChatPermissions: PrivateChatPermissions,
  private val chatChannelPermissions: ChatChannelPermissions?,
  private val chatMessagePermissions: ChatMessagePermissions?,
  private val roomNodeSocket: RoomNodeSocketService,
  private val sockratesSocketService: SockratesSocketService,
  private val apiBaseUrl: String,
  private val platformUtilsProvider: IDytePlatformUtilsProvider,
  private val socketIoMessageResponseParser: ISocketMessageResponseParser,
  private val meetingType: DyteMeetingType,
  private val roomUUID: String,
  private val chatChannelEventEmitter: EventEmitter<DyteChatChannelEventsListener>?,
) {
  @OptIn(ExperimentalCoroutinesApi::class, DelicateCoroutinesApi::class)
  fun create(chatSocketServer: String): DyteChat {
    val scope = CoroutineScope(newSingleThreadContext("DyteChat"))
    val chatFileUploader = DefaultChatFileUploader(roomUUID, PresignedUrlApiService(apiBaseUrl))
    val chatController: BaseChatController =
      when (chatSocketServer) {
        "socket-service" -> {
          DyteLogger.info("DyteChat::create::using socket-service-chat")
          ChatSocketServiceController(
            platformUtilsProvider,
            selfParticipant,
            privateChatPermissions.canSendText,
            privateChatPermissions.canSendFiles,
            sockratesSocketService,
            chatFileUploader,
            scope,
          )
        }
        else -> {
          DyteLogger.info("DyteChat::create::using room-node-chat")
          ChatRoomNodeController(
            platformUtilsProvider,
            selfParticipant,
            privateChatPermissions.canSendText,
            privateChatPermissions.canSendFiles,
            selfPeerId,
            selfDisplayName,
            roomNodeSocket,
            socketIoMessageResponseParser,
            chatFileUploader,
            scope,
          )
        }
      }

    val chatChannelController =
      if (
        meetingType == DyteMeetingType.CHAT &&
          chatChannelPermissions != null &&
          chatChannelEventEmitter != null
      ) {
        DefaultChatChannelController(
          selfUserId,
          chatChannelPermissions,
          chatFileUploader,
          platformUtilsProvider.getPlatformUtils(),
          sockratesSocketService,
          ChatMessageMapper(platformUtilsProvider.getPlatformUtils()),
          chatChannelEventEmitter,
          scope,
        )
      } else {
        null
      }

    return DyteChat(chatController, chatChannelController, scope)
  }
}
