package io.dyte.core.controllers

import io.dyte.core.events.InternalEvents
import io.dyte.core.network.ApiClient
import io.dyte.core.socket.RoomNodeSocketService
import io.dyte.core.socket.SocketMessageEventListener
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.payloadmodel.InboundMeetingEvent
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketMeetingPeerUser
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerLeftModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch

internal class RecordingRoomNodeController(
  apiClient: ApiClient,
  canRecord: Boolean,
  private val socket: RoomNodeSocketService,
  scope: CoroutineScope,
) : RecordingController(apiClient, canRecord, scope), InternalEvents {

  private var recordingPeerIds = mutableSetOf<String>()

  override fun setupEvents() {
    socket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_JOINED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          scope.launch {
            val peer = event.payload as WebSocketMeetingPeerUser
            updateRecordingParticipant(peer)
          }
        }
      },
    )

    socket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_LEFT,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {

          scope.launch {
            if (event.payload is WebSocketPeerLeftModel) {
              val peer = event.payload
              if (recordingPeerIds.contains(peer.id)) {
                recordingPeerIds.remove(peer.id)
                if (recordingPeerIds.size == 0) {
                  onRecordingPeerLeft()
                }
              }
            }
          }
        }
      },
    )
  }

  private fun updateRecordingParticipant(peer: WebSocketMeetingPeerUser) {
    if (peer.flags.recorder) {
      if (!recordingPeerIds.contains(peer.id)) {
        recordingPeerIds.add(peer.id)
        if (recordingPeerIds.size == 1) {
          onRecordingPeerJoined()
        }
      }
    }
  }

  override fun onRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel) {
    super.onRoomJoined(webSocketJoinRoomModel)
    scope.launch {
      val peers = webSocketJoinRoomModel.peers
      for (peer in peers) {
        updateRecordingParticipant(peer)
      }
    }
  }
}
