package io.dyte.core.cameramanager

import io.dyte.core.models.AudioDeviceType
import io.dyte.core.models.AudioDeviceType.BLUETOOTH
import io.dyte.core.models.AudioDeviceType.EAR_PIECE
import io.dyte.core.models.AudioDeviceType.SPEAKER
import io.dyte.core.models.AudioDeviceType.UNKNOWN
import io.dyte.core.models.AudioDeviceType.WIRED
import io.dyte.core.models.DyteAudioDevice
import io.dyte.core.observability.DyteLogger
import kotlinx.cinterop.ExperimentalForeignApi
import platform.AVFAudio.*
import platform.Foundation.*
import platform.UIKit.UIDevice
import platform.UIKit.UIDeviceProximityStateDidChangeNotification

@OptIn(ExperimentalForeignApi::class)
class InCallManager {
  private var _userSelectedAudioRoute: String = ""
  private var _currentAudioRoute: String = ""
  private var _forceSpeakerOn = 0
  private var _isProximityRegistered = false
  private var _proximityIsNear = false
  private var _currentDevice: UIDevice? = null
  private var _proximityObserver: Any? = null
  private lateinit var _audioSession: AVAudioSession
  var selectedAudioDevice: AudioDeviceType? = null
  private lateinit var routeObserver: Any

  fun init() {
    _currentDevice = UIDevice.currentDevice()
    selectedAudioDevice = AudioDeviceType.SPEAKER
    _audioSession = AVAudioSession.sharedInstance()
    _proximityIsNear = false
    _isProximityRegistered = false
    _proximityObserver = null
    _forceSpeakerOn = 0
    defaultAudioRouting()
    addObserver()
  }

  private fun addObserver() {
    val notificationName = platform.AVFAudio.AVAudioSessionRouteChangeNotification
    routeObserver =
      NSNotificationCenter.defaultCenter.addObserverForName(notificationName, null, null) {
        notification ->
        notification?.let { setRoute(it) }
      }
  }

  private fun setRoute(notification: NSNotification) {
    val session = AVAudioSession.sharedInstance()

    val info = notification.userInfo ?: return
    val reason =
      (info[platform.AVFAudio.AVAudioSessionRouteChangeReasonKey] as? Int)?.toULong() ?: return

    when (reason) {
      AVAudioSessionRouteChangeReasonOldDeviceUnavailable,
      AVAudioSessionRouteChangeReasonOverride -> {
        val previousRoute =
          info[AVAudioSessionRouteChangePreviousRouteKey] as? AVAudioSessionRouteDescription
        previousRoute?.let {
          val outputPort = it.outputs.firstOrNull() as? AVAudioSessionPortDescription
          outputPort?.let {
            if (
              it.portType == AVAudioSessionPortBluetoothLE ||
                it.portType == AVAudioSessionPortBluetoothHFP ||
                it.portType == AVAudioSessionPortBluetoothA2DP
            ) {
              // IF user selects manually it means current selected device is never be bluetooth,
              // IF bluetooth device is not available automatically then current selected device
              // will still be bluet tooth then
              // we have to route audio to speaker
              if (selectedAudioDevice == BLUETOOTH) {
                val currentOutputPort =
                  session.currentRoute.outputs.firstOrNull() as? AVAudioSessionPortDescription
                currentOutputPort?.let {
                  if (it.portType == AVAudioSessionPortBuiltInSpeaker) {
                    selectedAudioDevice = AudioDeviceType.SPEAKER
                  } else {
                    routeAudioFromSpeakerphone()
                  }
                }
              }
            }
          }
        }
      }
      AVAudioSessionRouteChangeReasonNewDeviceAvailable -> {

        val output = session.currentRoute.outputs.firstOrNull() as? AVAudioSessionPortDescription
        output?.let {
          if (
            it.portType == AVAudioSessionPortBluetoothLE ||
              it.portType == AVAudioSessionPortBluetoothHFP ||
              it.portType == AVAudioSessionPortBluetoothA2DP
          ) {
            selectedAudioDevice = AudioDeviceType.BLUETOOTH
          }
        }
      }
      AVAudioSessionRouteChangeReasonCategoryChange -> {
        if (session.category != AVAudioSessionCategoryPlayAndRecord) {
          if (selectedAudioDevice == AudioDeviceType.BLUETOOTH) {
            routeAudioFromBluetooth()
          } else if (selectedAudioDevice == AudioDeviceType.SPEAKER) {
            routeAudioFromSpeakerphone()
          } else {
            routeAudioFromEarpiece()
          }
        }
      }
      else -> {
        DyteLogger.warn("DyteIOSPlatformProvider:: setRoute Unhandle route change reason")
      }
    }
  }

  fun dispose() {
    NSNotificationCenter.defaultCenter.removeObserver(routeObserver)
  }

  @Suppress("Unused")
  fun startProximitySensor() {
    if (_isProximityRegistered) {
      return
    }

    // TODO:shift on main thread
    this._currentDevice?.proximityMonitoringEnabled ?: true

    this.stopObserve(
      _proximityObserver,
      UIDeviceProximityStateDidChangeNotification.toString(),
      null,
    )

    _proximityObserver =
      this.startObserve(
        UIDeviceProximityStateDidChangeNotification.toString(),
        _currentDevice,
        null,
        block = {
          val state: Boolean = this._currentDevice?.proximityState ?: false
          if (state != this._proximityIsNear) {
            this._proximityIsNear = state
            val obj =
              if (state) {
                { "isNear" to true }
              } else {
                { "isNear" to false }
              }

            NSNotificationCenter.defaultCenter().postNotificationName("Proximity", `object` = obj)
          }
        },
      )

    _isProximityRegistered = true
  }

  private fun startObserve(
    name: String,
    obj: Any?,
    queue: NSOperationQueue?,
    block: ((NSNotification?) -> Unit),
  ): Any {
    return NSNotificationCenter.defaultCenter().addObserverForName(name, obj, queue, block)
  }

  private fun stopObserve(observer: Any?, name: String, obj: Any?) {
    if (observer == null) {
      return
    }

    NSNotificationCenter.defaultCenter.removeObserver(observer, name, obj)
  }

  private fun stopProximitySensor() {
    if (!_isProximityRegistered) {
      return
    }

    _currentDevice?.proximityMonitoringEnabled ?: false

    // --- remove all no matter what object
    this.stopObserve(
      _proximityObserver,
      UIDeviceProximityStateDidChangeNotification.toString(),
      null,
    )

    _isProximityRegistered = false
  }

  private fun checkAudioRoute(targetPortTypeArray: List<String>, routeType: String): Boolean {
    val currentRoute = this._audioSession.currentRoute
    val routes: List<*> =
      if (routeType == "input") {
        currentRoute.inputs
      } else {
        currentRoute.outputs
      }

    for (portDescription in routes) {
      val desc = portDescription as AVAudioSessionPortDescription
      if (targetPortTypeArray.contains(desc.portType)) {
        return true
      }
    }
    return false
  }

  @OptIn(ExperimentalForeignApi::class)
  fun updateAudioRoute(): Boolean {
    val overrideAudioPort: AVAudioSessionPortOverride?
    var audioMode = AVAudioSessionModeVideoChat
    val mediaType = ""

    if (_forceSpeakerOn == 1) {
      overrideAudioPort = AVAudioSessionPortOverrideSpeaker
      if (mediaType == "video") {
        audioMode = AVAudioSessionModeVideoChat
        this.stopProximitySensor()
      }
      // Bluetooth device
    } else { // use default behavior
      // NSLog(@"👨‍💻 Else Audio Route");
      overrideAudioPort = AVAudioSessionPortOverrideNone
      if (mediaType == "video") {
        audioMode = AVAudioSessionModeVideoChat
        this.stopProximitySensor()
      }
    }
    // NOTE: Check Current route to speaker
    // isCurrentRouteToSpeaker = [self checkAudioRoute:@[AVAudioSessionPortBuiltInSpeaker]
    //                                               routeType:@"output"];
    val isCurrentRouteToSpeaker =
      this.checkAudioRoute(listOf(AVAudioSessionPortBuiltInSpeaker.toString()), "output")
    if (
      (overrideAudioPort == AVAudioSessionPortOverrideSpeaker && !isCurrentRouteToSpeaker) ||
        (overrideAudioPort == AVAudioSessionPortOverrideNone && isCurrentRouteToSpeaker)
    ) {
      try {
        _audioSession.overrideOutputAudioPort(overrideAudioPort, null)
      } catch (_: Exception) {}
    }

    return if (audioMode?.isNotEmpty() == true && _audioSession.mode != audioMode) {
      this.audioSessionSetMode(audioMode)
      // NSLog(@"👨‍💻 InCallManager.updateAudioRoute() audio mode has changed to %@", audioMode);
      true
    } else {
      // NSLog(@"👨‍💻 InCallManager.updateAudioRoute() did NOT change audio mode");
      false
    }
  }

  @OptIn(ExperimentalForeignApi::class)
  fun audioSessionSetMode(audioMode: String) {
    try {
      _audioSession.setMode(audioMode, null)
    } catch (e: Exception) {
      // TODO ADD LOGGING ("Error: ${e.message}")
    }
  }

  @Suppress("unused")
  fun chooseAudioRoute(
    audioRoute: String,
    resolve: ((String) -> Unit),
    reject: ((String, String, String) -> Unit),
  ) {
    var success = false
    //    SPEAKER_PHONE,
    //    WIRED_HEADSET,
    //    EARPIECE,
    //    BLUETOOTH,
    //    NONE
    // TODO: Create enums for AudioRoute
    _userSelectedAudioRoute = audioRoute
    if (_userSelectedAudioRoute != _currentAudioRoute) {
      when (audioRoute) {
        "SPEAKER_PHONE" -> {
          _forceSpeakerOn = 1
          this.updateAudioRoute()
          // TODO Hardcoding Success to true here
          success = true
        }
        "WIRED_HEADSET" -> {
          success = this.routeAudioFromEarpiece()
        }
        "EARPIECE" -> {
          success = this.routeAudioFromEarpiece()
        }
        "BLUETOOTH" -> {
          _forceSpeakerOn = 0
          success = this.routeAudioFromBluetooth()
        }
      }
      // TODO: Better Error Handling
      if (success) {
        resolve("Audio Route successfully changed")
      } else {
        reject("error_code", "getAudioUriJS() failed", "Failed to change audio route")
      }
    }
  }

  // This will set the audio route to speaker / bluetooth
  fun defaultAudioRouting() {
    if (!routeAudioFromBluetooth()) {
      if (!routeAudioFromSpeakerphone()) {
        routeAudioFromEarpiece()
      }
    }
  }

  @OptIn(ExperimentalForeignApi::class)
  fun routeAudioFromEarpiece(): Boolean {
    // TODO ADD LOGGING println("Trying::: routeAudioFromEarpiece")
    var success: Boolean
    // TODO ADD LOGGING println("Routing audio via Earpiece")
    try {
      @Suppress("UNUSED_VALUE")
      success =
        _audioSession.setCategory(
          AVAudioSessionCategoryPlayAndRecord,
          AVAudioSessionModeVoiceChat,
          AVAudioSessionCategoryOptionAllowBluetooth,
          null,
        )
      @Suppress("UNUSED_VALUE")
      getBuiltInMic()?.let {
        try {
          _audioSession.overrideOutputAudioPort(AVAudioSessionPortOverrideNone, null)
          _audioSession.setPreferredInput(it, null)
          _audioSession.setActive(true, null)
          selectedAudioDevice = AudioDeviceType.EAR_PIECE
        } catch (e: Exception) {
          // TODO ADD LOGGING println("Error: ${e.message}")
          return false
        }
      }
    } catch (e: Exception) {
      // TODO ADD LOGGING println("Error: ${e.message}")
    }
    return false
  }

  fun routeAudioFromBluetooth(): Boolean {
    // TODO ADD LOGGING println("Trying::: routeAudioFromBluetooth")
    _audioSession = AVAudioSession.sharedInstance()
    try {
      @Suppress("UNUSED_VALUE")
      _audioSession.setCategory(
        AVAudioSessionCategoryPlayAndRecord,
        AVAudioSessionModeVoiceChat,
        AVAudioSessionCategoryOptionAllowBluetooth,
        null,
      )
      // TODO ADD LOGGING if (!success) println("Error: Port override failed")
    } catch (e: Exception) {
      // TODO ADD LOGGING println("Error:>> ${e.message}")
      return false
    }
    val bluetoothDevice = checkBluetoothDeviceIsAvailabe()
    if (bluetoothDevice != null) {
      try {
        _audioSession.setPreferredInput(bluetoothDevice, null)
        _audioSession.setActive(true, null)
        selectedAudioDevice = AudioDeviceType.BLUETOOTH
      } catch (e: Exception) {
        // TODO ADD LOGGING println("Error: ${e.message}")
        return false
      }
      return true
    }
    return false
  }

  @OptIn(ExperimentalForeignApi::class)
  fun routeAudioFromSpeakerphone(): Boolean {
    // TODO ADD LOGGING println("Trying::: routeAudioFromSpeakerphone")

    var success: Boolean
    try {
      @Suppress("UNUSED_VALUE")
      success =
        _audioSession.setCategory(
          AVAudioSessionCategoryPlayAndRecord,
          AVAudioSessionModeVoiceChat,
          AVAudioSessionCategoryOptionAllowBluetooth,
          null,
        )
      success = _audioSession.overrideOutputAudioPort(AVAudioSessionPortOverrideSpeaker, null)
      // TODO ADD LOGGING if (!success) println("Error: Port override failed")
      success = _audioSession.setActive(true, null)
      return if (!success) {
        // TODO ADD LOGGING println("Error: Audio session override failed")
        false
      } else {
        // TODO ADD LOGGING println("Successfully routed audio from Loudspeaker")
        selectedAudioDevice = AudioDeviceType.SPEAKER
        true
      }
    } catch (e: Exception) {
      // TODO ADD LOGGING println("Error:>> ${e.message}")
      return false
    }
  }

  fun setAudioDevice(device: DyteAudioDevice) {
    DyteLogger.info("LocalMediaHandler::setAudioDevice::${device.type}")
    when (device.type) {
      WIRED -> {
        routeAudioFromEarpiece()
      }
      SPEAKER -> {
        routeAudioFromSpeakerphone()
      }
      BLUETOOTH -> {
        routeAudioFromBluetooth()
      }
      EAR_PIECE -> {
        routeAudioFromEarpiece()
      }
      UNKNOWN -> {
        routeAudioFromSpeakerphone()
      }
    }
  }

  fun isHeadsetPluggedIn(): Boolean {
    val route: AVAudioSessionRouteDescription = AVAudioSession.sharedInstance().currentRoute
    for (desc in route.outputs) {
      val descr = desc as? AVAudioSessionPortDescription
      if (descr != null && descr.portType == AVAudioSessionPortHeadphones) return true
    }
    return false
  }

  fun isBluetoothDeviceConnected(): Boolean {
    checkBluetoothDeviceIsAvailabe()?.let {
      return true
    }
    return false
  }

  private fun checkBluetoothDeviceIsAvailabe(): AVAudioSessionPortDescription? {
    val session = AVAudioSession.sharedInstance()
    val availableInputs = session.availableInputs ?: return null
    for (port in availableInputs) {
      val inputPort = port as? AVAudioSessionPortDescription
      inputPort?.let {
        if (
          it.portType == AVAudioSessionPortBluetoothA2DP ||
            it.portType == AVAudioSessionPortBluetoothHFP ||
            it.portType == AVAudioSessionPortBluetoothLE
        ) {
          return port
        }
      }
    }
    return null
  }

  private fun getBuiltInMic(): AVAudioSessionPortDescription? {
    val session = AVAudioSession.sharedInstance()
    val availableInputs = session.availableInputs ?: return null
    for (port in availableInputs) {
      val inputPort = port as? AVAudioSessionPortDescription
      inputPort?.let {
        if (
          it.portType == AVAudioSessionPortBuiltInMic ||
            it.portType == AVAudioSessionPortBuiltInReceiver
        ) {
          return port
        }
      }
    }
    return null
  }
}
