package io.dyte.core.media

import io.dyte.core.native.sockaddr_un
import kotlinx.cinterop.Arena
import kotlinx.cinterop.CPointer
import kotlinx.cinterop.ExperimentalForeignApi
import kotlinx.cinterop.UByteVar
import kotlinx.cinterop.addressOf
import kotlinx.cinterop.alloc
import kotlinx.cinterop.convert
import kotlinx.cinterop.pin
import kotlinx.cinterop.ptr
import kotlinx.cinterop.reinterpret
import kotlinx.cinterop.value
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.async
import platform.CoreFoundation.CFReadStreamRefVar
import platform.CoreFoundation.CFStreamCreatePairWithSocket
import platform.CoreFoundation.CFWriteStreamRefVar
import platform.CoreFoundation.kCFAllocatorDefault
import platform.Foundation.CFBridgingRelease
import platform.Foundation.NSDate
import platform.Foundation.NSDefaultRunLoopMode
import platform.Foundation.NSError
import platform.Foundation.NSFileManager
import platform.Foundation.NSInputStream
import platform.Foundation.NSLog
import platform.Foundation.NSOperationQueue
import platform.Foundation.NSOutputStream
import platform.Foundation.NSRunLoop
import platform.Foundation.NSRunLoopCommonModes
import platform.Foundation.NSStream
import platform.Foundation.NSStreamDelegateProtocol
import platform.Foundation.NSStreamEvent
import platform.Foundation.NSStreamEventErrorOccurred
import platform.Foundation.NSStreamEventHasBytesAvailable
import platform.Foundation.NSStreamEventHasSpaceAvailable
import platform.Foundation.NSStreamEventOpenCompleted
import platform.Foundation.NSStreamStatusAtEnd
import platform.Foundation.distantFuture
import platform.Foundation.run
import platform.Foundation.runMode
import platform.darwin.NSObject
import platform.darwin.dispatch_sync
import platform.posix.AF_UNIX
import platform.posix.SOCK_STREAM
import platform.posix.connect
import platform.posix.memset
import platform.posix.socket
import platform.posix.strncpy

@OptIn(ExperimentalForeignApi::class)
class ClientSocketConnection(private val filePath: String) : NSObject(), NSStreamDelegateProtocol {
  var didOpen: (() -> Unit)? = null
  var didClose: ((NSError?) -> Unit)? = null
  var streamHasSpaceAvailable: (() -> Unit)? = null
  @OptIn(ExperimentalForeignApi::class) private val arena = Arena()

  private var socketHandle: Int = -1
  private var address: sockaddr_un? = null

  private var inputStream: NSInputStream? = null
  private var outputStream: NSOutputStream? = null

  private var networkQueue: NSOperationQueue? = null
  private var shouldKeepRunning = false

  init {
    socketHandle = socket(AF_UNIX, SOCK_STREAM, 0)

    if (socketHandle == -1) {
      NSLog("DYTESS: Failure: Create socket")
    }
  }

  fun open(): Boolean {

    if (!NSFileManager.defaultManager.fileExistsAtPath(filePath)) {
      NSLog("DYTESS: Failure: Socket file missing $filePath")
      return false
    }

    if (!connectSocket()) {
      NSLog("DYTESS: connectSocket failed")
      return false
    }

    setupStreams()
    inputStream?.open()
    outputStream?.open()

    return true
  }

  fun close() {
    unscheduleStreams()

    inputStream?.delegate = null
    outputStream?.delegate = null

    inputStream?.close()
    outputStream?.close()

    inputStream = null
    outputStream = null
    arena.clear()
  }

  fun writeToStream(buffer: CPointer<UByteVar>, length: Int): Int {
    //        NSLog("DYTESS: writeToStream ${length}")
    return outputStream?.write(buffer, maxLength = length.toULong())?.toInt() ?: 0
  }

  fun connectSocket(): Boolean {
    with(arena) {
      val addr = alloc<sockaddr_un>()

      memset(addr.ptr, 0, sockaddr_un.size.convert())
      addr.sun_family = AF_UNIX.convert()

      strncpy(addr.sun_path, filePath, 108UL) // size of sun_path
      val r = connect(socketHandle, addr.ptr.reinterpret(), sockaddr_un.size.convert())
    }
    return true
  }

  fun setupStreams() {
    with(arena) {
      val readStreamPtr = alloc<CFReadStreamRefVar>()
      val writeStreamPtr = alloc<CFWriteStreamRefVar>()

      CFStreamCreatePairWithSocket(
        kCFAllocatorDefault,
        socketHandle,
        readStreamPtr.ptr,
        writeStreamPtr.ptr,
      )

      val readStream = readStreamPtr.value
      val writeStream = writeStreamPtr.value

      inputStream = CFBridgingRelease(readStream) as NSInputStream
      inputStream!!.delegate = this@ClientSocketConnection
      inputStream!!.setProperty("kCFBooleanTrue", "kCFStreamPropertyShouldCloseNativeSocket")

      outputStream = CFBridgingRelease(writeStream) as NSOutputStream?
      outputStream!!.delegate = this@ClientSocketConnection
      outputStream!!.setProperty("kCFBooleanTrue", "kCFStreamPropertyShouldCloseNativeSocket")
    }
    scheduleStreams()
  }

  fun scheduleStreams() {
    shouldKeepRunning = true

    CoroutineScope(Dispatchers.Default).async {
      inputStream?.scheduleInRunLoop(NSRunLoop.currentRunLoop, NSRunLoopCommonModes)
      outputStream?.scheduleInRunLoop(NSRunLoop.currentRunLoop, NSRunLoopCommonModes)

      var isRunning: Boolean
      NSRunLoop.currentRunLoop.run()

      do {
        isRunning =
          shouldKeepRunning &&
            NSRunLoop.currentRunLoop.runMode(
              NSDefaultRunLoopMode,
              beforeDate = NSDate.distantFuture,
            )
      } while (isRunning)
    }
  }

  fun unscheduleStreams() {
    networkQueue?.let { queue ->
      dispatch_sync(queue) {
        inputStream!!.removeFromRunLoop(NSRunLoop.currentRunLoop, NSRunLoopCommonModes)
        outputStream!!.removeFromRunLoop(NSRunLoop.currentRunLoop, NSRunLoopCommonModes)
      }
    }
    shouldKeepRunning = false
  }

  fun notifyDidClose(error: NSError?) {
    didClose?.let { it(error) }
  }

  override fun stream(aStream: NSStream, handleEvent: NSStreamEvent) {
    when (handleEvent) {
      NSStreamEventOpenCompleted -> {

        if (aStream == outputStream) {
          didOpen?.invoke()
        }
      }
      NSStreamEventHasBytesAvailable -> {
        if (aStream == inputStream) {
          val buffer = UByteArray(1)
          val bufferPinned = buffer.pin()
          val bufferptr = bufferPinned.addressOf(0)
          val numberOfBytesRead = inputStream?.read(bufferptr, maxLength = 1UL) ?: 0
          if (numberOfBytesRead.toInt() == 0 && aStream.streamStatus() == NSStreamStatusAtEnd) {
            NSLog("DYTESS: Server socket closed")
            close()
            notifyDidClose(null)
          }
          bufferPinned.unpin()
        }
      }
      NSStreamEventHasSpaceAvailable -> {
        if (aStream == outputStream) {
          streamHasSpaceAvailable?.invoke()
        }
      }
      NSStreamEventErrorOccurred -> {
        close()
        notifyDidClose(aStream.streamError())
      }
      else -> Unit // Do nothing for other cases
    }
  }
}
