package io.dyte.core.platform

import io.dyte.core.cameramanager.DyteCameraManager
import io.dyte.core.cameramanager.InCallManager
import io.dyte.core.controllers.DyteEventType
import io.dyte.core.incallmanager.InCallManagerEventCallbacks
import io.dyte.core.media.DyteCommonMediaUtils
import io.dyte.core.models.AudioDeviceType
import io.dyte.core.models.DyteAudioDevice
import io.dyte.core.models.DyteVideoDevice
import io.dyte.core.models.VideoDeviceType
import io.dyte.core.observability.DyteLogger

internal class DyteIOSMedia(private val utilsProvider: IDytePlatformUtilsProvider) :
  // iOS does not have a software encoder/decoder implementation
  DyteCommonMediaUtils(forceSoftwareEncoder = false, forceSoftwareDecoder = false),
  IDyteMediaUtils,
  InCallManagerEventCallbacks {

  private val audioDevices = arrayListOf<DyteAudioDevice>()
  private lateinit var inCallManagerModule: InCallManager
  private lateinit var dyteCameraManager: DyteCameraManager
  private val videoDevices = arrayListOf<DyteVideoDevice>()
  private var selectedVideoDevice: DyteVideoDevice? = null

  override fun init() {
    inCallManagerModule = InCallManager()
    inCallManagerModule.init()
    loadAudioDevices()
    dyteCameraManager = DyteCameraManager()
    videoDevices.clear()
    videoDevices.addAll(dyteCameraManager.getCameraDevices())
    selectedVideoDevice = videoDevices.find { it.type == VideoDeviceType.FRONT }
    DyteLogger.info("LocalMediaHandler::audio_devices::${audioDevices.map { it.type }}")
    DyteLogger.info("LocalMediaHandler::video_devices::${videoDevices.map { it.type }}")
  }

  override fun routeAudioFromSpeakerphone() {
    DyteLogger.info("LocalMediaHandler::setAudioDevice::${AudioDeviceType.SPEAKER}")
    inCallManagerModule.defaultAudioRouting()
  }

  private fun loadAudioDevices() {
    audioDevices.clear()
    audioDevices.add(DyteAudioDevice("earpiece", AudioDeviceType.EAR_PIECE))
    audioDevices.add(DyteAudioDevice("speaker", AudioDeviceType.SPEAKER))
    if (inCallManagerModule.isHeadsetPluggedIn()) {
      audioDevices.add(DyteAudioDevice("wired", AudioDeviceType.WIRED))
    }

    if (inCallManagerModule.isBluetoothDeviceConnected()) {
      audioDevices.add(DyteAudioDevice("bluetooth", AudioDeviceType.BLUETOOTH))
    }
  }

  override fun getAudioDevices(): List<DyteAudioDevice> {
    loadAudioDevices()
    return audioDevices
  }

  override fun setAudioDevice(device: DyteAudioDevice) {
    DyteLogger.info("LocalMediaHandler::setAudioDevice::${device.type}")
    inCallManagerModule.setAudioDevice(device)
  }

  override fun getSelectedAudioDevice(): DyteAudioDevice? {
    var selectedDevice = DyteAudioDevice("unknown", AudioDeviceType.UNKNOWN)
    val selectedAudioDevice = inCallManagerModule.selectedAudioDevice
    selectedAudioDevice?.let { device ->
      audioDevices.find { it.type.deviceType == device.deviceType }?.let { selectedDevice = it }
    }
    return selectedDevice
  }

  override fun getVideoDevices(): List<DyteVideoDevice> {
    return videoDevices
  }

  override fun setVideoDevice(device: DyteVideoDevice) {
    DyteLogger.info("LocalMediaHandler::setVideoDevice::${device.type}")
    utilsProvider.getControllerContainer().sfuUtils.switchCamera(device.id)
    selectedVideoDevice = device
  }

  override fun getSelectedVideoDevice(): DyteVideoDevice? {
    return selectedVideoDevice ?: videoDevices.firstOrNull()
  }

  override fun dispose() {
    super.dispose()
    inCallManagerModule.dispose()
  }

  override fun onEvent(eventName: String?, args: Any?) {
    eventName?.let { event ->
      when (event) {
        "WiredHeadset" -> {}
        "NoisyAudio" -> {}
        "MediaButton" -> {}
        "Proximity" -> {
          //          val data = args as Bundle?
          //          var isNear = false
          //          data?.let {
          //            isNear = data.getBoolean("isNear")
          //          }
          //          utilsProvider.getControllerContainer().eventController.triggerEvent(
          //            DyteEventType.OnProximityChanged(
          //              isNear
          //            )
          //          )
        }
        "onAudioFocusChange" -> {}
        "onAudioDeviceChanged" -> {
          DyteLogger.info("LocalMediaHandler::onAudioDeviceChanged")
          loadAudioDevices()
          utilsProvider
            .getControllerContainer()
            .eventController
            .triggerEvent(DyteEventType.OnAudioDevicesChanged)
        }
      }
    }
  }
}
