package io.dyte.core.platform

import io.dyte.core.observability.DyteLogger
import io.ktor.util.*
import kotlin.system.getTimeMillis
import kotlinx.cinterop.*
import kotlinx.coroutines.*
import platform.Foundation.*
import platform.UIKit.UIDevice
import platform.UniformTypeIdentifiers.UTType
import platform.darwin.NSInteger
import toByteArray

@OptIn(ExperimentalForeignApi::class)
internal class DyteIOSPlatform internal constructor() : IDytePlatformUtils {

  @Suppress("DEPRECATION")
  override fun getCurrentTime(): Long {
    return getTimeMillis()
  }

  override fun getUuid(): String {
    return NSUUID().UUIDString().lowercase()
  }

  override fun printThread(desc: String?) {
    // TODO:Not yet implemented
    println("DyteMobileClient | CURRENT THREAD: ${NSThread.currentThread} Desc: $desc")
  }

  override fun getAndroidApplicationContext(): Any {
    throw IllegalStateException("should not be called for ios")
  }

  override fun getActivity(): Any {
    TODO("Not yet implemented")
  }

  // "yyyy-MM-dd'T'HH:mm:ss'Z'"
  override fun getDiff(startTime: String, endTime: String): String {
    return "00:00"
  }

  // "2022-08-08T11:11:11.sTZD"
  override fun getUtcTimeNow(): String {
    return ""
  }

  override fun getUserDisplayableTime(time: Long): String {
    val timeInterval: NSTimeInterval = (time / 1000L).toDouble()
    val date: NSDate = NSDate.dateWithTimeIntervalSince1970(timeInterval)
    val dateFormatter = NSDateFormatter()
    dateFormatter.dateFormat = "HH:mm"
    return dateFormatter.stringFromDate(date)
  }

  override fun getFileContent(filePath: String): ByteArray {
    val nsData = NSData.dataWithContentsOfFile(filePath)
    if (nsData != null) {
      return nsData.toByteArray()
    } else {
      throw Exception("File not found!")
    }
  }

  override fun getOsName(): String {
    return "iOS"
  }

  override fun getOsVersion(): String {
    return UIDevice.currentDevice.systemVersion
  }

  override fun getDeviceInfo(): String {
    return UIDevice.currentDevice.name +
      " | " +
      UIDevice.currentDevice.model +
      " | " +
      UIDevice.currentDevice.systemVersion
  }

  companion object {
    private const val LAST_KNOWN_EXCEPTION_KEY = "LAST_KNOWN_EXCEPTION_KEY"
  }

  override fun listenForCrashes() {
    val exception = NSUserDefaults.standardUserDefaults.stringForKey(LAST_KNOWN_EXCEPTION_KEY)
    if (exception != null) {
      DyteLogger.error("unhandled_exception", mapOf("error.stack" to exception))
      NSUserDefaults.standardUserDefaults.removeObjectForKey(LAST_KNOWN_EXCEPTION_KEY)
    }

    memScoped {
      @Suppress("UNCHECKED_CAST")
      val uncaughtExceptionHandler =
        staticCFunction { exception: NSException ->
          val exceptionHandler = NSGetUncaughtExceptionHandler()
          NSSetUncaughtExceptionHandler(exceptionHandler)
          val cleanCallStack = exception.callStackSymbols.toString().replace(",", ",\n")
          val exceptionString = "$exception,\n$cleanCallStack"
          NSUserDefaults.standardUserDefaults.setValue(
            exceptionString,
            forKey = LAST_KNOWN_EXCEPTION_KEY,
          )
          NSUserDefaults.standardUserDefaults.synchronize()
        }
          as CPointer<NSUncaughtExceptionHandler>
      NSSetUncaughtExceptionHandler(uncaughtExceptionHandler)
    }
  }

  override fun getUrlEncodedString(stringToEncode: String): String {
    @Suppress("CAST_NEVER_SUCCEEDS")
    // It definitely succeeds
    // https://kotlinlang.org/api/latest/jvm/stdlib/kotlinx.cinterop/-create-n-s-string-from-k-string.html
    val str = stringToEncode as NSString
    return str.stringByAddingPercentEscapesUsingEncoding(enc = NSUTF8StringEncoding) ?: ""
  }

  override fun getPlatformFile(path: String): PlatformFile? {
    return try {
      val fileUrl = NSURL.fileURLWithPath(path)
      val name = fileUrl.lastPathComponent
      val mimeType = UTType.typeWithFilenameExtension(fileUrl.pathExtension!!)!!.preferredMIMEType
      val size = getFileSize(fileUrl)

      val content = getFileContent(path)
      // TODO ADD LOGGING println("DyteMobileClient | getPlatformFile -> $name, $mimeType, $size")
      PlatformFile(name!!, content, size, mimeType)
    } catch (e: Exception) {
      // In case if provided path is not absolute or file doesn't exist we return null
      DyteLogger.error("DyteAndroidPlatform | getPlatformFile | ${e.message} | path=$path")
      null
    }
  }

  override fun getPlatformFile(uri: Uri): PlatformFile? {
    return try {
      val path = uri.path.toString()
      val fileUrl = NSURL.fileURLWithPath(path)
      val name = fileUrl.lastPathComponent
      val mimeType = UTType.typeWithFilenameExtension(fileUrl.pathExtension!!)!!.preferredMIMEType
      val size = getFileSize(uri)

      val content = getFileContent(path)
      // TODO ADD LOGGING println("DyteMobileClient | getPlatformFile -> $name, $mimeType, $size")
      PlatformFile(name!!, content, size, mimeType)
    } catch (e: Exception) {
      // In case if provided path is not absolute or file doesn't exist we return null
      DyteLogger.error("DyteIOSPlatform | getPlatformFile | ${e.message} | uri=$uri")
      null
    }
  }

  private fun getFileSize(uri: Uri): Long {
    memScoped {
      uri.startAccessingSecurityScopedResource()
      val attributes =
        NSFileManager.defaultManager.attributesOfItemAtPath(uri.path.toString(), null)
      val size = attributes?.get(NSFileSize) as NSInteger
      // TODO ADD LOGGING println("size from sdk::: $size")
      uri.startAccessingSecurityScopedResource()
      return size
    }
  }

  override fun getDeviceModel(): String {
    return UIDevice.currentDevice.model
  }

  override fun getDeviceType(): String {
    return "iOS"
  }

  override fun getSdkType(): String {
    return "iOS-core"
  }

  override fun getBatteryLevel(): String {
    return UIDevice.currentDevice.batteryLevel.toString()
  }

  override fun runOnMainThread(runnable: () -> Unit) {
    runnable()
  }
}

@Suppress("CONFLICTING_OVERLOADS")
actual typealias Uri = NSURL
