package io.dyte.core.platform

import io.dyte.core.controllers.Permission
import io.dyte.core.controllers.PermissionType
import platform.AVFoundation.*

internal class DyteIOSPlatformPermissions : IDytePlatformPermissionUtils {
  override fun isPermissionGranted(permission: Permission): Boolean {
    if (permission.type == PermissionType.CAMERA) {
      return AVCaptureDevice.authorizationStatusForMediaType(AVMediaTypeVideo) ==
        AVAuthorizationStatusAuthorized
    }

    if (permission.type == PermissionType.MICROPHONE) {
      return AVCaptureDevice.authorizationStatusForMediaType(AVMediaTypeAudio) ==
        AVAuthorizationStatusAuthorized
    }

    return false
  }

  override fun arePermissionsGranted(permissions: List<Permission>): Boolean {
    for (permission in permissions) {
      if (permission.type == PermissionType.CAMERA) {
        if (
          AVCaptureDevice.authorizationStatusForMediaType(AVMediaTypeVideo) !=
            AVAuthorizationStatusAuthorized
        ) {
          return false
        }
      }
      if (permission.type == PermissionType.MICROPHONE) {
        if (
          AVCaptureDevice.authorizationStatusForMediaType(AVMediaTypeAudio) !=
            AVAuthorizationStatusAuthorized
        ) {
          return false
        }
      }
    }
    return true
  }

  private fun askPermission(
    status: AVAuthorizationStatus,
    type: AVMediaType,
    onPermissionsGranted: () -> Unit,
    onPermissionsDenied: () -> Unit,
  ) {
    if (status == AVAuthorizationStatusNotDetermined) {
      AVCaptureDevice.requestAccessForMediaType(
        type,
        completionHandler = {
          if (it) {
            onPermissionsGranted.invoke()
          } else {
            onPermissionsDenied.invoke()
          }
        },
      )
    } else {
      if (status == AVAuthorizationStatusAuthorized) {
        onPermissionsGranted.invoke()
      } else {
        onPermissionsDenied.invoke()
      }
    }
  }

  override fun askCameraPermissions(
    onPermissionsGranted: () -> Unit,
    onPermissionsDenied: () -> Unit,
  ): Boolean {
    val authorisationStatus = AVCaptureDevice.authorizationStatusForMediaType(AVMediaTypeVideo)
    askPermission(
      status = authorisationStatus,
      type = AVMediaTypeVideo,
      onPermissionsGranted = onPermissionsGranted,
      onPermissionsDenied = onPermissionsDenied,
    )
    return true
  }

  override fun askAudioPermissions(
    onPermissionsGranted: () -> Unit,
    onPermissionsDenied: () -> Unit,
  ): Boolean {
    val authorisationStatus = AVCaptureDevice.authorizationStatusForMediaType(AVMediaTypeAudio)
    askPermission(
      status = authorisationStatus,
      type = AVMediaTypeAudio,
      onPermissionsGranted = onPermissionsGranted,
      onPermissionsDenied = onPermissionsDenied,
    )
    return true
  }

  override fun askPermissions(
    permissions: List<Permission>,
    onPermissionsGranted: () -> Unit,
    onPermissionsDenied: () -> Unit,
  ): Boolean {
    return false
  }
}
