package io.dyte.core.platform

import DyteWebRTC.DyteRTCMTLVideoView
import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.observability.DyteLogger
import kotlinx.cinterop.ExperimentalForeignApi
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.launch
import platform.CoreGraphics.CGAffineTransformMakeScale
import platform.UIKit.UIView
import platform.UIKit.UIViewContentMode

actual typealias VideoView = UIView

@OptIn(ExperimentalForeignApi::class)
class DyteIOSVideoUtils
internal constructor(private val utilsProvider: IDytePlatformUtilsProvider) : IDyteVideoUtils {
  private var useCache = true

  private val videoViews = hashMapOf<String, UIView>()
  private val screenShareViews = hashMapOf<String, UIView>()

  override fun getVideoView(participant: DyteJoinedMeetingParticipant): UIView? {
    val localTrack = participant.videoTrack

    return if (useCache && videoViews.containsKey(participant.id)) {
      val videoView = videoViews.getValue(participant.id)
      localTrack?.addRenderer(videoView as DyteRTCMTLVideoView)
      return videoView
    } else {
      if (localTrack == null) {
        return null
      } else {
        val oldView = videoViews[participant.id]

        val videoView = DyteRTCMTLVideoView()

        if (participant is DyteSelfParticipant) {
          videoView.transform = CGAffineTransformMakeScale(-1.0, 1.0)
        }
        videoView.videoContentMode = UIViewContentMode.UIViewContentModeScaleAspectFill
        localTrack.addRenderer(videoView)

        if (useCache) {
          videoViews[participant.id] = videoView
          if (oldView != null) destroyVideoView(oldView)
        }

        videoView
      }
    }
  }

  override fun getSelfPreview(): VideoView {
    val videoView = DyteRTCMTLVideoView()
    val participant = utilsProvider.getControllerContainer().selfController.getSelf()
    val localTrack = participant.videoTrack
    videoView.transform = CGAffineTransformMakeScale(-1.0, 1.0)
    videoView.videoContentMode = UIViewContentMode.UIViewContentModeScaleAspectFill
    localTrack?.addRenderer(videoView)
    return videoView
  }

  override fun getScreenShareView(participant: DyteJoinedMeetingParticipant): VideoView {
    return if (useCache && screenShareViews.containsKey(participant.id)) {
      screenShareViews.getValue(participant.id)
    } else {
      val oldView = screenShareViews[participant.id]

      val videoView = DyteRTCMTLVideoView()
      videoView.videoContentMode = UIViewContentMode.UIViewContentModeScaleAspectFit
      val localTrack = participant._screenShareTrack

      localTrack?.addRenderer(videoView)

      if (useCache) {
        screenShareViews[participant.id] = videoView
        if (oldView != null) destroyVideoView(oldView)
      }

      videoView
    }
  }

  override fun destroyView(participant: DyteMeetingParticipant) {
    val videoView = videoViews[participant.id] as DyteRTCMTLVideoView?

    if (videoView != null) {
      participant.participantController.joinedParticipants
        .find { it.id == participant.id }
        ?.let { it.videoTrack?.removeRenderer(videoView) }
      destroyVideoView(videoView)
    }
    videoViews.remove(participant.id)
  }

  override fun destroyScreenShare(participant: DyteMeetingParticipant) {
    val screenShareView = screenShareViews[participant.id] as DyteRTCMTLVideoView?
    if (screenShareView != null) {
      participant.participantController.joinedParticipants
        .find { it.id == participant.id }
        ?.let { it.screenShareTrack?.removeRenderer(screenShareView) }
      destroyVideoView(screenShareView)
    }
    screenShareViews.remove(participant.id)
  }

  private fun destroyVideoView(view: VideoView) {
    try {
      MainScope().launch { view.removeFromSuperview() }
    } catch (e: Exception) {
      e.printStackTrace()
    }
  }

  override fun destroyAll() {
    videoViews.values.forEach { view ->
      try {
        destroyVideoView(view)
      } catch (e: Exception) {
        e.printStackTrace()
      }
    }
    videoViews.clear()

    screenShareViews.values.forEach { view ->
      try {
        destroyVideoView(view)
      } catch (e: Exception) {
        e.printStackTrace()
      }
    }
    screenShareViews.clear()
  }

  override fun disableCache() {
    DyteLogger.info("video_utils::disableCache")
    useCache = false
    destroyAll()
  }

  override fun enableCache() {
    DyteLogger.info("video_utils::enableCache")
    useCache = true
    destroyAll()
  }
}
