package io.dyte.core.platform

import io.dyte.core.feat.DytePlugin
import io.ktor.http.*
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.jsonObject
import platform.Foundation.NSDate
import platform.Foundation.NSURL
import platform.Foundation.NSURLRequest
import platform.Foundation.dateWithTimeIntervalSince1970
import platform.WebKit.*
import platform.darwin.NSObject

actual typealias WebView = WKWebView

actual typealias DytePluginFile = Unit

@Suppress("PARAMETER_NAME_CHANGED_ON_OVERRIDE")
class DyteIOSWebView : IDyteWebView {

  private var webView: WebView = WKWebView()
  private var listener: IDyteWebViewListener? = null

  override fun initializeWebView(dytePlugin: DytePlugin) {
    setJavaScriptMessagehandler(webView)
    webView.navigationDelegate = webViewNavigationDelegate
    webView.userInteractionEnabled = dytePlugin.config.accessControl == "FULL_ACCESS"
  }

  override fun submitEvent(payload: JsonObject) {
    webView.evaluateJavaScript("window.postMessage($payload)", completionHandler = null)
  }

  override fun loadURL(url: String) {
    val urlToLoad = NSURL.URLWithString(url)
    urlToLoad?.let { webView.loadRequest(NSURLRequest.requestWithURL(it)) }
  }

  override fun getWebView(): WebView {
    return this.webView
  }

  private val webViewJavaScriptListener: WKScriptMessageHandlerProtocol =
    object : NSObject(), WKScriptMessageHandlerProtocol {
      override fun userContentController(
        userContentController: WKUserContentController,
        didReceiveScriptMessage: WKScriptMessage,
      ) {
        val messageJsonString: String? = didReceiveScriptMessage.body as? String

        messageJsonString?.let {
          listener?.onWebViewMessage(Json.parseToJsonElement(it).jsonObject)
        }
      }
    }

  private val webViewNavigationDelegate: WKNavigationDelegateProtocol =
    object : NSObject(), WKNavigationDelegateProtocol {
      override fun webView(webView: WKWebView, didCommitNavigation: WKNavigation?) {
        webView.evaluateJavaScript(
          """
          window.parent = {}
          window.parent.postMessage = function(message) {
            window.webkit.messageHandlers.pluginMessageHandler.postMessage(JSON.stringify(message))
          };
          """
            .trimIndent(),
          completionHandler = null,
        )
      }
    }

  private fun setJavaScriptMessagehandler(webView: WebView) {
    WKWebsiteDataStore.defaultDataStore()
      .removeDataOfTypes(
        dataTypes = setOf(WKWebsiteDataTypeDiskCache, WKWebsiteDataTypeMemoryCache),
        modifiedSince = NSDate.dateWithTimeIntervalSince1970(0.0),
        completionHandler = {},
      )
    val contentController = webView.configuration.userContentController
    contentController.addScriptMessageHandler(
      scriptMessageHandler = webViewJavaScriptListener,
      name = "pluginMessageHandler",
    )
  }

  override fun setListener(listener: IDyteWebViewListener) {
    this.listener = listener
  }

  override fun removeListener() {
    this.listener = null
  }

  override fun uploadFile(dytePluginFile: DytePluginFile) {
    // on iOS this is the default functionality
  }
}
