package io.dyte.core.chat

import io.dyte.core.Result
import io.dyte.core.chat.ChatFileUploader.UploadFileResult
import io.dyte.core.network.PresignedUrlApiService
import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.PlatformFile
import io.ktor.client.HttpClient
import io.ktor.client.plugins.HttpTimeout
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.client.request.header
import io.ktor.client.request.put
import io.ktor.client.request.setBody
import io.ktor.http.HttpHeaders
import io.ktor.http.HttpStatusCode
import io.ktor.serialization.kotlinx.json.json
import kotlinx.serialization.json.Json

internal interface ChatFileUploader {
  fun setRoomUUID(uuid: String)

  suspend fun uploadFile(file: PlatformFile): UploadFileResult

  sealed interface UploadFileResult {
    data class Uploaded(val getLocation: String) : UploadFileResult

    data class UploadFailed(val reason: String) : UploadFileResult
  }
}

internal class DefaultChatFileUploader(
  private var roomUUID: String? = null,
  private val presignedUrlApiService: PresignedUrlApiService,
) : ChatFileUploader {
  /*
   * Note(swapnil): We are using a separate instance of HttpClient for Chat to upload files.
   * Need to investigate more to see if we can eliminate this extra instance.
   *
   * Reason: We have added a default Authorization header to the shared HttpClient in the ApiClient class. This
   * adds the Authorization header to every request. But the pre-signed upload Urls already have authorization
   * mechanism through some query params and don't allow additional Authorization header. In case, it is
   * added to the request, we get a 400 Bad request response with "Invalid Argument" message.
   * */
  private val httpClient: HttpClient by lazy { createPrivateHttpClient() }

  /** TODO: Remove this */
  override fun setRoomUUID(uuid: String) {
    roomUUID = uuid
  }

  override suspend fun uploadFile(file: PlatformFile): UploadFileResult {
    val uuid = roomUUID ?: return UploadFileResult.UploadFailed("roomUUID is null")

    val presignedUrlResult = getPresignedUrls(file.name, uuid)
    val presignedUrls =
      when (presignedUrlResult) {
        is Result.Success -> {
          presignedUrlResult.value
        }
        is Result.Failure -> {
          DyteLogger.error("ChatFileUploader::uploadFile::received invalid presigned urls")
          return UploadFileResult.UploadFailed("Failed to upload file")
        }
      }

    val uploadFileResponse =
      httpClient.put(presignedUrls.putLocation) {
        if (!file.mimeType.isNullOrBlank()) {
          header(HttpHeaders.ContentType, file.mimeType)
        }
        setBody(file.content)
      }

    if (uploadFileResponse.status.value != HttpStatusCode.OK.value) {
      return UploadFileResult.UploadFailed(
        "Failed to upload file, status code: ${uploadFileResponse.status.value}"
      )
    }

    return UploadFileResult.Uploaded(presignedUrls.getLocation)
  }

  private suspend fun getPresignedUrls(
    fileName: String,
    roomUUID: String,
  ): Result<FilePresignedUrls, String> {
    if (roomUUID.isBlank()) {
      return Result.Failure("roomUUID is blank")
    }
    val presignedUrl = presignedUrlApiService.getPresignedUrl2(fileName, roomUUID)

    val putLocation = presignedUrl?.data?.putLocation
    val getLocation = presignedUrl?.data?.getLocation

    if (putLocation.isNullOrBlank() || getLocation.isNullOrBlank()) {
      return Result.Failure("invalid presignedUrls")
    }

    return Result.Success(FilePresignedUrls(putLocation, getLocation))
  }

  private fun createPrivateHttpClient(): HttpClient {
    return HttpClient {
      install(ContentNegotiation) {
        json(
          Json {
            prettyPrint = true
            isLenient = true
            ignoreUnknownKeys = true
          }
        )
      }
      install(HttpTimeout) {
        socketTimeoutMillis = 30000
        requestTimeoutMillis = 30000
        connectTimeoutMillis = 30000
      }
    }
  }

  private data class FilePresignedUrls(val putLocation: String, val getLocation: String)
}
