package io.dyte.core.controllers.stage

import io.dyte.core.controllers.DyteStageStatus
import io.dyte.core.controllers.ParticipantController
import io.dyte.core.controllers.SelfController
import io.dyte.core.controllers.StageStatus
import io.dyte.core.host.IHostController
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.models.DyteMediaPermission
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.network.info.SelfPermissions
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.IRoomNodeSocketService
import io.dyte.core.socket.SocketMessageEventListener
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.payloadmodel.InboundMeetingEvent
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketMeetingPeerUser
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerLeftModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWebinarRequestToJoinPeerAdded
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWebinarStagePeer
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.addJsonObject
import kotlinx.serialization.json.buildJsonArray
import kotlinx.serialization.json.buildJsonObject

// TODO : get rid of participant controller from here
@Suppress("DEPRECATION") // DyteSelfEventsListener.onStageStatusUpdated is used in public API
internal class StageRoomNodeController(
  private val self: DyteSelfParticipant,
  private val participantController: ParticipantController,
  private val hostController: IHostController,
  private val selfController: SelfController,
  private val permissions: SelfPermissions,
  private val socketController: IRoomNodeSocketService,
  scope: CoroutineScope,
) : BaseStageController(self, scope) {
  private val _accessRequests = arrayListOf<DyteJoinedMeetingParticipant>()
  override val accessRequests: List<DyteJoinedMeetingParticipant>
    get() = _accessRequests

  override val viewers: List<DyteJoinedMeetingParticipant>
    get() =
      participantController.joinedParticipants.toSafeList().filter {
        it.stageStatus != StageStatus.ON_STAGE
      }

  override fun setupEvents() {
    DyteLogger.info("DyteStage::using room-node stage")
    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_REQUEST_TO_JOIN_STAGE_PEER_ADDED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          DyteLogger.info("DyteStage::onMessageEvent::join stage request peer added")
          val joinedPeer =
            participantController.joinedParticipants.find {
              it.id == (event.payload as WebSocketWebinarRequestToJoinPeerAdded).id
            } ?: return

          if (_accessRequests.contains(joinedPeer)) {
            return
          }

          joinedPeer._stageStatus = DyteStageStatus.REQUESTED_TO_JOIN_STAGE
          _accessRequests.add(joinedPeer)
          emitEvent { it.onPresentRequestAdded(joinedPeer) }
          emitEvent { it.onStageRequestsUpdated(_accessRequests) }
          participantController.triggerOnUpdate()
        }
      },
    )
    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_REQUEST_TO_JOIN_STAGE_PEER_WITHDRAWN,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          DyteLogger.info("DyteStage::onMessageEvent::join stage request withdrawn")
          val withdrawnPeer =
            participantController.joinedParticipants.find {
              it.id == (event.payload as WebSocketWebinarStagePeer).id
            } ?: return
          withdrawnPeer._stageStatus = DyteStageStatus.OFF_STAGE
          _accessRequests.remove(withdrawnPeer)
          emitEvent { it.onPresentRequestWithdrawn(withdrawnPeer) }
          emitEvent { it.onStageRequestsUpdated(_accessRequests) }
          participantController.triggerOnUpdate()
        }
      },
    )
    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_STARTED_PRESENTING,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          DyteLogger.info("DyteStage::onMessageEvent::peer started presenting")
          val addedPeer =
            participantController.joinedParticipants.find {
              it.id == (event.payload as WebSocketWebinarStagePeer).id
            } ?: return
          addedPeer._stageStatus = DyteStageStatus.ON_STAGE
          emitEvent { it.onParticipantStartedPresenting(addedPeer) }
          participantController.triggerOnUpdate()
        }
      },
    )
    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_STOPPED_PRESENTING,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          DyteLogger.info("DyteStage::onMessageEvent::peer stopped presenting")
          val leftPeer =
            participantController.joinedParticipants.find {
              it.id == (event.payload as WebSocketWebinarStagePeer).id
            } ?: return
          if (participantController.pinnedParticipant?.id == leftPeer.id) {
            participantController.onPinPeer(null)
          }
          leftPeer._stageStatus = DyteStageStatus.OFF_STAGE
          emitEvent { it.onParticipantStoppedPresenting(leftPeer) }
          participantController.triggerOnUpdate()
        }
      },
    )
    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_REJECTED_TO_JOIN_STAGE,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          DyteLogger.info("DyteStage::onMessageEvent::peer rejected to join stage")
          val rejectedPeer =
            participantController.joinedParticipants.find {
              it.id == (event.payload as WebSocketWebinarStagePeer).id
            } ?: return
          rejectedPeer._stageStatus = DyteStageStatus.OFF_STAGE
          val requestListModified = _accessRequests.removeAll { it.id == rejectedPeer.id }
          emitEvent { it.onPresentRequestRejected(rejectedPeer) }
          if (requestListModified) {
            emitEvent { it.onStageRequestsUpdated(_accessRequests) }
          }
          participantController.triggerOnUpdate()
        }
      },
    )
    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_REMOVED_FROM_STAGE,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          DyteLogger.info("DyteStage::onMessageEvent::peer removed from stage")
          val removedPeer =
            participantController.joinedParticipants.find {
              it.id == (event.payload as WebSocketWebinarStagePeer).id
            } ?: return
          removedPeer._stageStatus = DyteStageStatus.OFF_STAGE
          if (participantController.pinnedParticipant?.id == removedPeer.id) {
            participantController.onPinPeer(null)
          }
          emitEvent { it.onParticipantRemovedFromStage(removedPeer) }
          participantController.triggerOnUpdate()
        }
      },
    )
    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_ADDED_TO_STAGE,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          DyteLogger.info("DyteStage::onMessageEvent::peer added to stage")
          val acceptedPeer =
            participantController.joinedParticipants.find {
              it.id == (event.payload as WebSocketWebinarStagePeer).id
            } ?: return
          acceptedPeer._stageStatus = DyteStageStatus.ACCEPTED_TO_JOIN_STAGE

          // Participant will not be present in _accessRequests if they are added to stage
          // directly by the host. In that case, the list will not be updated and the client
          // should
          // not be notified with onStageRequestsUpdated.
          val requestListModified = _accessRequests.removeAll { it.id == acceptedPeer.id }
          emitEvent { it.onParticipantStartedPresenting(acceptedPeer) }
          if (requestListModified) {
            emitEvent { it.onStageRequestsUpdated(_accessRequests) }
          }
          participantController.triggerOnUpdate()
        }
      },
    )
    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_REQUEST_TO_JOIN_STAGE_REJECTED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          DyteLogger.info("DyteStage::onMessageEvent::peer join request rejected")
          self._stageStatus = DyteStageStatus.REJECTED_TO_JOIN_STAGE
          emitEvent { it.onStageStatusUpdated(self.stageStatus) }
          selfController.emitEvent { it.onStageStatusUpdated(self.stageStatus) }
        }
      },
    )
    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_LEFT,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          DyteLogger.info("DyteStage::onMessageEvent::peer left")
          // note: if this peer is removed from joinedParticipants by ParticipantController,
          // before StageRoomNodeController gets this callback, then we will not actually find the
          // peer.
          // On getting this socket message we actually want to check if the peer who left had a
          // pending stage request.
          // val leftPeer =
          //   participantController.joinedParticipants.find {
          //     it.id == (event.payload as WebSocketPeerLeftModel).id
          //   }

          val leftParticipantPeerId = (event.payload as WebSocketPeerLeftModel).id
          val pendingRequestIndex = _accessRequests.indexOfFirst { it.id == leftParticipantPeerId }

          // pending request peer who left is not present
          if (pendingRequestIndex == -1) {
            return
          }

          val pendingRequestParticipant = _accessRequests.removeAt(pendingRequestIndex)
          emitEvent { it.onPresentRequestClosed(pendingRequestParticipant) }
          emitEvent { it.onStageRequestsUpdated(_accessRequests) }
        }
      },
    )
    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_REQUEST_TO_JOIN_STAGE_ACCEPTED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          DyteLogger.info("DyteStage::onMessageEvent::request accepted for self")
          self._stageStatus = DyteStageStatus.ACCEPTED_TO_JOIN_STAGE
          emitEvent { it.onStageStatusUpdated(self.stageStatus) }
          selfController.emitEvent { it.onStageStatusUpdated(self.stageStatus) }
          emitEvent { it.onPresentRequestReceived() }
        }
      },
    )

    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_STARTED_PRESENTING,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          DyteLogger.info("DyteStage::onMessageEvent::self started presenting")
          self._stageStatus = DyteStageStatus.ON_STAGE
          emitEvent { it.onStageStatusUpdated(self.stageStatus) }
          selfController.emitEvent { it.onStageStatusUpdated(self.stageStatus) }
        }
      },
    )

    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_REMOVED_FROM_STAGE,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          DyteLogger.info("DyteStage::onMessageEvent::self removed from stage")
          self._stageStatus = DyteStageStatus.OFF_STAGE
          self.disableVideo()
          self.disableAudio()
          emitEvent { it.onStageStatusUpdated(self.stageStatus) }
          selfController.emitEvent { it.onStageStatusUpdated(self.stageStatus) }
          emitEvent { it.onRemovedFromStage() }
        }
      },
    )

    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_JOINED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          DyteLogger.info("DyteStage::onMessageEvent::peer joined")
          val joinedPeerPayload = event.payload as WebSocketMeetingPeerUser
          val joinedParticipant =
            WebSocketMeetingPeerUser.fromWebSocketMeetingPeerUser(
              joinedPeerPayload,
              self,
              participantController,
              hostController,
            )
          if (joinedParticipant.stageStatus == StageStatus.REQUESTED_TO_JOIN_STAGE) {
            _accessRequests.add(joinedParticipant)
            emitEvent { it.onPresentRequestAdded(joinedParticipant) }
            emitEvent { it.onStageRequestsUpdated(_accessRequests) }
          }
        }
      },
    )
  }

  // TODO: set the stageStatus for self using the webSocketJoinRoomModel.onRoomJoined
  override fun onRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel) {
    super.onRoomJoined(webSocketJoinRoomModel)
    _accessRequests.clear()
    val waitingPeers = arrayListOf<DyteJoinedMeetingParticipant>()
    webSocketJoinRoomModel.requestToJoinPeersList.forEach { waitingPeer ->
      val joinedPeer = participantController.joinedParticipants.find { it.id == waitingPeer.id }
      joinedPeer?.let {
        it._stageStatus = DyteStageStatus.REQUESTED_TO_JOIN_STAGE
        waitingPeers.add(it)
      }
    }
    _accessRequests.addAll(waitingPeers)
    emitEvent { it.onStageRequestsUpdated(_accessRequests) }
  }

  override suspend fun requestAccess() {
    DyteLogger.info("DyteStage::requestAccess::")
    socketController.sendMessageAsync(
      OutboundMeetingEventType.REQUEST_TO_JOIN_STAGE,
      JsonPrimitive("REQUEST_TO_PRESENT"),
    )
    self._stageStatus = DyteStageStatus.REQUESTED_TO_JOIN_STAGE
    emitEvent { it.onStageStatusUpdated(self.stageStatus) }
    selfController.emitEvent { it.onStageStatusUpdated(self.stageStatus) }
  }

  override suspend fun cancelRequestAccess() {
    DyteLogger.info("DyteStage::cancelRequestAccess::")
    socketController.sendMessage(OutboundMeetingEventType.WITHDRAW_REQUEST_TO_JOIN_STAGE, null)
    self._stageStatus = DyteStageStatus.OFF_STAGE
    emitEvent { it.onStageStatusUpdated(self.stageStatus) }
    selfController.emitEvent { it.onStageStatusUpdated(self.stageStatus) }
  }

  override suspend fun join() {
    DyteLogger.info("DyteStage::join::")
    if (
      permissions.media.video.permission == DyteMediaPermission.ALLOWED ||
        permissions.media.audioPermission == DyteMediaPermission.ALLOWED ||
        self._stageStatus == DyteStageStatus.ACCEPTED_TO_JOIN_STAGE ||
        self._stageStatus ==
          DyteStageStatus
            .ON_STAGE // if user was on stage already, add it to stage after reconnection
    ) {
      DyteLogger.info("DyteStage::join::allowed_to_join_stage")
      socketController.sendMessage(OutboundMeetingEventType.START_PRESENTING, null)
      emitEvent { it.onStageStatusUpdated(self._stageStatus) }
      selfController.emitEvent { it.onStageStatusUpdated(self.stageStatus) }
      emitEvent { it.onAddedToStage() }
    } else {
      DyteLogger.info("DyteStage::join::not allowed to join stage")
    }
  }

  override suspend fun leave() {
    DyteLogger.info("DyteStage::leave::")
    self._stageStatus = DyteStageStatus.OFF_STAGE
    socketController.sendMessage(OutboundMeetingEventType.STOP_PRESENTING, null)
    emitEvent { it.onStageStatusUpdated(self.stageStatus) }
    selfController.emitEvent { it.onStageStatusUpdated(self.stageStatus) }
    emitEvent { it.onRemovedFromStage() }
  }

  override suspend fun kick(id: String) {
    DyteLogger.info("DyteStage::kick::$id")
    if (permissions.host.canAcceptRequests.not()) {
      DyteLogger.error("DyteStage::unauthorized_kick")
      throw UnsupportedOperationException("not allowed to kick stage participant")
    }
    val payload = buildJsonObject {
      put("id", JsonPrimitive(id))
      put("type", JsonPrimitive("REQUESTED_BY_MODERATOR"))
    }
    socketController.sendMessage(OutboundMeetingEventType.REMOVE_PEER_FROM_STAGE, payload)
  }

  override suspend fun grantAccess(id: String) {
    DyteLogger.info("DyteStage::grantAccess::$id")
    if (permissions.host.canAcceptRequests.not()) {
      DyteLogger.error("DyteStage::unauthorized_acceptRequest")
      throw UnsupportedOperationException("not allowed to accept webinar requests")
    }
    val idToAccept = buildJsonArray { addJsonObject { put("id", JsonPrimitive(id)) } }
    socketController.sendMessage(OutboundMeetingEventType.ACCEPT_PRESENTING_REQUEST, idToAccept)
  }

  override suspend fun grantAccessAll() {
    DyteLogger.info("DyteStage::grantAccessAll::")
    if (permissions.host.canAcceptRequests.not()) {
      DyteLogger.error("DyteStage::unauthorized_acceptRequest")
      throw UnsupportedOperationException("not allowed to accept webinar requests")
    }
    val idToAccept = buildJsonArray {
      _accessRequests.forEach { addJsonObject { put("id", JsonPrimitive(it.id)) } }
    }
    socketController.sendMessage(OutboundMeetingEventType.ACCEPT_PRESENTING_REQUEST, idToAccept)
  }

  override suspend fun denyAccess(id: String) {
    DyteLogger.info("DyteStage::denyAccess::$id")
    if (permissions.host.canAcceptRequests.not()) {
      DyteLogger.error("DyteStage::unauthorized_rejectRequest")
      throw UnsupportedOperationException("not allowed to reject webinar requests")
    }
    val payload = buildJsonObject { put("id", JsonPrimitive(id)) }
    socketController.sendMessage(OutboundMeetingEventType.REJECT_PRESENTING_REQUEST, payload)
  }

  override suspend fun denyAccessAll() {
    DyteLogger.info("DyteStage::denyAccessAll::")
    if (permissions.host.canAcceptRequests.not()) {
      DyteLogger.error("DyteStage::unauthorized_rejectRequest")
      throw UnsupportedOperationException("not allowed to reject webinar requests")
    }
    _accessRequests
      .map { it.id }
      .forEach { idToReject ->
        val payload = buildJsonObject { put("id", JsonPrimitive(idToReject)) }
        socketController.sendMessage(OutboundMeetingEventType.REJECT_PRESENTING_REQUEST, payload)
      }
  }
}
