package io.dyte.core.media

import io.dyte.callstats.CallStats
import io.dyte.callstats.media.ConsumerFacade
import io.dyte.callstats.media.ProducerFacade
import io.dyte.callstats.media.TransportFacade
import io.dyte.callstats.models.AuthPayload
import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.models.DyteRoomParticipants
import io.dyte.core.network.models.IceServerData
import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.IDyteMediaUtils
import io.dyte.core.socket.RoomNodeSocketService
import io.dyte.core.socket.SocketMessageEventListener
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.payloadmodel.InboundMeetingEvent
import io.dyte.core.socket.events.payloadmodel.inbound.ConsumerAppData
import io.dyte.core.socket.events.payloadmodel.inbound.ConsumerRtpParameters
import io.dyte.core.socket.events.payloadmodel.inbound.Encodings
import io.dyte.core.socket.events.payloadmodel.inbound.Rtcp
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerClosedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerResumedModel
import io.dyte.core.socket.events.payloadmodel.outbound.Codec
import io.dyte.core.socket.events.payloadmodel.outbound.CodecParameters
import io.dyte.core.socket.events.payloadmodel.outbound.CodecRtcpFeedback
import io.dyte.core.socket.events.payloadmodel.outbound.Fingerprint
import io.dyte.core.socket.events.payloadmodel.outbound.HeaderExtension
import io.dyte.core.socket.events.payloadmodel.outbound.RouterCapabilitiesModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebRtcCreateTransportModel
import io.dyte.media.Consumer
import io.dyte.media.Device
import io.dyte.media.DtlsFingerprint
import io.dyte.media.DtlsParameters
import io.dyte.media.DtlsRole
import io.dyte.media.EmitData
import io.dyte.media.IceCandidate
import io.dyte.media.IceCandidateType
import io.dyte.media.IceParameters
import io.dyte.media.Producer
import io.dyte.media.Protocol
import io.dyte.media.TcpType
import io.dyte.media.Transport
import io.dyte.media.hive.REASON_DISCONNECTION_CLEANUP
import io.dyte.media.utils.IMediaClientLogger
import io.dyte.media.utils.LocalRtcpParameters
import io.dyte.media.utils.LocalRtpCodecParameters
import io.dyte.media.utils.LocalRtpParameters
import io.dyte.media.utils.RTCRtpMediaType
import io.dyte.media.utils.RtcpFeedback
import io.dyte.media.utils.RtpCapabilities
import io.dyte.media.utils.RtpHeaderDirection
import io.dyte.media.utils.RtpHeaderExtension
import io.dyte.media.utils.RtpHeaderExtensionParameters
import io.dyte.webrtc.AudioStreamTrack
import io.dyte.webrtc.CommonRtpEncodingParameters
import io.dyte.webrtc.IceServer
import io.dyte.webrtc.MediaStream
import io.dyte.webrtc.MediaStreamTrackState
import io.dyte.webrtc.VideoStreamTrack
import io.dyte.webrtc.readyState
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

internal class DyteMediaSoup(
  private val callStatsClient: CallStats,
  private val data: DyteRoomParticipants,
  private val socketService: RoomNodeSocketService,
) : IDyteMediaSoupUtils {
  private lateinit var controllerContainer: IControllerContainer
  private lateinit var mediaUtils: IDyteMediaUtils

  private var sendTransport: Transport? = null
  private var receiveTransport: Transport? = null

  private var onCameraStreamKilled = false
  private val consumersToParticipants = HashMap<String, DyteJoinedMeetingParticipant>()
  private val consumers = arrayListOf<Consumer>()

  private var micProducer: Producer? = null
  private var cameraProducer: Producer? = null
  private var screenshareProducer: Producer? = null

  private lateinit var mediaSoupDevice: Device

  private var localVideoTrack: VideoStreamTrack? = null
  private var localAudioTrack: AudioStreamTrack? = null

  private val consumerMutex = Mutex()

  @OptIn(DelicateCoroutinesApi::class, ExperimentalCoroutinesApi::class)
  private val serialScope = CoroutineScope(newSingleThreadContext("DyteMediaSoup"))

  override fun init(controllerContainer: IControllerContainer) {
    mediaUtils = controllerContainer.platformUtilsProvider.getMediaUtils()

    mediaUtils.setPlatform(controllerContainer.platformUtilsProvider.getPlatformUtils().getOsName())
    this.controllerContainer = controllerContainer

    startCallStats()

    CoroutineScope(Dispatchers.Default).launch {
      DyteUserMedia.observer.collect {
        if (it == "screenShareStopped") {
          stopShareScreen()
        }
      }
    }

    if (
      controllerContainer.metaController.getMeetingConfig().enableVideo &&
        controllerContainer.permissionController.isPermissionGrated(CAMERA) &&
        controllerContainer.selfController.canPublishVideo()
    ) {
      localVideoTrack = mediaUtils.getVideoTrack()
      controllerContainer.selfController.getSelf()._videoTrack = localVideoTrack
      controllerContainer.selfController.getSelf()._videoEnabled = true
    }

    if (
      controllerContainer.metaController.getMeetingConfig().enableAudio &&
        controllerContainer.permissionController.isPermissionGrated(MICROPHONE) &&
        controllerContainer.selfController.canPublishAudio()
    ) {
      localAudioTrack = mediaUtils.createAudioTrack()
      controllerContainer.selfController.getSelf()._audioEnabled = true
    }

    socketService.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_RESUME_CONSUMER,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          val payload = event.payload as WebSocketConsumerResumedModel
          resumeConsumer(payload.id)
        }
      },
    )
  }

  private fun startCallStats() {
    callStatsClient.authenticate(AuthPayload(this.controllerContainer.metaController.getPeerId()))

    /*
     * Disabling Pre call tests temporarily for version 1 of CallStats library.
     * */
    /*callStatsClient.sendPreCallTestBeginEvent(object : TestObserver {
      override fun onDone(data: Any) {
        println("DyteClient | DyteMediaSoup | sendPreCallTestBegin onDone: $data")
      }

      override fun onError(ex: Exception) {
        println("DyteClient | DyteMediaSoup | sendPreCallTestBegin onError: ${ex.message}")
      }

      override fun onFailure(reason: String, lastUpdatedResults: Any?) {
        println("DyteClient | DyteMediaSoup | sendPreCallTestBegin onFailure: $reason")
      }
    })*/

    callStatsClient.startPingStats(interval = 7000)
  }

  override suspend fun loadRouterRtpCapabilities(routerRtpCapabilities: RouterCapabilitiesModel) {
    val mediasoupLoggger = MediasoupLoggger()

    mediaSoupDevice = Device(mediasoupLoggger)
    DyteLogger.info("Mediasoup::device_created")

    val localCodecs: MutableList<LocalRtpCodecParameters> = mutableListOf()
    val localHeaderExtensions: MutableList<RtpHeaderExtension> = mutableListOf()

    routerRtpCapabilities.codecs?.forEach {
      val localRtcpFeedback: MutableList<RtcpFeedback> = mutableListOf()
      it.rtcpFeedback?.forEach { fb ->
        localRtcpFeedback.add(RtcpFeedback(type = fb.type!!, parameter = fb.parameter!!))
      }

      localCodecs.add(
        LocalRtpCodecParameters(
          kind = RTCRtpMediaType.fromString(it.kind!!),
          mimeType = it.mimeType!!,
          preferredPayloadType = it.preferredPayloadType,
          clockRate = it.clockRate!!,
          numChannels = it.channels,
          parameters = mutableMapOf(), // it.parameters as MutableMap<String, String?>,
          rtcpFeedback = localRtcpFeedback,
        )
      )
    }

    routerRtpCapabilities.headerExtensions?.forEach {
      localHeaderExtensions.add(
        RtpHeaderExtension(
          kind = RTCRtpMediaType.fromString(it.kind!!),
          uri = it.uri,
          preferredId = it.preferredId,
          preferredEncrypt = it.preferredEncrypt,
          direction = RtpHeaderDirection.fromString(it.direction!!),
        )
      )
    }

    val newRtp = RtpCapabilities(codecs = localCodecs, headerExtensions = localHeaderExtensions)

    localHeaderExtensions.forEachIndexed { index, rtpHeaderExtension ->
      if (rtpHeaderExtension.uri == "urn:3gpp:video-orientation") {
        localHeaderExtensions[index].uri = "do not respect !urn:3gpp:video-orientation!"
      }
    }

    mediaSoupDevice.load(newRtp)
    DyteLogger.info("Mediasoup::device_loaded")
  }

  override suspend fun createWebRtcTransportRecv(
    model: WebRtcCreateTransportModel,
    iceServers: List<IceServerData>,
  ) {
    val id = requireNotNull(model.id) { "Cannot create recv transport because id is null" }

    val iceServersList = arrayListOf<IceServer>()
    iceServers
      .filter { it.username?.isNotBlank() ?: false }
      .forEach {
        val iceServer = IceServer(listOf(it.url), it.username!!, it.credential!!)
        iceServersList.add(iceServer)
      }

    val localIceParameters =
      IceParameters(
        usernameFragment = model.iceParameters!!.usernameFragment!!,
        password = model.iceParameters!!.password!!,
        iceLite = model.iceParameters!!.iceLite!!,
      )

    val localIceCandidate: MutableList<IceCandidate> = mutableListOf()

    model.iceCandidates?.forEach {
      localIceCandidate.add(
        IceCandidate(
          foundation = it.foundation,
          ip = it.ip!!,
          port = it.port!!,
          priority = it.priority!!,
          protocol = Protocol.fromString(it.protocol!!),
          type = IceCandidateType.fromString(it.type!!),
          tcpType = if (it.tcpType != null) TcpType.fromString(it.tcpType!!) else null,
          // Check
          raddr = null,
          rport = null,
          transport = if (it.tcpType != null) "tcp" else "udp",
        )
      )
    }

    val localFingerprints: MutableList<DtlsFingerprint> = mutableListOf()

    model.dtlsParameters?.fingerprints?.forEach {
      localFingerprints.add(DtlsFingerprint(value = it.value!!, algorithm = it.algorithm!!))
    }

    val localDtlsParameters =
      DtlsParameters(
        role = DtlsRole.fromString(model.dtlsParameters!!.role!!),
        fingerprints = localFingerprints,
      )

    DyteLogger.info("Mediasoup::recvTransport::initializing")
    receiveTransport =
      mediaSoupDevice.createRecvTransport(
        id = id,
        iceParameters = localIceParameters,
        iceCandidates = localIceCandidate,
        dtlsParameters = localDtlsParameters,
        iceServers = iceServersList,
      )
    DyteLogger.info("Mediasoup::recvTransport::initialized")

    receiveTransport?.let {
      callStatsClient.registerConsumingTransport(
        TransportFacade(isHive = false, mediasoupTransport = it)
      )
    }

    handleReceiveTransport()
  }

  override suspend fun createWebRtcTransportProd(
    model: WebRtcCreateTransportModel,
    iceServers: List<IceServerData>,
  ) {
    val id = requireNotNull(model.id) { "Cannot create prod transport because id is null" }

    val iceServersList = arrayListOf<IceServer>()
    iceServers
      .filter { it.username?.isNotBlank() ?: false }
      .forEach {
        val iceServer = IceServer(listOf(it.url), it.username!!, it.credential!!)
        iceServersList.add(iceServer)
      }

    val localIceParameters =
      IceParameters(
        usernameFragment = model.iceParameters!!.usernameFragment!!,
        password = model.iceParameters!!.password!!,
        iceLite = model.iceParameters!!.iceLite!!,
      )

    val localIceCandidate: MutableList<IceCandidate> = mutableListOf()

    model.iceCandidates?.forEach {
      localIceCandidate.add(
        IceCandidate(
          foundation = it.foundation,
          ip = it.ip!!,
          port = it.port!!,
          priority = it.priority!!,
          protocol = Protocol.fromString(it.protocol!!),
          type = IceCandidateType.fromString(it.type!!),
          tcpType = if (it.tcpType != null) TcpType.fromString(it.tcpType!!) else null,
          // Check
          raddr = null,
          rport = null,
          transport = if (it.tcpType != null) "tcp" else "udp",
        )
      )
    }

    val localFingerprints: MutableList<DtlsFingerprint> = mutableListOf()

    model.dtlsParameters?.fingerprints?.forEach {
      localFingerprints.add(DtlsFingerprint(value = it.value!!, algorithm = it.algorithm!!))
    }

    val localDtlsParameters =
      DtlsParameters(
        role = DtlsRole.fromString(model.dtlsParameters!!.role!!),
        fingerprints = localFingerprints,
      )

    DyteLogger.info("Mediasoup::sendTransport::initializing")
    sendTransport =
      mediaSoupDevice.createSendTransport(
        id = id,
        iceParameters = localIceParameters,
        iceCandidates = localIceCandidate,
        dtlsParameters = localDtlsParameters,
        iceServers = iceServersList,
      )
    DyteLogger.info("Mediasoup::sendTransport::initialized")

    sendTransport?.let {
      callStatsClient.registerProducingTransport(
        TransportFacade(isHive = false, mediasoupTransport = it)
      )
    }

    handleProduceTransport()
    produceMedia()
  }

  override suspend fun produceMedia() {
    DyteLogger.info("Mediasoup::produceMedia")

    if (controllerContainer.selfController.canPublishVideo()) {
      connectCameraTransport()
    }

    if (controllerContainer.selfController.canPublishAudio()) {
      connectAudioTransport("")
    }
  }

  override fun handleNewConsumer(
    webSocketConsumerModel: WebSocketConsumerModel,
    onDone: () -> Unit,
  ) {
    serialScope.launch {
      consumerMutex.withLock {
        try {
          handleNewConsumerInternal(webSocketConsumerModel, onDone)
        } catch (e: Exception) {
          DyteLogger.error("DyteMediaSoup::handleNewConsumer", e)
        }
      }
    }
  }

  @Throws(Exception::class)
  suspend fun handleNewConsumerInternal(
    webSocketConsumerModel: WebSocketConsumerModel,
    onDone: () -> Unit,
  ) {
    val participant =
      data.joined.find { it.id == webSocketConsumerModel.peerId }
        ?: error("Cannot find participant with peerId ${webSocketConsumerModel.peerId}")

    val id = webSocketConsumerModel.id
    val producerId = webSocketConsumerModel.producerId
    val kind = webSocketConsumerModel.kind
    val rtpParameters = webSocketConsumerModel.rtpParameters
    val enc = arrayListOf<CommonRtpEncodingParameters>()

    rtpParameters.encodings?.forEach {
      val a = CommonRtpEncodingParameters()
      a.ssrc = it.ssrc!!.toLong()
      enc.add(a)
    }

    DyteLogger.info(
      "Mediasoup::consumer::initializing",
      mapOf("consumer.kind" to kind, "consumer.producerId" to producerId),
    )

    val localRtpParameters =
      LocalRtpParameters(
        rtpParameters.mid!!,
        rtpParameters.codecs!!
          .map {
            LocalRtpCodecParameters(
              if (it.mimeType!!.contains("video")) RTCRtpMediaType.RTCRtpMediaTypeVideo
              else RTCRtpMediaType.RTCRtpMediaTypeAudio,
              it.mimeType!!,
              it.preferredPayloadType,
              it.clockRate,
              it.channels,
              mutableMapOf(
                "apt" to JsonPrimitive(it.parameters!!.apt),
                "profileId" to JsonPrimitive(it.parameters!!.profileId),
                "profileLevelId" to JsonPrimitive(it.parameters!!.profileLevelId),
                "packetizationMode" to JsonPrimitive(it.parameters!!.packetizationMode),
                "levelAsymmetryAllowed" to JsonPrimitive(it.parameters!!.levelAsymmetryAllowed),
              ),
              it.rtcpFeedback!!
                .map { feedback -> RtcpFeedback(feedback.type, feedback.parameter) }
                .toMutableList(),
              it.payloadType,
            )
          }
          .toMutableList(),
        rtpParameters.headerExtensions!!
          .map { RtpHeaderExtensionParameters(it.uri, it.id, it.encrypt) }
          .toMutableList(),
        enc,
        LocalRtcpParameters(
          rtpParameters.rtcp!!.mux,
          rtpParameters.rtcp.cname!!,
          rtpParameters.rtcp.reducedSize!!,
        ),
      )

    val consumer =
      receiveTransport!!.consume(
        id = id,
        producerId = producerId,
        kind = RTCRtpMediaType.fromString(kind),
        rtpParameters = localRtpParameters,
        appData =
          mapOf(
            "peerId" to webSocketConsumerModel.appData!!.peerId,
            "screenShare" to webSocketConsumerModel.appData.screenShare,
          ),
        peerId = webSocketConsumerModel.peerId,
      )

    consumers.add(consumer)
    consumersToParticipants[id] = participant

    DyteLogger.info(
      "Mediasoup::consumer::initialized",
      mapOf(
        "consumer.kind" to kind,
        "consumer.id" to consumer.id,
        "consumer.producerId" to producerId,
      ),
    )
    callStatsClient.registerConsumer(ConsumerFacade(isHive = false, mediasoupConsumer = consumer))
    onDone.invoke()
  }

  override fun handleCloseConsumer(webSocketConsumerModel: WebSocketConsumerClosedModel) {
    val consumerId = webSocketConsumerModel.id

    DyteLogger.info("Mediasoup::consumer::closing", mapOf("consumer.id" to consumerId))

    val consumer =
      consumers.find { it.id == consumerId }
        ?: kotlin.run {
          DyteLogger.error(
            "Mediasoup::consumer::close::not_found",
            mapOf("consumer.id" to consumerId),
          )
          return
        }

    val appData =
      ConsumerAppData(
        peerId = consumer.appData["peerId"] as String,
        screenShare = consumer.appData["screenShare"] as? Boolean ?: false,
      )

    if (appData.screenShare) {
      val joinedParticipant = consumersToParticipants[consumerId] ?: return
      val screenShareParticipant = data.joined.find { it.id == joinedParticipant.id } ?: return
      screenShareParticipant._screenShareTrack = null
      controllerContainer.participantController.onPeerScreenSharedEnded(screenShareParticipant)
    }
    consumers.remove(consumer)
    consumersToParticipants.remove(consumerId)
    DyteLogger.info("Mediasoup::consumer::closed", mapOf("consumer.id" to consumerId))
  }

  override fun resumeConsumer(id: String) {
    serialScope.launch { consumerMutex.withLock { resumeConsumerInternal(id) } }
  }

  private fun resumeConsumerInternal(id: String) {
    val consumer =
      consumers.find { it.id == id }
        ?: kotlin.run {
          DyteLogger.error(
            "Mediasoup::resumeConsumer::consumer not found",
            mapOf("consumer.id" to id),
          )
          return
        }

    val participant =
      consumersToParticipants[id]
        ?: kotlin.run {
          DyteLogger.warn(
            "Mediasoup::resumeConsumer::corresponding participant not found",
            mapOf("consumer.id" to id),
          )
          return
        }

    val isAudioTrack = consumer.track is AudioStreamTrack

    if (isAudioTrack) {
      //          participant._audioEnabled = (consumer.track as AudioStreamTrack).enabled
      //          controllerContainer.eventController.triggerEvent(
      //            DyteEventType.OnPeerAudioUpdate(participant)
      //          )
    } else {
      if (consumer.appData["screenShare"] as? Boolean == true) {
        val screenshareTrack = consumer.track as VideoStreamTrack
        participant._screenShareTrack = screenshareTrack
        controllerContainer.participantController.onPeerScreenShareStarted(participant)
      } else {
        participant._videoTrack = consumer.track as VideoStreamTrack
        participant._videoEnabled = (consumer.track as VideoStreamTrack).enabled
        controllerContainer.participantController.onVideoUpdate(participant)
      }
    }
  }

  override suspend fun connectCameraTransport() {
    if (
      controllerContainer.selfController.canPublishVideo() &&
        controllerContainer.permissionController.isPermissionGrated(CAMERA)
    ) {
      enableCamImpl()
    } else {
      DyteLogger.error("Mediasoup::producer::produce_not_permitted", mapOf("kind" to "video"))
    }
  }

  override suspend fun connectAudioTransport(id: String) {
    if (
      controllerContainer.selfController.canPublishAudio() &&
        controllerContainer.permissionController.isPermissionGrated(MICROPHONE)
    ) {
      enableMicImpl()
    } else {
      DyteLogger.error("Mediasoup::producer::produce_not_permitted", mapOf("kind" to "audio"))
    }
  }

  private suspend fun enableCamImpl() {
    try {
      if (cameraProducer != null) {
        DyteLogger.warn("Mediasoup::producer::producer_exist", mapOf("kind" to "video"))
        return
      }
      if (!mediaSoupDevice.loaded) {
        DyteLogger.error("Mediasoup::producer::device_not_loaded")
        return
      }
      if (!mediaSoupDevice.canProduce(RTCRtpMediaType.RTCRtpMediaTypeVideo)) {
        DyteLogger.error("Mediasoup::producer::cannot_produce", mapOf("kind" to "video"))
        return
      }
      if (!controllerContainer.selfController.getSelf()._videoEnabled) {
        DyteLogger.warn("Mediasoup::producer::video_disabled")
        return
      }
      if (localVideoTrack == null) {
        localVideoTrack = mediaUtils.getVideoTrack()
        localVideoTrack?.enabled = true
      }
      createCameraProducer()
    } catch (e: Exception) {
      DyteLogger.error(
        "Mediasoup::producer::error",
        mapOf("kind" to "video", "stack" to e.stackTraceToString()),
      )
    }
  }

  private suspend fun enableMicImpl() {
    try {
      if (micProducer != null) {
        DyteLogger.warn("Mediasoup::producer::producer_exist", mapOf("kind" to "audio"))
        return
      }
      if (!mediaSoupDevice.loaded) {
        DyteLogger.error("Mediasoup::producer::device_not_loaded")
        return
      }
      if (!mediaSoupDevice.canProduce(RTCRtpMediaType.RTCRtpMediaTypeAudio)) {
        DyteLogger.error("Mediasoup::producer::cannot_produce", mapOf("kind" to "audio"))
        return
      }
      if (localAudioTrack == null) {
        localAudioTrack = mediaUtils.createAudioTrack()
        localAudioTrack?.enabled = controllerContainer.selfController.getSelf()._audioEnabled
      }
      createMicProducer()
    } catch (e: Exception) {
      DyteLogger.error(
        "Mediasoup::producer::error",
        mapOf("kind" to "audio", "stack" to e.stackTraceToString()),
      )
      if (localAudioTrack != null) {
        localAudioTrack?.enabled = false
      }
    }
  }

  private suspend fun createMicProducer() {
    if (sendTransport == null || localAudioTrack == null) {
      DyteLogger.error("Mediasoup::producer::Send transport or local audio track is null")
      return
    }
    if (controllerContainer.selfController.canPublishAudio().not()) {
      DyteLogger.warn("Mediasoup::producer::produce_not_permitted", mapOf("kind" to "audio"))
      return
    }
    if (micProducer == null) {
      val data = ConsumerAppData(false, controllerContainer.metaController.getPeerId())
      DyteLogger.warn("Mediasoup::producer::initializing")
      micProducer =
        sendTransport!!.produce(
          track = localAudioTrack!!,
          stream = mediaUtils.getAudioStream()!!,
          // codec = mediaSoupDevice.rtpCapabilities()?.codecs?.first{ it -> it.mimeType.lowercase()
          // == "audio/opus"},
          appData = mapOf("peerId" to data.peerId),
          source = "",
        )

      micProducer?.let {
        DyteLogger.warn(
          "Mediasoup::producer::initialized",
          mapOf("producer.kind" to "audio", "producer.id" to it.id),
        )
        callStatsClient.registerProducer(ProducerFacade(isHive = false, mediasoupProducer = it))
      }
    } else {
      DyteLogger.warn("Mediasoup::producer::already_exists")
    }
  }

  private suspend fun createCameraProducer() {
    if (cameraProducer != null && cameraProducer?.closed == false) {
      DyteLogger.error("MediasoupCamera producer already exists and is open")
      return
    }
    if (sendTransport == null || localVideoTrack == null) {
      DyteLogger.error("Mediasoup::producer::Send transport or local video track is null")
      return
    }
    if (controllerContainer.selfController.canPublishVideo().not()) {
      DyteLogger.warn("Mediasoup::producer::produce_not_permitted", mapOf("kind" to "video"))
      return
    }
    val data = ConsumerAppData(false, controllerContainer.metaController.getPeerId())

    if (localVideoTrack?.readyState is MediaStreamTrackState.Ended) {
      localVideoTrack = mediaUtils.getVideoTrack()
    }
    controllerContainer.selfController.getSelf()._videoTrack = localVideoTrack

    DyteLogger.warn("Mediasoup::producer::initializing", mapOf("producer.kind" to "video"))
    val nonSimulcastEncodings = CommonRtpEncodingParameters()
    nonSimulcastEncodings.active = true
    nonSimulcastEncodings.scaleResolutionDownBy = 1.0

    cameraProducer =
      sendTransport!!.produce(
        track = localVideoTrack!!,
        stream = mediaUtils.getVideoStream()!!,
        codec =
          mediaSoupDevice.rtpCapabilities()?.codecs?.first {
            it.mimeType.lowercase() == "video/vp8"
          },
        encodings = listOf(nonSimulcastEncodings),
        appData =
          mapOf(
            "peerId" to data.peerId
          ), // BaseApiService.json.encodeToString(data) as Map<String, Any>,
        source = "",
      )

    cameraProducer?.let {
      DyteLogger.warn(
        "Mediasoup::producer::initialized",
        mapOf("producer.kind" to "video", "producer.id" to it.id),
      )
      callStatsClient.registerProducer(ProducerFacade(isHive = false, mediasoupProducer = it))
    }
  }

  override fun startShareScreen() {
    if (screenshareProducer != null && screenshareProducer?.closed == false) {
      DyteLogger.error("MediasoupCamera producer already exists and is open")
      return
    }
    if (sendTransport == null) {
      DyteLogger.error("Mediasoup::producer::Send transport is null")
      return
    }
    val data = ConsumerAppData(false, controllerContainer.metaController.getPeerId())

    val ssStartedCallback: suspend (VideoStreamTrack, MediaStream) -> Unit = { ssTrack, stream ->
      controllerContainer.selfController.getSelf()._screenShareTrack = ssTrack

      DyteLogger.warn(
        "Mediasoup::producer::initializing",
        mapOf("producer.kind" to "video", "screenshare" to "true"),
      )

      screenshareProducer =
        sendTransport!!.produce(
          track = ssTrack,
          stream = stream,
          codec =
            mediaSoupDevice.rtpCapabilities()?.codecs?.first {
              it.mimeType.lowercase() == "video/vp8"
            },
          appData =
            mapOf(
              "screenShare" to true,
              "peerId" to data.peerId,
            ), // BaseApiService.json.encodeToString(data) as Map<String, Any>,
          source = "",
        )

      controllerContainer.participantController.onPeerScreenShareStarted(
        controllerContainer.selfController.getSelf()
      )

      controllerContainer.selfController.emitEvent { it.onScreenShareStarted() }

      screenshareProducer?.let {
        DyteLogger.warn(
          "Mediasoup::producer::initialized screenshare",
          mapOf("producer.kind" to "video", "producer.id" to it.id),
        )
        callStatsClient.registerProducer(ProducerFacade(isHive = false, mediasoupProducer = it))
      }
    }

    mediaUtils.getScreenshareStream(
      ssStartedCallback,
      serialScope,
      controllerContainer.selfController,
    )
  }

  override suspend fun stopShareScreen() {
    DyteLogger.warn("stopShareScreen()")
    controllerContainer.selfController.getSelf()._screenShareTrack = null
    mediaUtils.cleanupScreenshare()
    val content = HashMap<String, JsonElement>()
    screenshareProducer?.let {
      content["producerId"] = JsonPrimitive(it.id)
      controllerContainer.roomNodeSocketService.sendMessage(
        OutboundMeetingEventType.CLOSE_PRODUCER,
        JsonObject(content),
      )
      screenshareProducer?.close()
      controllerContainer.participantController.onPeerScreenSharedEnded(
        controllerContainer.selfController.getSelf()
      )
      controllerContainer.selfController.emitEvent { it.onScreenShareStopped() }
    }
  }

  override suspend fun muteSelfAudio(): Boolean {
    DyteLogger.info("Mediasoup::muteSelf")

    if (controllerContainer.selfController._roomJoined) callStatsClient.sendAudioToggleEvent(false)

    localAudioTrack?.enabled = false
    return true
  }

  override suspend fun unmuteSelfAudio(): Boolean {
    DyteLogger.info("Mediasoup::unmuteSelf")
    if (localAudioTrack == null)
      localAudioTrack = controllerContainer.platformUtilsProvider.getMediaUtils().createAudioTrack()

    if (controllerContainer.selfController._roomJoined) {
      createMicProducer()
      callStatsClient.sendAudioToggleEvent(true)
    } else {
      DyteLogger.info("Mediasoup::unmuteSelf::room_not_joined")
    }

    localAudioTrack?.enabled = true
    return true
  }

  override suspend fun muteSelfVideo(): Boolean {
    DyteLogger.info("Mediasoup::muteSelfVideo")

    if (controllerContainer.selfController._roomJoined) {
      callStatsClient.sendVideoToggleEvent(false)
      cameraProducer?.let { producer ->
        producer.close()
        controllerContainer.mediaSoupController.videoProduceId?.let { producerId ->
          controllerContainer.mediaSoupController.onProducerClose(producerId)
        }
      }
    } else {
      DyteLogger.info("Mediasoup::muteSelfVideo::room_not_joined")
      localVideoTrack?.enabled = false
    }

    localVideoTrack = null
    return true
  }

  override suspend fun unmuteSelfVideo(): Boolean {
    DyteLogger.info("Mediasoup::unmuteSelfVideo")
    if (
      onCameraStreamKilled ||
        localVideoTrack == null ||
        localVideoTrack?.readyState is MediaStreamTrackState.Ended
    ) {
      localVideoTrack = mediaUtils.getVideoTrack()
    }

    if (controllerContainer.selfController._roomJoined) {
      createCameraProducer()
      callStatsClient.sendVideoToggleEvent(true)
    } else {
      DyteLogger.info("Mediasoup::unmuteSelfVideo::room_not_joined")
    }

    localVideoTrack?.enabled = true
    controllerContainer.selfController.getSelf()._videoTrack = localVideoTrack
    return true
  }

  override suspend fun cleanupConsumers() {
    val localConsumers = ArrayList(consumers)
    localConsumers.forEach { consumer ->
      if (!consumer.closed) {
        consumer.close()
      }
    }
    consumersToParticipants.clear()
    consumers.clear()
    DyteLogger.info("Mediasoup::cleanupConsumers")
  }

  override suspend fun cleanupProducers(reason: String?) {
    cameraProducer?.close(REASON_DISCONNECTION_CLEANUP)
    micProducer?.close(REASON_DISCONNECTION_CLEANUP)
    screenshareProducer?.close(REASON_DISCONNECTION_CLEANUP)
    cameraProducer = null
    micProducer = null
    screenshareProducer = null
  }

  override suspend fun refreshTracks() {
    localAudioTrack = controllerContainer.platformUtilsProvider.getMediaUtils().createAudioTrack()
    localVideoTrack = controllerContainer.platformUtilsProvider.getMediaUtils().createVideoTrack()
  }

  override suspend fun cleanupTransport() {
    receiveTransport?.close()
    sendTransport?.close()
    receiveTransport = null
    sendTransport = null
  }

  override suspend fun leaveCall() {
    DyteLogger.info("Mediasoup::leaveCall")

    try {
      muteSelfAudio()
      muteSelfVideo()

      cleanupTransport()

      mediaUtils.dispose()

      localAudioTrack = null
      localVideoTrack = null

      callStatsClient.stopPingStats()
      callStatsClient.sendDisconnectEvent()
    } catch (e: Exception) {
      DyteLogger.error("Mediasoup::leaveCall::error", e)
    }
  }

  override fun getSelfTrack(): Pair<Any?, VideoStreamTrack?> {
    return Pair(localAudioTrack, localVideoTrack)
  }

  override fun switchCamera(deviceId: String?) {
    if (localVideoTrack?.enabled == true) {
      mediaUtils.switchCamera(deviceId)
    }
  }

  override fun handleResumeConsumer(id: String) {
    TODO("Not yet implemented")
  }

  override fun handlePauseConsumer(id: String) {
    TODO("Not yet implemented")
  }

  override suspend fun handleCloseProducer(id: String) {
    val REASON_SERVER_CLOSE = "server producer close"
    if (micProducer?.id == id) {
      micProducer?.close(REASON_SERVER_CLOSE)
      micProducer = null
    } else if (cameraProducer?.id == id) {
      cameraProducer?.close(REASON_SERVER_CLOSE)
      cameraProducer = null
    } else if (screenshareProducer?.id == id) {
      screenshareProducer?.close(REASON_SERVER_CLOSE)
      screenshareProducer = null
    }
  }

  override fun onCameraStreamKilled() {
    onCameraStreamKilled = true
  }

  private fun handleReceiveTransport() {
    receiveTransport!!
      .observer
      .onEach {
        when (it.name) {
          "connect" -> {
            DyteLogger.info("Mediasoup::recvTransport::connect")

            val mediasoupDtlsParameters = it.data!!["dtlsParameters"] as DtlsParameters
            val listenerFingerpints: MutableList<Fingerprint> = mutableListOf()

            mediasoupDtlsParameters.fingerprints.forEach { fingerprint ->
              listenerFingerpints.add(
                Fingerprint(algorithm = fingerprint.algorithm, value = fingerprint.value)
              )
            }

            val localDtlsParameters =
              io.dyte.core.socket.events.payloadmodel.outbound.DtlsParameters(
                role = DtlsRole.value(mediasoupDtlsParameters.role),
                fingerprints = listenerFingerpints,
              )

            controllerContainer.mediaSoupController.onReceiveTransportConnected(
              receiveTransport!!.id,
              Json.encodeToString(localDtlsParameters),
            )
          }
          "connectionstatechanged" -> {
            DyteLogger.info(
              "Mediasoup::recvTransport::status_changed",
              mapOf("state" to it.data?.get("state").toString()),
            )
            controllerContainer.connectionController.onReceiveTransportStateChanged(
              it.data?.get("state").toString()
            )
          }
          "close" -> {
            DyteLogger.info("Mediasoup::recvTransport::close")
            consumers.forEach { consumer ->
              callStatsClient.registerConsumerListenerOnTransportClose(
                ConsumerFacade(isHive = false, mediasoupConsumer = consumer)
              )
            }
          }
        }
      }
      .launchIn(serialScope)
  }

  private fun handleProduceTransport() {
    sendTransport!!
      .observer
      .onEach {
        when (it.name) {
          "connect" -> {
            DyteLogger.info("Mediasoup::sendTransport::ice_state_connected")

            val mediasoupDtlsParameters = it.data!!["dtlsParameters"] as DtlsParameters
            val listenerFingerpints: MutableList<Fingerprint> = mutableListOf()

            mediasoupDtlsParameters.fingerprints.forEach { fingerprint ->
              listenerFingerpints.add(
                Fingerprint(algorithm = fingerprint.algorithm, value = fingerprint.value)
              )
            }

            val localDtlsParameters =
              io.dyte.core.socket.events.payloadmodel.outbound.DtlsParameters(
                role = DtlsRole.value(mediasoupDtlsParameters.role),
                fingerprints = listenerFingerpints,
              )

            controllerContainer.mediaSoupController.onSendTransportConnected(
              sendTransport!!.id,
              Json.encodeToString(localDtlsParameters),
            )
          }
          "connectionstatechanged" -> {
            DyteLogger.info(
              "Mediasoup::sendTransport::status_changed",
              mapOf("status" to it.data?.get("state").toString()),
            )
            controllerContainer.connectionController.onSendTransportStateChanged(
              it.data?.get("state").toString()
            )
          }
          "produce" -> {
            DyteLogger.info("Mediasoup::sendTransport::onProduce")
            val mediasoupRtpParameters = it.data?.get("rtpParameters") as LocalRtpParameters

            val localCodecs: MutableList<Codec> = mutableListOf()
            val localHeaderExtensions: MutableList<HeaderExtension> = mutableListOf()
            val localEncodings: MutableList<Encodings> = mutableListOf()

            val localRtcp =
              Rtcp(
                cname = mediasoupRtpParameters.rtcp?.cname,
                reducedSize = mediasoupRtpParameters.rtcp?.reducedSize,
                mux = mediasoupRtpParameters.rtcp?.mux,
              )

            var localkind: String

            mediasoupRtpParameters.codecs.forEach { parameters ->
              val localRtcpFeedback: MutableList<CodecRtcpFeedback> = mutableListOf()

              parameters.rtcpFeedback.forEach { fb ->
                localRtcpFeedback.add(CodecRtcpFeedback(type = fb.type, parameter = fb.parameter))
              }

              localkind = RTCRtpMediaType.value(parameters.kind)

              localCodecs.add(
                Codec(
                  kind = RTCRtpMediaType.value(parameters.kind),
                  mimeType = parameters.mimeType,
                  payloadType = parameters.payloadType,
                  clockRate = parameters.clockRate,
                  channels = if (localkind == "audio") 2 else parameters.numChannels,
                  rtcpFeedback = localRtcpFeedback,
                  parameters =
                    CodecParameters(
                      xGoogleStartBitrate = parameters.parameters["x-google-start-bitrate"] as Int?,
                      apt = parameters.parameters["apt"] as Int?,
                      profileId = parameters.parameters["profileId"] as Int?,
                      packetizationMode = parameters.parameters["packetization-mode"] as Int?,
                      levelAsymmetryAllowed =
                        parameters.parameters["level-asymmetry-allowed"] as Int?,
                      profileLevelId = parameters.parameters["profile-level-id"] as String?,
                    ),
                  preferredPayloadType = parameters.preferredPayloadType,
                )
              )
            }

            mediasoupRtpParameters.headerExtension.forEach { parameters ->
              localHeaderExtensions.add(
                HeaderExtension(
                  uri = parameters.uri,
                  id = parameters.id,
                  encrypt = parameters.encrypted,
                  // parameters = ??
                )
              )
            }

            mediasoupRtpParameters.encodings?.forEach { params ->
              localEncodings.add(
                Encodings(
                  ssrc = params.ssrc?.toInt()
                  // scalabilityMode = ??
                )
              )
            }

            val produceRtpParameters =
              ConsumerRtpParameters(
                codecs = localCodecs,
                headerExtensions = localHeaderExtensions,
                encodings = localEncodings,
                rtcp = localRtcp,
                mid = mediasoupRtpParameters.mid,
              )

            val produceRtpParamsString = Json.encodeToString(produceRtpParameters)
            val appData = (it.data?.get("appData") ?: mapOf<String, Any>()) as Map<*, *>
            val consumerAppData = ConsumerAppData(peerId = appData["peerId"].toString())
            var isSceenshare = false
            if (appData.containsKey("screenShare")) {
              consumerAppData.screenShare = appData.get("screenShare") as Boolean
              isSceenshare = true
            }
            val appDataString = Json.encodeToString(ConsumerAppData.serializer(), consumerAppData)
            DyteLogger.info("Mediasoup::produce $appData")
            val producerId =
              controllerContainer.mediaSoupController.onProduce(
                sendTransport!!.id,
                it.data?.get("kind").toString().lowercase(),
                produceRtpParamsString,
                appDataString,
                isSceenshare,
              )

            sendTransport!!
              .externalObserver
              .emit(EmitData("returnProduce", mapOf("producerId" to producerId)))
          }
          "close" -> {
            DyteLogger.info("Mediasoup::sendTransport::onClose")
            micProducer?.let { producer ->
              callStatsClient.registerProducerListenerOnTransportClose(
                ProducerFacade(isHive = false, mediasoupProducer = producer)
              )
            }
            cameraProducer?.let { producer ->
              callStatsClient.registerProducerListenerOnTransportClose(
                ProducerFacade(isHive = false, mediasoupProducer = producer)
              )
            }
          }
          "newproducer" -> {}
          "producerClose" -> {
            // We should receive a transport here now when producer.close() is called
          }
        }
      }
      .launchIn(serialScope)
  }
}

class MediasoupLoggger() : IMediaClientLogger {
  override fun traceError(message: String) {
    DyteLogger.error(message)
  }

  override fun traceLog(message: String) {
    DyteLogger.info(message)
  }

  override fun traceWarning(message: String) {
    DyteLogger.warn(message)
  }
}
