package io.dyte.core.waitingroom.hive

import io.dyte.core.Result
import io.dyte.core.controllers.ParticipantController
import io.dyte.core.events.EventEmitter
import io.dyte.core.events.InternalEvents
import io.dyte.core.feat.DyteWaitlistedParticipant
import io.dyte.core.host.IHostController
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.models.WaitListStatus.ACCEPTED
import io.dyte.core.models.WaitListStatus.REJECTED
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.waitingroom.BaseWaitlistController
import io.dyte.core.waitingroom.WaitlistSubscription
import io.dyte.core.waitingroom.WaitlistSubscription.WaitlistSubscriber
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.runBlocking

internal class HiveWaitlistController(
  private val waitlistHostSocketHandler: WaitlistHostSocketHandler,
  private val waitlistSubscription: WaitlistSubscription,
  internalEventsEmitter: EventEmitter<InternalEvents>,
  selfParticipant: DyteSelfParticipant,
  scope: CoroutineScope,
  participantController: ParticipantController,
  hostController: IHostController,
) :
  BaseWaitlistController(
    internalEventsEmitter,
    selfParticipant,
    scope,
    participantController,
    hostController,
  ) {

  /*
   * note: This should ideally be inside SelfController and subscription should happen based on the
   * result of joinRoom operation.
   * */
  private val waitlistSubscriber =
    object : WaitlistSubscriber {
      override fun onWaitingRequestAccepted() {
        DyteLogger.info("WaitlistController::onWaitingRequestAccepted::")
        selfParticipant._waitListStatus = ACCEPTED
        internalEventsEmitter.emitEvent { it.onWaitlistEntryAccepted() }
        internalEventsEmitter.emitEvent { it.connectToRoomNode() }
        waitlistSubscription.unsubscribe(this)
      }

      override fun onWaitingRequestRejected() {
        DyteLogger.info("WaitlistController::onWaitingRequestRejected::")
        selfParticipant._waitListStatus = REJECTED
        internalEventsEmitter.emitEvent { it.onWaitlistEntryRejected() }
        waitlistSubscription.unsubscribe(this)
      }
    }

  init {
    waitlistHostSocketHandler.hostEvents
      .onEach { event -> onWaitingRoomHostEvent(event) }
      .launchIn(scope)
    /*
     * note: This subscription should ideally be inside SelfController as Self is waitlisted and
     * would want to get notified if they are accepted or reject to enter meeting.
     * Also, the subscription should happen based on the result of joinRoom operation.
     * */
    waitlistSubscription.subscribe(waitlistSubscriber)
  }

  override fun onRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel) {
    super.onRoomJoined(webSocketJoinRoomModel)
    if (selfPermissions.waitingRoom.canAcceptRequests) {
      runBlocking(scope.coroutineContext) {
        waitlistedParticipants.clear()
        try {
          val waitlistedParticipants = waitlistHostSocketHandler.getWaitlistedParticipants()
          updateLocalWaitlist(waitlistedParticipants)
        } catch (e: Exception) {
          logger.error("WaitlistController::onRoomJoined::failed to get waitlisted participants", e)
        }
      }
    }
  }

  override suspend fun acceptAllWaitingRequestsInternal(
    waitlist: List<DyteWaitlistedParticipant>
  ): Result<List<DyteWaitlistedParticipant>, Exception> {
    DyteLogger.info("WaitlistController::acceptAllWaitingRequestsInternal::")
    val userIds = waitlistedParticipants.map { it.userId }
    val result = waitlistHostSocketHandler.acceptWaitlistedParticipant(userIds)
    if (result is Result.Failure) {
      return result
    }
    return Result.Success(waitlist)
  }

  override suspend fun acceptWaitlistedRequestInternal(
    waitlistedParticipant: DyteWaitlistedParticipant
  ): Result<Unit, Exception> {
    return waitlistHostSocketHandler.acceptWaitlistedParticipant(
      listOf(waitlistedParticipant.userId)
    )
  }

  override suspend fun rejectWaitlistedRequestInternal(
    waitlistedParticipant: DyteWaitlistedParticipant
  ): Result<Unit, Exception> {
    return waitlistHostSocketHandler.rejectWaitlistedParticipant(
      listOf(waitlistedParticipant.userId)
    )
  }

  private fun onWaitingRoomHostEvent(hostEvent: WaitlistHostEvent) {
    when (hostEvent) {
      is WaitlistHostEvent.WaitlistUpdated -> {
        updateLocalWaitlist(hostEvent.waitlist)
      }
      is WaitlistHostEvent.PeerJoinedMeeting -> {
        refreshWaitlistOnPeerJoined(hostEvent.peerId)
      }
    }
  }

  private fun updateLocalWaitlist(updatedWaitlist: List<WaitlistedParticipant>) {
    val existingWaitlistMapByUserId: Map<String, DyteWaitlistedParticipant> =
      waitlistedParticipants.associateBy { it.userId }

    val updatedWaitlistMapByUserId: Map<String, WaitlistedParticipant> =
      updatedWaitlist.associateBy { it.userId }

    // Remove old participants that no more exist in waiting room
    val participantsToRemove: List<DyteWaitlistedParticipant> =
      existingWaitlistMapByUserId.values.filter {
        !updatedWaitlistMapByUserId.containsKey(it.userId)
      }
    participantsToRemove.forEach { removeParticipantFromWaitlist(it.id) }

    // Add new participants who entered waiting room
    val participantsToAdd: List<WaitlistedParticipant> =
      updatedWaitlistMapByUserId.values.filter {
        !existingWaitlistMapByUserId.containsKey(it.userId)
      }
    participantsToAdd.forEach {
      addParticipantToWaitlist(id = it.peerId, userId = it.userId, displayName = it.displayName)
    }
  }
}
