package io.dyte.core.media

import DyteWebRTC.DyteRTCMediaConstraints
import io.dyte.core.controllers.SelfController
import io.dyte.core.media.DyteUserMedia.Companion.observer
import io.dyte.core.native.ObserverProtocol
import io.dyte.webrtc.AudioStreamTrack
import io.dyte.webrtc.CameraVideoCaptureController
import io.dyte.webrtc.MediaStream
import io.dyte.webrtc.MediaStreamConstraintsBuilder
import io.dyte.webrtc.VideoStreamTrack
import io.dyte.webrtc.WebRtc
import io.dyte.webrtc.toMandatoryMap
import io.dyte.webrtc.toOptionalMap
import kotlin.native.concurrent.ThreadLocal
import kotlinx.cinterop.COpaquePointer
import kotlinx.cinterop.ExperimentalForeignApi
import kotlinx.cinterop.reinterpret
import kotlinx.cinterop.staticCFunction
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch
import platform.CoreFoundation.CFDictionaryRef
import platform.CoreFoundation.CFNotificationCenterAddObserver
import platform.CoreFoundation.CFNotificationCenterGetDarwinNotifyCenter
import platform.CoreFoundation.CFNotificationCenterRef
import platform.CoreFoundation.CFNotificationSuspensionBehaviorDeliverImmediately
import platform.CoreFoundation.CFStringRef
import platform.Foundation.CFBridgingRelease
import platform.Foundation.NSUUID
import platform.darwin.NSObject

internal actual val userMedia: DyteUserMedia = DyteUserMediaImpl

@OptIn(ExperimentalForeignApi::class)
fun notificationCallback(
  center: CFNotificationCenterRef?,
  observer: COpaquePointer?,
  name: CFStringRef?,
  `object`: COpaquePointer?,
  userInfo: CFDictionaryRef?,
) {
  val notificationName = name?.let { CFBridgingRelease(it) as String }
}

@ThreadLocal
private object DyteUserMediaImpl : DyteUserMedia {

  private var screenshareTrackObserver = Observer()

  private var screenCaptureController: ScreenCaptureController? = null
  @OptIn(ExperimentalForeignApi::class) val notif = CFNotificationCenterGetDarwinNotifyCenter()

  override suspend fun getUserMedia(
    streamConstraints: MediaStreamConstraintsBuilder.() -> Unit
  ): MediaStream {
    val constraints =
      MediaStreamConstraintsBuilder().let {
        streamConstraints(it)
        it.constraints
      }

    val audioTrack =
      constraints.audio?.let { audioConstraints ->
        val mediaConstraints =
          DyteRTCMediaConstraints(
            mandatoryConstraints = audioConstraints.toMandatoryMap(),
            optionalConstraints = audioConstraints.toOptionalMap(),
          )
        val audioSource = WebRtc.peerConnectionFactory.audioSourceWithConstraints(mediaConstraints)
        val track =
          WebRtc.peerConnectionFactory.audioTrackWithSource(audioSource, NSUUID.UUID().UUIDString())
        AudioStreamTrack(track)
      }

    val videoTrack =
      constraints.video?.let { videoConstraints ->
        val videoSource = WebRtc.peerConnectionFactory.videoSource()
        val iosVideoTrack =
          WebRtc.peerConnectionFactory.videoTrackWithSource(videoSource, NSUUID.UUID().UUIDString())
        val videoCaptureController = CameraVideoCaptureController(videoConstraints, videoSource)
        VideoStreamTrack(iosVideoTrack, videoCaptureController)
      }

    val localMediaStream =
      WebRtc.peerConnectionFactory.mediaStreamWithStreamId(NSUUID.UUID().UUIDString())
    return MediaStream(localMediaStream).apply {
      if (audioTrack != null) addTrack(audioTrack)
      if (videoTrack != null) addTrack(videoTrack)
    }
  }

  override fun getDisplayMedia(
    gotScreenShareStreamCallback: (MediaStream) -> Unit,
    selfController: SelfController,
  ) {
    val uuid = NSUUID.UUID().UUIDString()
    val stream = WebRtc.peerConnectionFactory.mediaStreamWithStreamId(uuid)
    val retStream = MediaStream(stream)

    createScreenCaptureVideoTrack(gotScreenShareStreamCallback, retStream)
  }

  override fun cleanupDisplay() {
    screenCaptureController?.stopCapture()
  }

  @OptIn(ExperimentalForeignApi::class)
  fun createScreenCaptureVideoTrack(
    gotScreenShareStreamCallback: (MediaStream) -> Unit,
    stream: MediaStream,
  ) {
    val videoSource = WebRtc.peerConnectionFactory.videoSourceForScreenCast(true)
    val trackUUID = NSUUID.UUID().UUIDString()

    val videoTrack = WebRtc.peerConnectionFactory.videoTrackWithSource(videoSource, trackUUID)

    val screenCapturer = ScreenCapturer(videoSource)
    screenCaptureController = ScreenCaptureController(screenCapturer)

    CoroutineScope(Dispatchers.Default).launch {
      observer.collect { event ->
        when (event) {
          "screenShareStarted" -> {
            stream.addTrack(VideoStreamTrack(videoTrack, screenCaptureController))
            gotScreenShareStreamCallback(stream)
            this.cancel()
          }
          "screenShareStopped" -> this.cancel()
        }
      }
    }

    val callback =
      staticCFunction<Unit, Unit> {
        CoroutineScope(Dispatchers.Default).launch { observer.emit("screenShareStarted") }
      }

    screenCaptureController?.startCapture()

    CFNotificationCenterAddObserver(
      notif,
      null,
      callback.reinterpret(),
      "broadcastStarted".toCString(),
      null,
      CFNotificationSuspensionBehaviorDeliverImmediately,
    )

    val ssStopped =
      staticCFunction<Unit, Unit> {
        CoroutineScope(Dispatchers.Default).launch { observer.emit("screenShareStopped") }
      }

    CFNotificationCenterAddObserver(
      notif,
      null,
      ssStopped.reinterpret(),
      "broadcastStopped".toCString(),
      null,
      CFNotificationSuspensionBehaviorDeliverImmediately,
    )
  }
}

@OptIn(ExperimentalForeignApi::class)
class Observer : ObserverProtocol, NSObject() {
  @OptIn(ExperimentalForeignApi::class)
  override fun observeValueForKeyPath(
    keyPath: String?,
    ofObject: Any?,
    change: Map<Any?, *>?,
    context: COpaquePointer?,
  ) {
    println("keyPath $keyPath")
    println("ofObject $ofObject")
    println("change $change")
    println("context $context")
  }
}
